# -*- coding: utf-8 -*-
import re
import unicodedata

import six
from six import iteritems


DEFAULT_REASON = 'invalid'

CODE_TO_REASON = {
    'empty': 'empty',
    'missingValue': 'empty',
    'tooLong': 'long',
    'tooShort': 'short',
    'startsWithDigit': 'startswithdigit',
    'startsWithDot': 'startswithdot',
    'startsWithHyphen': 'startswithhyphen',
    'endsWithDot': 'endwithdot',
    'endsWithHyphen': 'endswithhyphen',
    'doubledDot': 'doubleddot',
    'doubledHyphen': 'doubledhyphen',
    'prohibitedSymbols': 'prohibitedsymbols',
    'dotHyphen': 'dothyphen',
    'hyphenDot': 'hyphendot',
    'native': 'native',
    'notAvailable': 'notavailable',
    'weak': 'weak',
    'likeLogin': 'likelogin',
    'likePhoneNumber': 'likephonenumber',
    'likeOldPassword': 'equals_previous',
    'foundInHistory': 'found_in_history',
    'yandexIPAddress': 'yandex_network',
    'listTooShort': 'list_too_short',
    'listTooLong': 'list_too_long',
    'badFormat': 'bad_format',
    'fileTooLarge': 'file_too_large',
    'tooearly': 'tooearly',
    'notMatched': 'not_matched',
}

# https://en.wikipedia.org/wiki/ASCII#Printable_characters
PRINTABLE_ASCII_CHARS_RE = re.compile(r'^[\x20-\x7E]+$')
PRINTABLE_ASCII_CHARS_AND_NEW_LINES_RE = re.compile(r'^[\x0D\x0A\x20-\x7E]+$')

NAME_WHITESPACE_RE = re.compile(r' +')
# https://www.compart.com/en/unicode/category
UNICODE_WHITESPACE_CATEGORIES = (
    'Zl',  # Line Separator
    'Zp',  # Paragraph Separator
    'Zs',  # Space Separator
)


def fold_whitespace(value):
    if isinstance(value, six.text_type):
        value = u''.join(map(lambda x: x if unicodedata.category(x) not in UNICODE_WHITESPACE_CATEGORIES else ' ', value))
    return NAME_WHITESPACE_RE.sub(' ', value)


def is_printable_ascii(value, allow_new_lines=False):
    if allow_new_lines:
        expr = PRINTABLE_ASCII_CHARS_AND_NEW_LINES_RE
    else:
        expr = PRINTABLE_ASCII_CHARS_RE
    return bool(expr.match(value))


def convert_formencode_invalid_to_error_list(invalid_exception, field=None):
    return _flatten_formencode_invalid(invalid_exception, field=field)


def convert_formencode_error_code(code):
    return CODE_TO_REASON.get(code, DEFAULT_REASON)


def _format_formencode_error(code, field):
    reason = convert_formencode_error_code(code)
    return '%s.%s' % (field, reason)


def _flatten_formencode_internal(exc, field):
    errors = set()

    for field, e in iteritems((exc.error_dict or {})):
        errors.update(_flatten_formencode_internal(e, field))
    for e in exc.error_list or []:
        errors.update(_flatten_formencode_internal(e, field))

    if not errors:
        errors.add(_format_formencode_error(exc.code, field))

    return errors


def _flatten_formencode_invalid(exc, field=None):
    """Преобразует сложную структуру ошибки валидации в плоский список кодов ошибок."""
    return sorted(list(_flatten_formencode_internal(exc, field)))
