# -*- coding: utf-8 -*-

import base64
import binascii
import datetime
import json
import re
import time
import unicodedata

from formencode import (
    compound,
    Schema as fSchema,
)
from formencode.validators import (
    _,
    FancyValidator,
    FormValidator,
    Int,
    Number,
    OneOf,
    Regex,
    RequireIfPresent,
    StringBool,
    UnicodeString,
)
from google.protobuf import json_format
from google.protobuf.message import EncodeError
import jsonschema
from passport.backend.core.conf import settings
from passport.backend.core.geobase import (
    is_valid_country_code,
    Region,
)
from passport.backend.core.grants import get_grants_config
from passport.backend.core.lazy_loader import LazyLoader
from passport.backend.core.models import person
from passport.backend.core.models.plus import convert_plus_subscriber_state_proto_to_json
from passport.backend.core.portallib import get_net
from passport.backend.core.services import (
    get_service,
    Service as PassportService,
    slugs,
)
from passport.backend.core.tracks.exceptions import InvalidTrackIdError
from passport.backend.core.tracks.utils import check_track_id
from passport.backend.core.types import (
    birthday,
    gender,
    restore_id,
)
from passport.backend.core.types.account.account import (
    MAX_KPID,
    MIN_KPID,
)
from passport.backend.core.types.email.email import domain_from_email
from passport.backend.core.types.ip import ip
from passport.backend.core.types.login.login import (
    normalize_login,
    raw_login_from_email,
)
from passport.backend.core.types.phone_number import phone_number
from passport.backend.core.types.totp_secret import (
    InvalidPinError,
    MAX_PIN,
    parse_pin_as_string,
    PIN_DIGITS_MAX_COUNT,
    PIN_DIGITS_MIN_COUNT,
)
from passport.backend.core.validators.base import (
    Invalid,
    Validator,
)
from passport.backend.core.validators.login.login import (
    Login,
    YandexTeamLogin,
)
from passport.backend.core.validators.utils import (
    fold_whitespace,
    is_printable_ascii,
)
from passport.backend.utils.common import (
    unique_preserve_order,
    url_to_ascii,
)
from passport.backend.utils.string import (
    smart_bytes,
    smart_text,
    snake_case_to_camel_case,
)
from passport.protobuf.billing_features.billing_features_pb2 import FeatureAttributes
from passport.protobuf.plus_subscriber_state.plus_subscriber_state_pb2 import PlusSubscriberState
import pytz
import six
from six.moves.urllib.parse import (
    unquote,
    urlparse,
)
import yenv


# https://www.compart.com/en/unicode/category
UNICODE_CATEGORIES_WHITELIST = (
    'Ll',  # Lowercase Letter
    'Lm',  # Modifier Letter
    'Lo',  # Other Letter
    'Lt',  # Titlecase Letter
    'Lu',  # Uppercase Letter
    'Mc',  # Spacing Mark
    'Me',  # Enclosing Mark
    'Mn',  # Nonspacing Mark
    'Nd',  # Decimal Number
    'Nl',  # Letter Number
    'No',  # Other Number
    'Pc',  # Connector Punctuation
    'Pd',  # Dash Punctuation
    'Pe',  # Close Punctuation
    'Pf',  # Final Punctuation
    'Pi',  # Initial Punctuation
    'Po',  # Other Punctuation
    'Ps',  # Open Punctuation
    'Sc',  # Currency Symbol
    'Sk',  # Modifier Symbol
    'Sm',  # Math Symbol
    'So',  # Other Symbol
    'Zs',  # Space Separator
)
MIN_TIMESTAMP_VALUE = int(time.mktime(datetime.date(1900, 1, 1).timetuple()))
NUMBER_FROM_DATE_TEMPLATES_WITHOUT_YEAR = (
    '%d%m',
)
NUMBER_FROM_DATE_TEMPLATES_WITH_YEAR = (
    '%d%m%y',
    '%d%m%Y',
)
MIN_YEAR = 1900  # с меньшими не умеет работать strftime
MAX_LONG_VALUE = 2147483647

MAX_UID = 2 ** 63 - 2  # Все что больше ЧЯ не считает валидным UID-ом
MAX_UID_HEX_LENGTH = len('%x' % MAX_UID)

MAX_HOSTNAME_LENGTH = 255
MAX_ASCII_STRING_LENGTH = 255

MAX_DOMAIN_ID = 2 ** 63 - 2

MAX_DISPLAY_NAME_LENGTH = 60


def is_allowed_unicode_symbol(symbol):
    category = unicodedata.category(symbol)
    # пропускаем эмодзи во втором питоне
    # диспазоны на основе https://en.wikipedia.org/wiki/Emoji#Unicode_blocks
    if category == 'Cn':
        for start, end in [
            (u'\U0001F300', u'\U0001F5FF'),  # Miscellanous Symbols and Pictographs
            (u'\U0001F600', u'\U0001F64F'),  # Emoticons
            (u'\U0001F680', u'\U0001F6D7'),  # Transport and Map Symbols
            (u'\U0001F6E0', u'\U0001F6EC'),
            (u'\U0001F6F0', u'\U0001F6FC'),
            (u'\U0001F900', u'\U0001F978'),  # Supplemental Symbols and Pictographs
            (u'\U0001F97A', u'\U0001F9CB'),
            (u'\U0001F9CD', u'\U0001F9FF'),
            (u'\U0001FA70', u'\U0001FA74'),  # Symbols and Pictographs Extended-A
            (u'\U0001FA78', u'\U0001FA7A'),
            (u'\U0001FA80', u'\U0001FA86'),
            (u'\U0001FA90', u'\U0001FAA8'),
            (u'\U0001FAB0', u'\U0001FAB6'),
            (u'\U0001FAC0', u'\U0001FAC2'),
            (u'\U0001FAD0', u'\U0001FAD6'),
        ]:
            if start <= symbol <= end:
                return True
    return category in UNICODE_CATEGORIES_WHITELIST


def build_fraud_regular_expressions():
    tlds = sorted(settings.TOP_LEVEL_DOMAINS)
    # \.TLD, а фигурные скобки {} нужны для вызова .format
    return [re.compile(smart_text(r'.+\.(?:{})(?:[^a-z]|$)').format('|'.join(tlds)), re.UNICODE | re.IGNORECASE)]


LazyLoader.register('build_fraud_regular_expressions', build_fraud_regular_expressions)

# Ставим хоть какой-то валидатор на логин для конкретного алиаса,
# зависит от окружения и списка алиасов
ALT_DOMAIN_LOGIN_VALIDATORS = {}
if yenv.name == 'intranet':
    ALT_DOMAIN_LOGIN_VALIDATORS = {domain: YandexTeamLogin for domain in settings.ALT_DOMAINS}


class State(object):
    def __init__(self, env, files=None):
        self.env = env
        self.files = files
        self.password_quality = None


class Schema(fSchema):
    allow_extra_fields = True
    filter_extra_fields = True


class StrictUnicodeString(UnicodeString):
    """
    Расширение валидатора строки с опциональной проверкой типа значения
    """
    strict = False

    def _to_python(self, value, state):
        if self.strict:
            # TODO: возможно, безопасно всегда делать эту проверку?
            self.assert_string(value, state)
        return super(StrictUnicodeString, self)._to_python(value, state)

    def is_empty(self, value):
        # Пустые значения - только None и пустая строка
        return value is None or value == ''


# На входе от Flask валидаторы всегда получают unicode-строку,
# отличий в работе UnicodeString и String для нас нет, по хорошему везде нужно использовать один валидатор
String = UnicodeString = StrictUnicodeString


class AnyOfStringsOrEmptyValidator(String):
    def __init__(self, allowed_values, *args, **kwargs):
        self.allowed_values = {value.lower() for value in allowed_values}
        super(AnyOfStringsOrEmptyValidator, self).__init__(*args, **kwargs)

    def _to_python(self, value, state):
        value = super(AnyOfStringsOrEmptyValidator, self)._to_python(value, state)
        if value.lower() in self.allowed_values:
            return value
        else:
            return ''


class AntiFraudString(String):
    messages = {
        'invalid': _('Invalid value'),
    }

    def __init__(self, bad_symbols_limit=10, bad_symbols_limit_and_url=2, *args, **kwargs):
        super(AntiFraudString, self).__init__(*args, **kwargs)
        self.bad_symbols_count = bad_symbols_limit
        self.bad_symbols_count_and_url = bad_symbols_limit_and_url

    def _to_python(self, value, state):
        value = super(AntiFraudString, self)._to_python(value, state)
        lcase_value = value.lower()
        bad_symbols = [c for c in lcase_value if not c.isalpha()]
        if len(bad_symbols) >= self.bad_symbols_count:
            raise Invalid(
                self.message('invalid', state),
                value,
                state,
            )
        # чтобы меньше фолсило, помимо урла ожидаем ещё каких-нибудь запрещённых символов
        if len(bad_symbols) >= self.bad_symbols_count_and_url:
            for regular_expression in LazyLoader.get_instance('build_fraud_regular_expressions'):
                if regular_expression.search(lcase_value):
                    raise Invalid(
                        self.message('invalid', state),
                        value,
                        state,
                    )
        return value


class Regex(Regex):
    def is_empty(self, value):
        # Пустые значения - только None и пустая строка
        return value is None or value == ''


class TransformingRegex(String):
    """
    Проверяет, что строка матчится регулярным выражением.
    Возвращает словарь именованных групп совпадения.
    """
    regex = None

    messages = {
        'invalid': _('Invalid value'),
    }

    def _to_python(self, value, state):
        value = super(TransformingRegex, self)._to_python(value, state)
        match = self.regex.match(value)
        if not match:
            raise Invalid(self.message('invalid', state), value, state)

        return match.groupdict()


class RequireSome(FormValidator):
    # List of fields to check
    some_fields = None

    # минимальное число разрешённых параметров
    min_ = 1
    # максимальное число разрешённых параметров
    max_ = 1

    rule_name = 'form'

    empty_values = [None]

    __unpackargs__ = ('some_fields',)

    messages = {
        'empty': _('Please enter a value'),
        'tooFew': _('At least %(min_)s of %(fields)s is required'),
        'tooMany': _('At most %(max_)s of %(fields)s should be present'),
    }

    def _to_python(self, value_dict, state):
        set_fields = len(
            set(self.some_fields) &
            set([
                key
                for key in value_dict
                if value_dict.get(key) not in self.empty_values
            ]),
        )
        if set_fields < self.min_:
            message = self.message('tooFew', state, min_=self.min_, fields=', '.join(self.some_fields))
            raise Invalid(message,
                          value_dict, state,
                          error_dict={self.rule_name: Invalid(message, value_dict, state)})
        if set_fields > self.max_:
            message = self.message('tooMany', state, max_=self.max_, fields=', '.join(self.some_fields))
            raise Invalid(message,
                          value_dict, state,
                          error_dict={self.rule_name: Invalid(message, value_dict, state)})

        return value_dict


class RequireSet(FormValidator):
    messages = {
        'invalidSet': _('The specified set of parameters is not sufficient. Expected parameters: %(params)s'),
        'abundantSet': _(
            'Only one set of valid parameters is required, not more. Expected parameters: %(params)s',
        ),
    }

    __unpackargs__ = ('allowed_sets', 'allow_empty')

    def __init__(self, allowed_sets, allow_empty=False):
        if not allowed_sets:
            raise ValueError('allowed_sets is empty')
        super(RequireSet, self).__init__(allowed_sets)
        self.allowed_sets = sorted(map(set, self.allowed_sets), key=len, reverse=True)
        self.fields = set.union(*self.allowed_sets)
        self.allow_empty = allow_empty

    def expected_parameters(self):
        retval = []
        for fields_set in self.allowed_sets:
            retval.append('<' + ', '.join('"%s"' % f for f in sorted(fields_set)) + '>')
        if self.allow_empty:
            retval.append('all values are missing')
        return ' | '.join(retval)

    def _to_python(self, value_dict, state):
        valid_set = None
        value_subdict = dict((key, value) for key, value in value_dict.items()
                             if key in self.fields)

        for fields_set in self.allowed_sets:
            if valid_set and valid_set.issuperset(fields_set):
                continue
            if all(value_subdict[f] is not None for f in fields_set):
                if valid_set:
                    message = self.message('abundantSet', state, params=self.expected_parameters())
                    raise Invalid(message, value_dict, state,
                                  error_dict={'form': Invalid(message, value_dict, state)})
                else:
                    valid_set = fields_set
        if not valid_set and self.allow_empty and all(not value for value in value_subdict.values()):
            return value_dict

        if not valid_set:
            message = self.message('invalidSet', state, params=self.expected_parameters())
            raise Invalid(message, value_dict, state,
                          error_dict={'form': Invalid(message, value_dict, state)})
        return value_dict


class AliasFields(FormValidator):
    # {primary_name: [aliases]} dictionary
    aliases = None

    __unpackargs__ = ('aliases',)

    def _to_python(self, value_dict, state):
        for key in self.aliases:
            if key in value_dict:
                continue
            values = list(filter(
                lambda x: x[1] is not None,
                ((alias, value_dict.get(alias)) for alias in self.aliases[key])
            ))
            if values:
                del value_dict[values[0][0]]
                value_dict[key] = values[0][1]

        return value_dict


class Timezone(FancyValidator):
    messages = {
        'badTimezone': _('Unknown timezone'),
    }

    def _to_python(self, value, state):
        try:
            return pytz.timezone(value)
        except pytz.UnknownTimeZoneError:
            raise Invalid(self.message('badTimezone', state), value, state)


class Birthday(FancyValidator):
    messages = {
        'badBirthday': _('Wrong birhday value'),
        'missingDateValues': _('Year, month and day are all required'),
    }

    # Валидатор потребует, чтобы год (если он известен) был в отрезке
    # [год сейчас - year_offset, год сейчас].
    year_offset = 100

    # Валидатор потребует, чтобы все компоненты даты были заполнены, иначе
    # допукается замена любого компонента нолём.
    need_full = False

    def _to_python(self, value, state):
        self.assert_string(value, state)
        today = datetime.datetime.today()
        try:
            _birthday = birthday.Birthday.parse(value)
            if str(_birthday)[:4] != '0000' and \
                    not (today.year - self.year_offset <= _birthday.date.year <= today.year):
                raise Invalid(self.message('badBirthday', state), value, state)
            if self.need_full and not _birthday.is_date_full:
                raise Invalid(self.message('missingDateValues', state), value, state)
            return _birthday
        except (TypeError, ValueError):
            raise Invalid(self.message('badBirthday', state), value, state)

    def is_empty(self, value):
        # Пустые значения - только None и пустая строка
        return value is None or value == ''


class LooseDateValidator(FancyValidator):
    messages = {
        'invalid': _('Loose date should be in form yyyy-mm-dd or yyyy-mm-00 or yyyy-00-00'),
        'tooearly': _('Date is too early'),
        'toolate': _('Date is too late'),
    }

    earliest_year = 1900
    max_days_in_future = None  # Максимальное увеличение относительно текущей даты в днях

    date_re = re.compile(r'^(\d{4})-(\d{2})-(\d{2})$')

    def _to_python(self, value, state):
        self.assert_string(value, state)
        match = self.date_re.match(value)
        if match:
            year, month, day = map(int, match.groups())
            if month or not day:
                dt = None
                try:
                    dt = datetime.datetime(year, month or 1, day or 1)
                except ValueError:
                    pass
                if dt and year < self.earliest_year:
                    raise Invalid(self.message('tooearly', state), value, state)
                elif (
                    dt and
                    self.max_days_in_future is not None and
                    dt - datetime.timedelta(days=self.max_days_in_future) > datetime.datetime.today()
                ):
                    raise Invalid(self.message('toolate', state), value, state)
                elif dt:
                    return dt
        raise Invalid(self.message('invalid', state), value, state)

    def is_empty(self, value):
        # Пустые значения - только None и пустая строка
        return value is None or value == ''


class TimestampInPast(Int):
    min = MIN_TIMESTAMP_VALUE

    def _to_python(self, value, state=None):
        self.max = int(time.time())
        return super(TimestampInPast, self)._to_python(value, state)


class Unixtime(Number):
    messages = dict(
        invalid=_('Please enter an valid unixtime'),
    )
    strip = True
    not_empty = True
    min = datetime.datetime.fromtimestamp(-2 ** 31)
    max = datetime.datetime.fromtimestamp((2 ** 31) - 1)
    allow_milliseconds = False

    def _to_python(self, value, state):
        timestamp = super(Unixtime, self)._to_python(value, state)
        if not self.allow_milliseconds and int(timestamp) != timestamp:
            raise Invalid(self.message('invalid', state), value, state)
        try:
            return datetime.datetime.fromtimestamp(timestamp)
        except (ValueError, OverflowError):
            # Сюда попадём на числах, бОльших чем time_t
            raise Invalid(self.message('invalid', state), value, state)


class PhoneNumber(FormValidator):
    messages = {
        'badPhoneNumber': _('Wrong PhoneNumber value'),
    }

    contains_letter_re = re.compile(r'[a-zA-Z]')

    def __init__(
        self,
        country_key='country',
        phone_number_key='phone_number',
        allow_impossible=False,
    ):
        self._country_key = country_key
        self._phone_number_key = phone_number_key
        self._allow_impossible = allow_impossible

    def _to_python(self, value_dict, state):
        key = self._phone_number_key
        raw_phone_number = value_dict[key]

        if raw_phone_number is None:
            return value_dict

        new_value_dict = dict(value_dict)
        country = value_dict.get(self._country_key, None)
        try:
            if self.contains_letter_re.search(raw_phone_number):
                raise phone_number.InvalidPhoneNumber()

            phonenumber = phone_number.PhoneNumber.parse(
                raw_phone_number,
                country,
                allow_impossible=self._allow_impossible,
            )
            new_value_dict[key] = phonenumber
            return new_value_dict
        except phone_number.InvalidPhoneNumber:
            message = self.message('badPhoneNumber', state)
            raise Invalid(
                message,
                value_dict,
                state,
                error_dict={
                    self._phone_number_key: Invalid(message, value_dict, state),
                },
            )


class PhoneId(Int):
    strip = True
    not_empty = True


class GroupDisplayName(FormValidator):
    """Собирает поля profile_id, provider и display_name в словарь display_name"""

    def _to_python(self, value_dict, state):
        new_value_dict = {
            'display_name': {},
        }
        for key in value_dict:
            if key in ['display_name', 'profile_id', 'provider', 'is_from_variants']:
                new_value_dict['display_name'][key] = value_dict[key]
            else:
                new_value_dict[key] = value_dict[key]
        return new_value_dict


class SetDisplayName(FormValidator):
    """
    Формирует строку display_name из display_name, provider и profile_id
    """
    messages = {
        'badDisplayNameVariant': _('Wrong display name variant'),
    }

    def _to_python(self, value_dict, state):
        if value_dict['display_name'] is None:
            return

        # is_from_variants - параметр означает, что display_name выбран из заготовленных
        # нами вариантов, и строка уже соответствует сериализованному значению display_name.
        if value_dict.get('is_from_variants', False):
            display_name = person.DisplayName()
            try:
                display_name.set(value_dict['display_name'])
            except ValueError:
                raise Invalid(self.message('badDisplayNameVariant', state), value_dict, state)
            return display_name

        return person.DisplayName(
            value_dict['display_name'],
            value_dict.get('provider'),
            value_dict.get('profile_id'),
        )


class SocialProvider(Regex):
    # См. http://doc.yandex-team.ru/Passport/passport-api/reference/admsocialreg.xml
    regex = '^[a-z]{2}$'


class HexString(Regex):
    regex = '^[a-f0-9]+$'


class GpsPackageName(Regex):
    """
    Параметры для SMS Retriever (https://developers.google.com/identity/sms-retriever/overview)
    gps - Google Play Services
    Правила валидации взяты отсюда: https://developer.android.com/studio/build/application-id.html
      - It must have at least two segments (one or more dots).
      - Each segment must start with a letter.
      - All characters must be alphanumeric or an underscore [a-zA-Z0-9_].
    """
    regex = r'^[a-zA-Z][a-zA-Z0-9_]*\.[a-zA-Z][a-zA-Z0-9_]*(\.[a-zA-Z][a-zA-Z0-9_]*)*$'
    if_missing = None
    not_empty = True


class Name(String):
    strip = True
    if_empty = None

    messages = {
        'unallowedSymbol': _('Contains not allowed unicode symbol'),
    }

    def _to_python(self, value, state):
        value = super(Name, self)._to_python(value, state)
        return fold_whitespace(value)[:50]

    def validate_python(self, value, state):
        if isinstance(value, six.text_type):
            if not all(map(lambda x: is_allowed_unicode_symbol(x), value)):
                raise Invalid(self.message('unallowedSymbol', state), value, state)
        return value


class AntiFraudName(AntiFraudString):
    strip = True
    if_empty = None

    messages = {
        'unallowedSymbol': _('Contains not allowed unicode symbol'),
    }

    def _to_python(self, value, state):
        value = super(AntiFraudName, self)._to_python(value, state)
        return fold_whitespace(value)[:50]

    def validate_python(self, value, state):
        if isinstance(value, six.text_type):
            if not all(map(lambda x: is_allowed_unicode_symbol(x), value)):
                raise Invalid(self.message('unallowedSymbol', state), value, state)
        return value


FirstName = Name
LastName = Name

AntiFraudFirstName = AntiFraudName
AntiFraudLastName = AntiFraudName


class AntiFraudDisplayName(FormValidator):
    string_validator = AntiFraudName()

    def _to_python(self, value, state):
        if not value:
            return
        if value.is_template:
            return value
        self.string_validator.to_python(value.name, state)
        return value


class UserDefinedDisplayName(Schema):
    """
    Форма для валидации имени пользователя, которое явно задано пользователем

    Может использоваться как валидатор поля, если у родительской формы есть
    GroupDisplayName в pre_validators.
    """
    antifraud_enabled = True
    display_name = String(if_missing=None, not_empty=True, strip=True, max=MAX_DISPLAY_NAME_LENGTH)

    def __init__(self, *args, **kwargs):
        super(UserDefinedDisplayName, self).__init__(*args, **kwargs)

        if self.antifraud_enabled:
            self.add_chained_validator(AntiFraudDisplayName())

    chained_validators = [
        SetDisplayName(),
    ]


class DisplayName(Schema):
    """
    Форма для валидации имени пользователя

    Может использоваться как валидатор поля, если у родительской формы есть
    GroupDisplayName в pre_validators.
    """
    display_name = String(if_missing=None, not_empty=True, strip=True, max=MAX_DISPLAY_NAME_LENGTH)
    provider = SocialProvider(if_missing=None)
    profile_id = Int(if_missing=None)
    is_from_variants = StringBool(if_missing=False, not_empty=True, strip=True)

    chained_validators = [
        RequireIfPresent(required='provider', present='profile_id'),
        RequireIfPresent(required='profile_id', present='provider'),
        SetDisplayName(),
        AntiFraudDisplayName(),
    ]


class SocialDisplayName(Schema):
    """
    Форма для валидации имени для социального пользователя

    Может использоваться как валидатор поля, если у родительской формы есть
    GroupDisplayName в pre_validators.
    """
    display_name = String(if_missing=None, not_empty=True, strip=True)
    provider = SocialProvider(if_missing=None)
    profile_id = Int(if_missing=None)

    chained_validators = [
        RequireSet(
            allowed_sets=[
                [
                    'display_name',
                    'provider',
                    'profile_id',
                ],
            ],
            allow_empty=True,
        ),
        SetDisplayName(),
    ]


class Hostname(Regex):
    regex = None
    max_length = MAX_HOSTNAME_LENGTH
    regexOps = (re.IGNORECASE, re.UNICODE)
    messages = {
        'invalid': _(
            'Hostname parts should start with a letter or a digit, contain only letters, digits and hyphens, and end up with a letter or a digit',
        ),
        'tooLong': _('Hostname has more than %d symbols' % max_length),
        'idnaEncodingFailed': _('Hostname cannot be encoded into IDNA'),
        'idnaDecodingFailed': _('Malformed IDNA encoded hostname'),
        'native': _('Hostname within Yandex native domain'),
    }

    character_sets = {
        'default': smart_text(r'\w') + u'\u0300-\u036F',
        'strict': u'a-z0-9',
    }

    def __init__(self, decode_punycode=False, character_set='default', min_levels=2, reject_native=False, *args,
                 **kwargs):
        self.decode_punycode = decode_punycode
        self.reject_native = reject_native

        if character_set not in self.character_sets:
            raise ValueError(
                'Unknown character set for hostname validation: %s (available: %s)' % (
                    character_set,
                    ', '.join(self.character_sets.keys()),
                ),
            )

        regex_charset = self.character_sets[character_set]
        self.regex = smart_text(r'^(([{0}]|[{0}][{0}\-]*[{0}])\.){{{1},}}([{0}]|[{0}][{0}\-]*[{0}])$').format(
            regex_charset,
            min_levels - 1,
        )
        super(Hostname, self).__init__(*args, **kwargs)

    def _to_python(self, value, state):
        value = super(Hostname, self)._to_python(value, state)
        if '_' in value:
            raise Invalid(
                self.message('invalid', state),
                value, state,
            )

        try:
            encoded_value = value.encode('idna')
        except UnicodeError:
            raise Invalid(
                self.message('idnaEncodingFailed', state),
                value, state,
            )

        if len(encoded_value) > self.max_length:
            raise Invalid(
                self.message('tooLong', state),
                value, state,
            )

        if self.decode_punycode and 'xn--' in value:
            try:
                value = value.encode('utf8').decode('idna')
            except UnicodeError:
                raise Invalid(
                    self.message('idnaDecodingFailed', state),
                    value, state,
                )

        if self.reject_native and value in settings.NATIVE_EMAIL_DOMAINS:
            raise Invalid(self.message('native', state), value, state)

        return value


class IPAddress(FancyValidator):
    messages = {
        'invalidIPAddress': _('Invalid IP address'),
    }

    def _to_python(self, value, state):
        try:
            return ip.IP(value)
        except:
            raise Invalid(self.message('invalidIPAddress', state), value, state)


class CountryCode(FancyValidator):
    not_empty = True

    messages = {
        'badCountryCode': _('Wrong country code value'),
    }

    def _to_python(self, value, state):
        if is_valid_country_code(value):
            return value.lower()
        else:
            raise Invalid(self.message('badCountryCode', state), value, state)


class CityId(Int):
    min = 1
    max = MAX_LONG_VALUE

    messages = {
        'badCityId': _('Wrong city id'),
    }

    def validate_python(self, value, state):
        try:
            super(CityId, self).validate_python(value, state)
            if Region(id=value).city:
                return value
            else:
                raise Invalid(self.message('badCityId', state), value, state)
        except RuntimeError as e:
            if 'unknown id' in str(e):
                raise Invalid(self.message('badCityId', state), value, state)
            raise e  # pragma: no cover


class Service(FancyValidator):
    def __init__(self, ignore_unknown_service=False, *args, **kwargs):
        self.ignore_unknown_service = ignore_unknown_service
        super(Service, self).__init__(*args, **kwargs)

    def _to_python(self, value, state):
        if isinstance(value, PassportService):
            return value
        self.assert_string(value, state)
        if self.ignore_unknown_service and value not in slugs:
            return
        OneOf(slugs, messages={'notIn': 'Invalid service'}).to_python(value, state)
        if value.isdigit():
            return get_service(sid=int(value))
        return get_service(slug=value)

    def is_empty(self, value):
        # Пустые значения - только None и пустая строка
        return value is None or value == ''


class ListValidator(FancyValidator):
    messages = {
        'badvalues': _('List contains bad values: %(badvalues)s'),
        'listTooShort': _('Too few elements (expected at least %(min)d)'),
        'listTooLong': _('Too many elements (expected at most %(max)d)'),
    }

    min = max = None

    def __init__(self, validator, not_in=None, delimiter=',', unique=False, unquote=False, *args, **kwargs):
        super(FancyValidator, self).__init__(*args, **kwargs)
        self.delimiter = delimiter
        self.validator = validator
        self.not_in = not_in
        self.unique = unique
        self.unquote = unquote

    def _unquote_and_decode(self, value, state):
        """
        Приводим входную строку к байтовой строке, декодируем из URL encoding,
        после чего декодируем из UTF-8.
        """
        try:
            value = unquote(str(value))
            if six.PY2:
                value = value.decode('utf-8')
            return value
        except (UnicodeEncodeError, UnicodeDecodeError):
            raise Invalid('', value, state)  # параметры исключения не важны в этом случае

    def _to_python(self, value, state):
        values = value.split(self.delimiter)
        if self.min is not None and len(values) < self.min:
            raise Invalid(self.message('listTooShort', state, min=self.min), value, state)
        if self.max is not None and len(values) > self.max:
            raise Invalid(self.message('listTooLong', state, max=self.max), value, state)
        if self.unique:
            values = unique_preserve_order(values)

        new_values = []
        bad_values = []
        for value in values:
            try:
                if self.unquote:
                    value = self._unquote_and_decode(value, state)
                new_value = self.validator.to_python(value, state)
                if self.not_in and new_value in self.not_in:
                    bad_values.append(value)
                else:
                    new_values.append(new_value)
            except Invalid:
                bad_values.append(value)
        if bad_values:
            raise Invalid(
                self.message(
                    'badvalues',
                    state,
                    badvalues=', '.join(['`%s`' % x for x in bad_values]),
                ),
                value,
                state,
            )
        return new_values


class JSONValidator(FancyValidator):
    messages = {
        'badjson': _('Cannot parse/validate JSON: %(error)s'),
    }

    def __init__(self, schema=None, *args, **kwargs):
        super(JSONValidator, self).__init__(*args, **kwargs)
        self.schema = schema

    def _to_python(self, value, state):
        try:
            json_value = json.loads(value)
        except ValueError as e:
            raise Invalid(
                self.message('badjson', state, error=e),
                value,
                state,
            )
        try:
            if self.schema is not None:
                jsonschema.validate(json_value, self.schema)
        except jsonschema.ValidationError as e:
            raise Invalid(
                self.message('badjson', state, error=e.message),
                value,
                state,
            )
        return json_value


class Consumer(FancyValidator):
    not_empty = True
    strip = True

    def _to_python(self, value, state):
        return OneOf(
            get_grants_config().get_all_consumers(),
            messages={'notIn': 'Invalid consumer'},
        ).to_python(value, state)


class ContentRatingClass(Int):
    strip = True
    not_empty = True
    min = 0
    max = 255


class Place(FancyValidator):
    """
    поле `place` соц. брокера
    """
    not_empty = True
    strip = True

    def _to_python(self, value, state):
        return OneOf(
            ['fragment', 'query'],
            not_empty=True,
            messages={'notIn': 'Invalid place'},
        ).to_python(value, state)


class Gender(FancyValidator):  # originally from TurboGears

    male_values = ['1', 'male', 'm']
    female_values = ['2', 'female', 'f']
    sexless_values = ['0', 'unknown', 'u']

    messages = dict(
        badGender=_('Value should be %(male)r or %(female)r or %(sexless)r'),
    )

    def _to_python(self, value, state):
        if isinstance(value, six.string_types):
            value = value.strip().lower()
            if value in self.male_values:
                return gender.Gender.Male
            if value in self.female_values:
                return gender.Gender.Female
            if value in self.sexless_values:
                return gender.Gender.Unknown

            message = self.message(
                'badGender',
                state,
                male=self.male_values[0],
                female=self.female_values[0],
                sexless=self.sexless_values[0],
            )

            raise Invalid(
                message,
                value,
                state,
            )
        return value

    def _from_python(self, value, state):
        return str(value)


class Uid(Int):
    strip = True
    not_empty = True
    min = 0
    max = MAX_UID


class KpId(Int):
    strip = True
    not_empty = True
    min = MIN_KPID
    max = MAX_KPID


class DomainId(Int):
    strip = True
    not_empty = True
    min = 1
    max = MAX_DOMAIN_ID


class KarmaPrefix(Int):
    """
    Валидатор для префикса кармы
    """
    min = 0
    max = 9
    if_missing = None
    not_empty = True


class KarmaSuffix(compound.All):
    """
    Валидатор для суффикса кармы
    """
    validators = [OneOf([0, 75, 80, 85, 100], if_missing=None, not_empty=True), Int(if_missing=None)]


class TrackId(FancyValidator):
    not_empty = True

    messages = {
        'invalidId': _('Invalid track id value'),
    }

    def _to_python(self, value, state):
        try:
            check_track_id(value)
            return value
        except InvalidTrackIdError:
            raise Invalid(self.message('invalidId', state), value, state)


PERSISTENT_TRACK_ID_HEX_LENGTH = 32
PERSISTENT_TRACK_KEY_REGEX = re.compile(
    r'^(?P<track_key>(?P<track_id>[a-f0-9]{%d})(?P<uid>[a-f0-9]{1,%d}))$' % (
        PERSISTENT_TRACK_ID_HEX_LENGTH,
        MAX_UID_HEX_LENGTH,
    ),
)

MAGIC_LINK_SECRET_HEX_LENGTH = 20
MAGIC_LINK_SECRET_REGEX = re.compile(
    r'^(?P<secret>(?P<key>[a-f0-9]{%d})(?P<uid>[a-f0-9]{1,%d}))$' % (
        MAGIC_LINK_SECRET_HEX_LENGTH,
        MAX_UID_HEX_LENGTH,
    ),
)


class RandomKeyWithUid(TransformingRegex):
    not_empty = True
    strip = True

    def _to_python(self, value, state):
        parsed_value = super(RandomKeyWithUid, self)._to_python(value, state)
        parsed_value['uid'] = int(parsed_value['uid'], 16)
        try:
            Uid().to_python(parsed_value['uid'])
        except Invalid:
            raise Invalid(self.message('invalid', state), value, state)

        return parsed_value


class PersistentTrackKey(RandomKeyWithUid):
    messages = {
        'invalid': _('Invalid track key value'),
    }
    regex = PERSISTENT_TRACK_KEY_REGEX


class MagicLinkSecretKey(RandomKeyWithUid):
    messages = {
        'invalid': _('Invalid magic link secret value'),
    }
    regex = MAGIC_LINK_SECRET_REGEX


class CaptchaChecks(Int):
    if_missing = None
    not_empty = True
    min = 1
    max = 9


class HintQuestionId(Int):
    if_missing = None
    not_empty = True
    min = 1
    max = 19


class HintString(String):
    not_empty = True
    strip = True
    max = 100


class HintQuestion(HintString):
    """PASSP-3951 В perl сейчас такие ограничения на ручке mode=changehint"""
    if_missing = None
    max = 37


class HintAnswer(HintString):
    """PASSP-3951"""
    max = settings.HINT_ANSWER_MAX_LENGTH


class RetPath(FancyValidator):
    strip = True
    not_empty = True
    basic_schemes = {'http', 'https', ''}
    messages = dict(
        notin=u'Url "%(url)s" is not in allowed hosts range',
        nohost=u'Can\'t find hostname after "//" in "%(url)s"',
        badurl=u'Url "%(url)s" is invalid',
        badsymbol=u'Prohibited symbol "\\" has been found in "%(url)s"',
        badscheme=u'Scheme "%(scheme)s" is prohibited.',
        badhost=u'Url "%(url)s" in hosts blacklist',
    )

    def __init__(self, ignore_invalid=False, additional_allowed_schemes=None, additional_allowed_scheme_prefixes=None,
                 allow_any_host=False, *args, **kwargs):
        self.ignore_invalid = ignore_invalid
        self.additional_allowed_schemes = settings.ADDITIONAL_ALLOWED_RETPATH_SCHEMES.union(
            additional_allowed_schemes or [])
        self.additional_allowed_scheme_prefixes = settings.ALLOWED_RETPATH_SCHEME_PREFIXES.union(
            additional_allowed_scheme_prefixes or [])
        self.blacklisted_hosts = [
            re.compile(host) for host in settings.HOSTS_BLACKLIST
        ]
        self.blacklisted_hosts_paths = [
            (re.compile(host), re.compile(path)) for host, path in settings.HOSTS_WITH_PATHS_BLACKLIST
        ]
        self.allow_any_host = allow_any_host
        super(RetPath, self).__init__(*args, **kwargs)

    def _to_python(self, value, state):
        try:
            parsed = urlparse(value)
        except ValueError:
            if self.ignore_invalid:
                return
            raise Invalid(self.message('badurl', state, url=value), value, state)

        host = parsed.hostname
        scheme = parsed.scheme

        if self.additional_allowed_schemes:
            allowed_schemes = self.basic_schemes.union(self.additional_allowed_schemes)
        else:
            allowed_schemes = self.basic_schemes

        try:
            # PASSP-31277 На фронтенде много где парсер retpath с багом, который обрезает домен до "!"
            if host and '!' in host:
                raise Invalid(self.message('badsymbol', state, url=value), value, state)
            for field in (host, parsed.username, parsed.password, parsed.path, parsed.query):
                if field and any(c in settings.RETPATH_BAD_SYMBOLS for c in unquote(field)):
                    raise Invalid(self.message('badsymbol', state, url=value), value, state)

            if (
                scheme not in allowed_schemes and
                not any([
                    scheme.startswith(prefix)
                    for prefix in self.additional_allowed_scheme_prefixes or []
                ])
            ):
                raise Invalid(self.message('badscheme', state, scheme=scheme), value, state)

            # Для нестандартных разрешенных схем пропускаем любой хост.
            if scheme not in self.basic_schemes:
                return parsed.geturl()

            if not host:
                raise Invalid(self.message('nohost', state, url=value), value, state)

            if self.allow_any_host:
                return parsed.geturl()

            for hostmask in self.blacklisted_hosts:
                if hostmask.match(host):
                    raise Invalid(self.message('badhost', state, url=value), value, state)

            for hostmask, pathmask in self.blacklisted_hosts_paths:
                if hostmask.match(host) and pathmask.match(parsed.path):
                    raise Invalid(self.message('badhost', state, url=value), value, state)

            for allowed in settings.ALLOWED_HOSTS:
                if host == allowed or host.endswith('.' + allowed):
                    return parsed.geturl()

            raise Invalid(self.message('notin', state, url=value), value, state)
        except Invalid:
            if self.ignore_invalid:
                return
            raise


class SimpleUrlValidator(FancyValidator):
    strip = True
    base_allowed_schemes = {'http', 'https'}
    messages = dict(
        notin=u'Url "%(url)s" is not in allowed hosts range',
        nohost=u'Can\'t find hostname after "//" in "%(url)s"',
        badurl=u'Url "%(url)s" is invalid',
        relativeurl=u'Url "%(url)s" is relative, which is prohibited',
        badsymbol=u'Prohibited symbol "\\" has been found in "%(url)s"',
        badscheme=u'Scheme "%(scheme)s" is prohibited.',
        hasfragment=u'Url "%(url)s" has fragments, which is prohibited',
    )

    def __init__(self,
                 additional_allowed_schemes=None,
                 allow_empty_host=False,
                 allow_relative=False,
                 allow_fragments=False,
                 allowed_hosts=None,
                 *args,
                 **kwargs):
        self.allow_empty_host = allow_empty_host
        self.allow_relative = allow_relative
        self.allow_fragments = allow_fragments
        self.allowed_hosts = allowed_hosts
        self.allowed_schemes = self.base_allowed_schemes.union(set(additional_allowed_schemes or []))
        super(SimpleUrlValidator, self).__init__(*args, **kwargs)

    def _to_python(self, value, state):
        try:
            parsed = urlparse(value)
        except ValueError:
            raise Invalid(self.message('badurl', state, url=value), value, state)

        try:
            url_to_ascii(value)
        except:
            raise Invalid(self.message('badurl', state, url=value), value, state)

        scheme, host, _, _, _, fragment = parsed

        if not (scheme in self.allowed_schemes or self.allow_relative):
            raise Invalid(self.message('badscheme', state, scheme=scheme), value, state)

        if scheme and not host and not self.allow_empty_host:
            raise Invalid(self.message('nohost', state, url=value), value, state)

        if not (host or self.allow_relative or self.allow_empty_host):
            raise Invalid(self.message('relativeurl', state, url=value), value, state)
        elif host and self.allowed_hosts:
            for allowed in self.allowed_hosts:
                if host == allowed or host.endswith('.' + allowed):
                    return parsed.geturl()
            raise Invalid(self.message('notin', state, url=value), value, state)

        if not self.allow_fragments and fragment:
            raise Invalid(self.message('hasfragment', state, url=value), value, state)

        if any(c in settings.RETPATH_BAD_SYMBOLS for c in unquote(value)):
            raise Invalid(self.message('badsymbol', state, url=value), value, state)

        return parsed.geturl()


class FieldsMatch(FormValidator):
    field_names = None
    validate_partial_form = True

    __unpackargs__ = ('*', 'field_names')

    messages = {
        'invalidNoMatch': _('Fields do not match: %(base_field)s != %(other_fields)s'),
    }

    def __init__(self, *args, **kwargs):
        super(FieldsMatch, self).__init__(*args, **kwargs)
        if len(self.field_names) < 2:
            raise TypeError('FieldsMatch() requires at least two field names')

    def validate_partial(self, field_dict, state):
        for name in self.field_names:
            if name not in field_dict:
                return
        self.validate_python(field_dict, state)

    def validate_python(self, field_dict, state):
        ref = field_dict.get(self.field_names[0], '')

        not_matched_fields = []
        for name in self.field_names[1:]:
            if field_dict.get(name, '') != ref:
                not_matched_fields.append(name)
        if not_matched_fields:
            message = self.message('invalidNoMatch', state,
                                   base_field=self.field_names[0],
                                   other_fields=', '.join(self.field_names[1:]))
            raise Invalid(message, field_dict, state,
                          error_dict={'form': Invalid(message, field_dict, state)})


class RequireIfEquals(FormValidator):
    """
    Если значение параметра ``field_to_check`` совпадает с value_to_check
    Валидатор требует наличия всех полей из ``fields_required``
    Требуется любое не `None` значение
    """

    fields_required = None
    field_to_check = None
    value_to_check = None

    __unpackargs__ = ('fields_required', 'field_to_check', 'value_to_check')

    def _to_python(self, value_dict, state):
        # Выберем имена требуемых полей, которые пусты
        none_fields = list(filter(
            lambda field: value_dict.get(field) is None,
            self.fields_required,
        ))
        is_error = (
            value_dict.get(self.field_to_check) == self.value_to_check and
            len(none_fields)
        )
        if is_error:
            raise Invalid(
                _('You must give values for these fields: %s') % ', '.join(self.fields_required),
                value_dict, state,
                error_dict=dict([
                    (
                        field,
                        Invalid(
                            self.message('empty', state),
                            value_dict.get(field),
                            state,
                        ),
                    )
                    for field in none_fields
                ]),
            )
        return value_dict


class Base64String(String):
    messages = {
        'invalid': _('Not a base64 string'),
    }

    def __init__(self, decode=True, *args, **kwargs):
        super(Base64String, self).__init__(*args, **kwargs)
        self.decode = decode

    def _to_python(self, value, state):
        value_with_padding = value
        if len(value_with_padding) % 4:
            value_with_padding += '=' * (4 - len(value) % 4)
        try:
            result = base64.b64decode(smart_bytes(value_with_padding))
        except (TypeError, UnicodeEncodeError, binascii.Error):
            raise Invalid(self.message('invalid', state), value, state)

        if value and not result:
            # b64decode вопреки документации молча пропускает неалфавитные символы
            raise Invalid(self.message('invalid', state), value, state)

        return result if self.decode else value


class RestoreIdValidator(FancyValidator):
    not_empty = True

    messages = {
        'invalid': _('Restore ID should be in format: host_id,pid,timestamp,uid,track_id'),
    }

    def _to_python(self, value, state):
        if not isinstance(value, six.string_types):
            raise Invalid(self.message('invalid', state), value, state)
        try:
            return restore_id.RestoreId.from_string(value)
        except (TypeError, ValueError):
            raise Invalid(self.message('invalid', state), value, state)


class TotpPinValidator(String):
    strip = True
    messages = {
        'invalid': _('Invalid pin: should be integer between 0 and %s' % MAX_PIN),
        'weak': _(
            'Weak pin: should not consist of all same digits or be like year, phone number or credit card number',
        ),
    }
    year_regex = re.compile(r'^(19|20)\d{2}$')

    def __init__(self, phone_number=None, country=None, birthday=None, **kwargs):
        self._phone_number = phone_number
        self._country = country
        self._birthday = birthday
        super(TotpPinValidator, self).__init__(**kwargs)

    def digits_of(self, value):
        return [int(d) for d in str(value)]

    def is_of_same_digits(self, pin):
        return len(set(pin)) == 1

    def is_ascending(self, pin):
        digits = self.digits_of(pin)
        return all(digits[i + 1] - digits[i] == 1 for i in range(len(digits) - 1))

    def is_descending(self, pin):
        digits = self.digits_of(pin)
        return all(digits[i] - digits[i + 1] == 1 for i in range(len(digits) - 1))

    def is_like_phone_number(self, pin):
        if not self._phone_number:
            # FIXME: костыль. Убрать, когда отвалятся нетрековые версии ручек валидации
            return False
        return self._phone_number.is_similar_to(pin, self._country)

    def is_like_birthday(self, pin):
        if not self._birthday:
            # FIXME: костыль. Убрать, когда отвалятся нетрековые версии ручек валидации
            return False

        if self._birthday.is_year_set and self._birthday.date.year >= MIN_YEAR:
            templates = NUMBER_FROM_DATE_TEMPLATES_WITH_YEAR + NUMBER_FROM_DATE_TEMPLATES_WITHOUT_YEAR
            birthdate = self._birthday.date
        else:
            templates = NUMBER_FROM_DATE_TEMPLATES_WITHOUT_YEAR
            # strftime не умеет работать с датами до 1900 года, поэтому соберём фейковую дату
            birthdate = self._birthday.date.replace(year=MIN_YEAR)

        return any(pin == birthdate.strftime(template) for template in templates)

    def is_weak(self, pin):
        return (
            self.year_regex.match(pin) or
            self.is_of_same_digits(pin) or
            self.is_ascending(pin) or
            self.is_descending(pin) or
            self.is_like_phone_number(pin) or
            self.is_like_birthday(pin)
        )

    def _to_python(self, value, state):
        try:
            if len(value) < PIN_DIGITS_MIN_COUNT or len(value) > PIN_DIGITS_MAX_COUNT:
                raise Invalid(self.message('invalid', state), value, state)
            pin = parse_pin_as_string(value)
            if self.is_weak(pin):
                raise Invalid(self.message('weak', state), value, state)
            return pin
        except InvalidPinError:
            raise Invalid(self.message('invalid', state), value, state)


class AltDomainAliasValidator(FancyValidator):
    strip = True
    not_empty = True

    messages = {
        'invalid': _('Invalid alt domain alias'),
    }

    def __init__(self, allowed_domains=None, *args, **kwargs):
        super(AltDomainAliasValidator, self).__init__(*args, **kwargs)
        self.allowed_domains = set(allowed_domains or settings.ALT_DOMAINS)
        if not self.allowed_domains.issubset(set(settings.ALT_DOMAINS)):
            raise ValueError('Invalid alt domain')

    def _validate_login(self, login, domain, state):
        # провалидируем логинную часть в зависимости от требований конкретного домена
        if domain in ALT_DOMAIN_LOGIN_VALIDATORS:
            login_validator = ALT_DOMAIN_LOGIN_VALIDATORS[domain]()
            login_validator.to_python(login, state)

    def _to_python(self, value, state):
        login = raw_login_from_email(value)
        domain = domain_from_email(value).lower()
        if domain not in self.allowed_domains:
            raise Invalid(self.message('invalid', state), value, state)
        try:
            self._validate_login(login, domain, state)
            return '%s@%s' % (normalize_login(login), domain)
        except Invalid:
            raise Invalid(self.message('invalid', state), value, state)


class ExternalIPAddress(IPAddress):
    strip = True
    not_empty = True

    messages = {
        'yandexIPAddress': _('Yandex\'s IP address'),
    }

    def _to_python(self, value, state):
        ip = super(ExternalIPAddress, self)._to_python(value, state)
        if get_net(str(ip), {})['is_yandex']:
            raise Invalid(
                self.message('yandexIPAddress', state),
                value,
                state,
            )
        return ip


class ASCIIString(String):
    strip = True
    not_empty = True
    max = MAX_ASCII_STRING_LENGTH
    allow_new_lines = False

    messages = {
        'nonascii': _('Non-ASCII symbols found'),
    }

    def _to_python(self, value, state):
        self.assert_string(value, state)
        if not is_printable_ascii(value, self.allow_new_lines):
            raise Invalid(self.message('nonascii', state), value, state)
        return super(ASCIIString, self)._to_python(value, state)


DEVICE_ID_REGEX = re.compile(r'^([ -~]){6,50}$')  # все ASCII-символы с кодами от 32 до 126


class DeviceId(Regex):
    if_missing = None
    not_empty = True
    strip = True
    regex = DEVICE_ID_REGEX


class DeviceName(String):
    if_missing = None
    if_empty = None
    strip = True
    max = 100


class MusicPromoCode(String):
    not_empty = True
    max = 255


class MailishId(String):
    """
    Нечувствительная к регистру строка в base32
    Результат выдаем в нижнем регистре без падинга
    """

    if_empty = None
    strip = True
    max = 255

    messages = {
        'invalid': _('Not a base32 string'),
    }

    def _to_python(self, value, state):
        result = value.rstrip('=').lower()
        len_result = len(result)
        try:
            b32_value = result
            if len_result % 8:
                b32_value = result.ljust(len_result + 8 - len_result % 8, '=')
            base64.b32decode(smart_bytes(b32_value), casefold=True)
        except (TypeError, UnicodeEncodeError, binascii.Error):
            raise Invalid(self.message('invalid', state), value, state)

        return result


class UberId(ASCIIString):
    max = 40

    def _to_python(self, value_dict, state):
        return normalize_login(super(UberId, self)._to_python(value_dict, state))


class DriveDeviceIdValidator(Regex):
    strip = True
    # По условию задачи device_id такой
    regex = re.compile(r'^[0-9a-fA-F]{1,32}$')
    not_empty = True


class SignatureValidator(String):
    not_empty = True
    strip = True


class BlackboxSignSpaceValidator(String):
    not_empty = True
    strip = True


class BillingFeatures(FancyValidator):
    FEATURE_NAME_RE = re.compile(r'^[abcdefghijklmnopqrstuvwxyz0123456789_-]{1,20}$')
    STRING_ATTRIBUTES_RE = {
        'brand': FEATURE_NAME_RE,
    }
    MAX_FEATURES_COUNT = 20
    DEPRECATED_ATTRIBUTES_NAMES = ['name']
    messages = {
        'badjson': _('Cannot parse/validate JSON: %(error)s'),
    }

    def __init__(self, *args, **kwargs):
        super(BillingFeatures, self).__init__(*args, **kwargs)

    def _check_features(self, value):
        if len(value) > self.MAX_FEATURES_COUNT:
            raise ValueError('Max features count is {}'.format(self.MAX_FEATURES_COUNT))

        for name, feature in value.items():
            if self.FEATURE_NAME_RE.search(name) is None:
                raise ValueError(
                    'The name of the feature ({}) doesn\'t match the pattern {}'.format(
                        name,
                        self.FEATURE_NAME_RE.pattern,
                    ),
                )

            if not isinstance(feature, dict):
                raise ValueError(
                    'A feature ({}) value is not a dictionary'.format(name),
                )

            try:
                fa = FeatureAttributes()
                for option_name, option_value in feature.items():
                    if option_name in self.DEPRECATED_ATTRIBUTES_NAMES:
                        raise ValueError('"{}" is a deprecated attribute name'.format(option_name))
                    if option_name in self.STRING_ATTRIBUTES_RE:
                        if self.STRING_ATTRIBUTES_RE[option_name].search(option_value) is None:
                            raise ValueError(
                                'The value ({}) of attribute ({}) of the feature ({}) doesn\'t match the pattern {}'.format(
                                    option_value,
                                    option_name,
                                    name,
                                    self.STRING_ATTRIBUTES_RE[option_name].pattern,
                                ),
                            )
                    setattr(fa, snake_case_to_camel_case(option_name), option_value)
                fa.SerializeToString()
            except Exception as e:
                raise ValueError(str(e))

    def _to_python(self, value, state):
        try:
            json_value = json.loads(value)

            if not isinstance(json_value, dict):
                raise ValueError('A field value is not a dictionary')

            self._check_features(json_value)
        except ValueError as e:
            raise Invalid(self.message('badjson', state, error=e), value, state)

        return json_value


class PlusSubscriberStateValidator(FancyValidator):
    MAX_FEATURES_COUNT = 50
    FEATURE_VALUE_RE = re.compile(r'^[\x20-\x7E]{1,40}$')

    messages = {
        'badjson': _('Cannot parse/validate JSON: %(error)s'),
    }

    def __init__(self, *args, **kwargs):
        super(PlusSubscriberStateValidator, self).__init__(*args, **kwargs)

    def _validate_features(self, features):
        if len(features) > self.MAX_FEATURES_COUNT:
            raise ValueError('Feature list size must be less than {}'.format(self.MAX_FEATURES_COUNT))

        for feature in features:
            if not feature.Id:
                raise ValueError('Feature must contain an Id')

            if feature.Value and not self.FEATURE_VALUE_RE.match(feature.Value):
                raise ValueError('Feature value must match the pattern {}'.format(self.FEATURE_VALUE_RE.pattern))

    def _convert(self, serialized):
        deserialized = json.loads(serialized)

        if not isinstance(deserialized, dict):
            raise ValueError('A field value is not a dictionary')

        if not deserialized:
            return ''

        pss = PlusSubscriberState()

        try:
            json_format.ParseDict(deserialized, pss)
        except json_format.Error as e:
            raise ValueError(str(e))

        available_features = pss.AvailableFeatures
        available_features_end = pss.AvailableFeaturesEnd
        frozen_features = pss.FrozenFeatures

        if not available_features and not frozen_features:
            raise ValueError('At least AvailableFeatures or FrozenFeatures must be in the state')

        if available_features:
            self._validate_features(available_features)

            if not available_features_end:
                raise ValueError('AvailableFeaturesEnd must be in a state if AvailableFeatures list is not empty')

        if frozen_features:
            self._validate_features(frozen_features)

        try:
            # Выполняем явную сериализацию в proto, чтобы убедиться в соответствии переданных значений их типам
            pss.SerializeToString()
        except EncodeError as e:
            raise ValueError(str(e))

        try:
            return convert_plus_subscriber_state_proto_to_json(pss)
        except ValueError:
            raise

    def _to_python(self, value, state):
        try:
            return self._convert(value)
        except ValueError as e:
            raise Invalid(self.message('badjson', state, error=e), value, state)


class MailSubscriptionSetValidator(JSONValidator):
    def __init__(self):
        super(MailSubscriptionSetValidator, self).__init__(
            schema={
                'type': 'object',
                'patternProperties': {
                    r'^\d+$': {'type': 'boolean'},
                },
                'propertyNames': {
                    'pattern': r'^\d+$',
                },
            },
        )

    def _to_python(self, value, state):
        json_value = super(MailSubscriptionSetValidator, self)._to_python(value, state)
        json_value = {int(service_id): is_subscribed for service_id, is_subscribed in json_value.items()}
        return json_value


class Uuid(Regex):
    regex = re.compile(
        r'^[a-f0-9]{8}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{12}$',
    )


class PublicId(Login):
    _field_name = 'public_id'


class AvatarId(Regex):
    not_empty = True

    # 20 байт на представление строкой 2 ** 64
    # 200 байт на представление 1024 бит в base64_urlsafe (172 байта) и суффикс
    regex = re.compile('^[-0-9A-Za-z]{1,20}/[-0-9A-Za-z_=]{1,200}$')

    strip = True


class FamilyId(String):
    not_empty = True
    strip = True


class Version(Regex):
    regex = re.compile(
        r'^\d+(?:\.\d+)*$',
    )


def validators():
    """Return the names of all validators in this module."""
    return [name for name, value in globals().items()
            if isinstance(value, type) and issubclass(value, Validator)]


__all__ = ['Invalid', 'compound', '_', 'State', 'NAME_WHITESPACE_RE', 'MAGIC_LINK_SECRET_HEX_LENGTH'] + validators()
