# -*- coding: utf-8 -*-

from lxml import etree
from lxml.etree import (
    XML,
    XMLSyntaxError,
    XPath,
)
from passport.backend.utils.common import (
    float_or_default,
    int_or_default,
)


class FakeDTDResolver(etree.Resolver):
    def resolve(self, url, id, context):
        return self.resolve_string('', context)  # pragma: no cover


class XMLParser(etree.XMLParser):
    def __init__(self, *args, **kwargs):
        # resolve_entities запрещены в связи с этим тикетом:
        # https://st.yandex-team.ru/IS-408
        if kwargs.get('resolve_entities'):
            raise ValueError('resolve_entities must be disabled')
        kwargs['resolve_entities'] = False
        super(XMLParser, self).__init__(*args, **kwargs)
        self.resolvers.add(FakeDTDResolver())


def xpath_first(tree, pattern):
    result = pattern(tree)
    if result:
        return result[0]
    return None


def xpath_to_bool(tree, pattern, default=False):
    try:
        # Может быть None, может быть 0, может быть пусто
        flag = bool(int(xpath_first(tree, pattern)))
    except (TypeError, ValueError):
        flag = default
    return flag


def xpath_to_int(tree, pattern, default=None):
    return int_or_default(xpath_first(tree, pattern), default)


def xpath_to_float(tree, pattern, default=None):
    return float_or_default(xpath_first(tree, pattern), default)


__all__ = (
    FakeDTDResolver,
    XMLSyntaxError,
    XPath,
    XML,
    XMLParser,
    xpath_to_bool,
    xpath_first,
    xpath_to_int,
    xpath_to_float,
)
