# -*- coding: utf-8 -*-

import logging

from passport.backend.core.builders.yasms.exceptions import YaSmsError
from passport.backend.core.conf import settings
from passport.backend.core.mailer.utils import (
    get_tld_by_country,
    login_shadower,
    MailInfo,
    make_email_context,
    send_mail_for_account,
)
from passport.backend.core.models.account import get_preferred_language
from passport.backend.core.models.phones.phones import (
    RemoveSecureOperation,
    ReplaceSecurePhoneWithBoundPhoneOperation,
    ReplaceSecurePhoneWithNonboundPhoneOperation,
)


logger = logging.getLogger('passport.yasms')


def notify_user_by_email(
    account,
    body_template_pathname,
    subject_phrase_key,
    language=None,
    template_context=None,
):
    if language is None:
        language = get_preferred_language(account)
    context = make_email_context(
        language=language,
        account=account,
        context=template_context,
    )
    phrases = settings.translations.NOTIFICATIONS[language]
    send_mail_for_account(
        body_template_pathname,
        MailInfo(
            subject=phrases[subject_phrase_key],
            from_=phrases[u'email_sender_display_name'],
            tld=get_tld_by_country(account.person.country),
        ),
        context,
        account,
        context_shadower=login_shadower,
    )


def notify_user_by_email_that_phone_removal_started(account, **kwargs):
    """
    Выслать пользователю электронное письмо с уведомлением об удалении
    его номера телефона.
    """
    email_body_template = u'mail/phone_removal_started.html'
    if not account.have_password:
        email_body_template = u'mail/phone_removal_started_passwordless.html'

    notify_user_by_email(
        account,
        email_body_template,
        u'phone_removal_started_email_subject',
        **kwargs
    )


def notify_user_by_email_that_secure_phone_removed_without_quarantine(account, **kwargs):
    """
    Выслать пользователю электронное письмо с уведомлением об удалении
    защищённого номера без карантина.
    """
    email_body_template = u'mail/secure_phone_removed_without_quarantine.html'
    if not account.have_password:
        email_body_template = u'mail/secure_phone_removed_without_quarantine_passwordless.html'

    notify_user_by_email(
        account=account,
        body_template_pathname=email_body_template,
        subject_phrase_key=u'secure_phone_removed_without_quarantine_email_subject',
        **kwargs
    )


def notify_user_by_email_that_secure_phone_removed_with_quarantine(account, **kwargs):
    """
    Выслать пользователю электронное письмо с уведомлением об удалении
    защищённого номера прошедшего карантин.
    """
    email_body_template = u'mail/secure_phone_removed_with_quarantine.html'
    if not account.have_password:
        email_body_template = u'mail/secure_phone_removed_with_quarantine_passwordless.html'

    notify_user_by_email(
        account=account,
        body_template_pathname=email_body_template,
        subject_phrase_key=u'secure_phone_removed_with_quarantine_email_subject',
        **kwargs
    )


def notify_user_by_email_that_secure_phone_bound(account, **kwargs):
    """
    Выслать пользователю электронное письмо с уведомлением о защите телефона.
    """
    email_body_template = u'mail/phone_secured.html'
    if not account.have_password:
        email_body_template = u'mail/phone_secured_passwordless.html'

    notify_user_by_email(
        account,
        email_body_template,
        u'phone_secured_email_subject',
        **kwargs
    )


notify_user_by_email_that_secure_phone_binding_started = notify_user_by_email_that_secure_phone_bound


def notify_user_by_email_that_secure_phone_replacement_started(account, **kwargs):
    """
    Выслать пользователю электронное письмо с уведомлением о начале замены
    защищённого телефона через карантин.
    """
    email_body_template = u'mail/secure_phone_replacement_started.html'
    if not account.have_password:
        email_body_template = u'mail/secure_phone_replacement_started_passwordless.html'

    notify_user_by_email(
        account=account,
        body_template_pathname=email_body_template,
        subject_phrase_key=u'secure_phone_replacement_started_email_subject',
        **kwargs
    )


def notify_user_by_email_that_secure_phone_replaced(account, **kwargs):
    """
    Выслать пользователю электронное письмо с уведомлением о том, что
    защищённый телефон заменён другим.
    """
    notify_user_by_email(
        account=account,
        body_template_pathname=u'mail/secure_phone_replaced.html',
        subject_phrase_key=u'secure_phone_replaced_email_subject',
        **kwargs
    )


class notify_about_phone_changes(object):
    """
    Выслать пользователю электронное письмо и СМС с уведомлением о телефонных
    изменениях, которые произошли во время работы этого контекстного менеджера.
    """
    def __init__(self, account, yasms_builder, statbox, consumer,
                 language=None, action=None, client_ip=None, user_agent=None,
                 view=None):
        self._account = account
        self._yasms_builder = yasms_builder
        self._statbox = statbox
        self._consumer = consumer
        self._language = language
        self._snapshot = None
        self._client_ip = client_ip
        self._user_agent = user_agent
        self._view = view

    def __enter__(self):
        self.start()

    def __exit__(self, exc_type, exc_value, traceback):
        if exc_type is None:
            self.finish()

    def start(self):
        self._snapshot = self._account.phones.snapshot()

    def finish(self):
        account = self._account
        phones = self._account.phones

        old_secure = self._snapshot.secure
        new_secure = phones.secure
        old_logical_op = self._snapshot.get_secure_logical_operation(self._statbox)
        new_logical_op = phones.get_secure_logical_operation(self._statbox)

        is_quarantine_started = _in_quarantine(new_logical_op) and not _in_quarantine(old_logical_op)

        notify_account_modification = False
        if is_quarantine_started:
            if _is_replacement(new_logical_op):
                # Операция замены попала в карантин
                notify_user_by_email_that_secure_phone_replacement_started(account, language=self._language)
            elif _is_removal(new_logical_op):
                # Операция удаления попала в карантин
                notify_user_by_email_that_phone_removal_started(account, language=self._language)
            notify_account_modification = True

        if not old_secure and new_secure:
            notify_user_by_email_that_secure_phone_bound(account, language=self._language)
            notify_account_modification = True
        elif old_secure and not new_secure:
            if _in_quarantine(old_logical_op):
                notify_user_by_email_that_secure_phone_removed_with_quarantine(account, language=self._language)
            else:
                notify_user_by_email_that_secure_phone_removed_without_quarantine(account, language=self._language)
                notify_account_modification = True
        elif old_secure and new_secure and old_secure != new_secure:
            notify_user_by_email_that_secure_phone_replaced(account, language=self._language)
            notify_account_modification = True

        if (
            _is_removal(new_logical_op) and
            (
                # О начале удаления уведомляем, либо когда началась операция
                # удаления и пользователь указал, что владеет защищённым
                # номером,
                new_logical_op.does_user_admit_phone and not _is_removal(old_logical_op) or
                # или когда операция удаления попала в карантин.
                is_quarantine_started
            )
        ):
            notify_user_by_sms_that_secure_phone_removal_started(
                account,
                self._yasms_builder,
                self._statbox,
                self._consumer,
                self._language,
                self._client_ip,
                self._user_agent,
            )

        if (
            _is_replacement(new_logical_op) and
            (
                # О начале замены уведомляем, либо когда началась операция
                # замены и пользователь указал, что владеет защищённым
                # номером,
                new_logical_op.does_user_admit_phone and not _is_replacement(old_logical_op) or
                # или когда операция замены попала в карантин.
                is_quarantine_started
            )
        ):
            notify_user_by_sms_that_secure_phone_replacement_started(
                account,
                self._yasms_builder,
                self._statbox,
                self._consumer,
                self._language,
                self._client_ip,
                self._user_agent,
            )

        # Операции над не секьюрными телефонами
        old_phones = self._snapshot.non_secure()
        new_phones = phones.non_secure()
        if not notify_account_modification:
            # Проверяем, что телефон был привязан или добавлен привязанным
            for id_, phone in new_phones.items():
                old_phone = old_phones.get(id_)
                if phone.bound and not (old_phone and old_phone.bound):
                    notify_account_modification = True
                    break
        if not notify_account_modification:
            # Проверяем, что привязанный телефон был удалён или отвязан
            for id_, old_phone in old_phones.items():
                phone = new_phones.get(id_)
                if old_phone.bound and not (phone and phone.bound):
                    notify_account_modification = True
                    break

        if notify_account_modification and self._view:
            self._view.send_account_modification_push(event_name='phone_change')


def _in_quarantine(logical_op):
    return logical_op and logical_op.in_quarantine


def _is_replacement(logical_op):
    return type(logical_op) in {
        ReplaceSecurePhoneWithBoundPhoneOperation,
        ReplaceSecurePhoneWithNonboundPhoneOperation,
    }


def _is_removal(logical_op):
    return type(logical_op) is RemoveSecureOperation


def notify_user_by_sms(phone_number, message, uid, yasms_builder, statbox,
                       consumer, identity=None, ignore_errors=True,
                       client_ip=None, user_agent=None):
    """
    Выслать на данный номер СМСку с уведомлением.

    Пишет сообщение об ошибке в статбокс и паспортный журнал, но по умолчанию не бросает
    исключений.
    """
    kwargs = {
        'number': phone_number.masked_format_for_statbox,
        'uid': uid,
    }
    if identity:
        kwargs['action'] = statbox.Link(identity)
    with statbox.make_context(**kwargs):
        try:
            response = yasms_builder.send_sms(
                phone_number.e164,
                message,
                from_uid=uid,
                caller=consumer,
                identity='%s.notify' % identity if identity else 'notify',
                client_ip=client_ip,
                user_agent=user_agent,
            )
            statbox.log(
                action=statbox.Link('notification_sent'),
                sms_id=response['id'],
            )
        except YaSmsError as e:
            statbox.log(error=u'sms.isnt_sent')
            if not ignore_errors:
                raise
            logger.error(
                'Unable to send sms to phone_number "%s" for uid "%s". '
                'Error: "%s"', phone_number, uid, e,
            )


def notify_user_by_sms_that_secure_phone_removal_started(account, yasms_builder,
                                                         statbox, consumer, language=None,
                                                         client_ip=None, user_agent=None):
    """
    Выслать пользователю СМСку с уведомлением об удалении его защищённого номера
    телефона.

    Пишет сообщение об ошибке в статбокс и паспортный журнал, но не бросает
    исключений.
    """
    if language is None:
        language = get_preferred_language(account)
    phrases = settings.translations.NOTIFICATIONS[language]

    notify_user_by_sms(
        phone_number=account.phones.secure.number,
        message=phrases['phone_removal_started_sms'],
        uid=account.uid,
        yasms_builder=yasms_builder,
        statbox=statbox,
        consumer=consumer,
        identity='notify_user_by_sms_that_secure_phone_removal_started',
        client_ip=client_ip,
        user_agent=user_agent,
    )


def notify_user_by_sms_that_secure_phone_replacement_started(account, yasms_builder,
                                                             statbox, consumer, language=None,
                                                             client_ip=None, user_agent=None):
    if language is None:
        language = get_preferred_language(account)
    phrases = settings.translations.NOTIFICATIONS[language]

    notify_user_by_sms(
        phone_number=account.phones.secure.number,
        message=phrases['notify_user_by_sms_that_secure_phone_replacement_started'],
        uid=account.uid,
        yasms_builder=yasms_builder,
        statbox=statbox,
        consumer=consumer,
        identity='notify_user_by_sms_that_secure_phone_replacement_started',
        client_ip=client_ip,
        user_agent=user_agent,
    )


def notify_user_by_email_that_phone_unbound(
    account,
    phone_number,
    phone_was_secure=True,
):
    """
    Выслать пользователю электронное письмо с уведомлением о автоматической
    отвязке телефона из-за того, что телефон привязали к другому аккаунту.

    phone_was_secure Признак, что отвязан основной телефон
    """
    email_body_template = 'mail/phone_unbound_secure.html'

    if not phone_was_secure or not account.have_password:
        email_body_template = 'mail/phone_unbound_simple.html'

    template_context = dict(
        feedback_key='feedback.robot',
        feedback_url_key='feedback_url.phone',
        PHONE=phone_number.masked_format_for_challenge,
        signature_key='signature.soulless_robot',
    )

    notify_user_by_email(
        account,
        email_body_template,
        'phone_unbound.email_subject',

        # Письма про безопасность должны быть на языке аккаунта
        language=None,

        template_context=template_context,
    )
