# -*- coding: utf-8 -*-

from passport.backend.core.builders.blackbox.utils import add_phone_arguments
from passport.backend.core.exceptions import UnknownUid
from passport.backend.core.models.account import Account
from passport.backend.core.types.phone_number.phone_number import (
    PhoneNumber,
    PhoneNumbering,
)
from passport.backend.core.undefined import Undefined
from passport.backend.core.yasms.phonenumber_alias import Aliasification


def _merge_phones(old, new, account):
    """
    Удаляет один из телефонов, когда на аккаунте есть, и старый, и новый телефон
    """
    assert old
    assert new

    if new is account.phones.secure:
        useless_phone = old
    elif old is account.phones.secure:
        useless_phone = new
    else:
        operation_on_new = new.get_logical_operation(None)
        operation_on_old = old.get_logical_operation(None)

        if operation_on_new and operation_on_new.is_secure:
            useless_phone = old
        elif operation_on_old and operation_on_old.is_secure:
            useless_phone = new
        elif new.bound:
            useless_phone = old
        elif old.bound:
            useless_phone = new
        elif operation_on_new and operation_on_new.is_binding:
            useless_phone = old
        elif operation_on_old and operation_on_old.is_binding:
            useless_phone = new
        else:
            useless_phone = old

    if useless_phone is new:
        old.number = new.number
    account.phones.remove(useless_phone)

    return useless_phone


class PhoneNumberTranslator(object):
    """
    С помощью данного правила преобразует старый телефон в новый.

    Если для старого телефона уже есть новый телефон, то один из них удаляется,
    а второй, если нужно, преобразуется в новый.
    """
    def __init__(self):
        self._account = None
        self._aliasification = Undefined
        self._blackbox = None
        self._consumer = None
        self._new_phone = Undefined
        self._new_phone_number = Undefined
        self._old_phone = Undefined
        self._old_phone_number = None
        self._old_phonenumber_alias_owner = Undefined
        self._rule = None
        self._statbox = None

    @classmethod
    def build_from_account(
        cls,
        account,
        blackbox,
        consumer,
        old_phone_number,
        rule,
        statbox,
    ):
        self = PhoneNumberTranslator()
        self._account = account
        self._blackbox = blackbox
        self._consumer = consumer
        self._old_phone_number = old_phone_number
        self._rule = rule
        self._statbox = statbox
        return self

    @classmethod
    def build_from_uid(
        cls,
        uid,
        blackbox,
        consumer,
        old_phone_number,
        statbox,
    ):
        account = PhoneNumberTranslator._get_account_by_uid(blackbox, uid)
        rule = PhoneNumberTranslator._find_phone_translation_rule(old_phone_number)

        self = PhoneNumberTranslator()
        self._account = account
        self._blackbox = blackbox
        self._consumer = consumer
        self._old_phone_number = old_phone_number
        self._rule = rule
        self._statbox = statbox

        return self

    @property
    def account(self):
        return self._account

    def check_all(self):
        self.check_rule_applicable_to_phone()
        self.check_phone_belongs_to_account()
        self.check_phonenumber_alias_allows_translation()

    def check_phone_belongs_to_account(self):
        if not self.old_phone:
            raise self.PhoneNotFoundError()

    def check_phonenumber_alias_allows_translation(self):
        if self._account.phonenumber_alias and self._account.phonenumber_alias.number == self._old_phone_number:
            if self._account.phonenumber_alias.enable_search:
                # Нельзя менять ПЦА, потому что это почтовый адрес, на который
                # пользователю приходят письма.
                raise self.UnableToTranslatePhoneNumber()

            self._aliasification = self._build_aliasification()

            self._old_phonenumber_alias_owner = self._aliasification.get_owner()
            if self._old_phonenumber_alias_owner:
                if self._old_phonenumber_alias_owner.is_neophonish:
                    # У неофониша невозможно отобрать ЦА, потому что это его
                    # основной и единственный способ входа.
                    raise self.UnableToTranslatePhoneNumber()
                if self._old_phonenumber_alias_owner.phonenumber_alias.enable_search:
                    # Нельзя отбирать ПЦА, потому что это почтовый адрес, на
                    # который пользователю приходят письма.
                    raise self.UnableToTranslatePhoneNumber()

    def check_rule_applicable_to_phone(self):
        if not self._rule.e164_is_old(self._old_phone_number.e164):
            raise self.InapplicablePhoneTranslationRule()

    @property
    def new_phone(self):
        if self._new_phone is Undefined:
            self._new_phone = self._account.phones.by_number(self.new_phone_number)
        return self._new_phone

    @property
    def new_phone_number(self):
        if self._new_phone_number is Undefined:
            self._new_phone_number = PhoneNumber.parse(self._rule.old_e164_to_new(self._old_phone_number.e164))
        return self._new_phone_number

    @property
    def old_phone(self):
        if self._old_phone is Undefined:
            self._old_phone = self._account.phones.by_number(self._old_phone_number)
        return self._old_phone

    @property
    def old_phone_number(self):
        return self._old_phone_number

    @property
    def old_phonenumber_alias_owner(self):
        return self._old_phonenumber_alias_owner

    def take_phonenumber_alias_away_from_old_owner(self):
        self._aliasification.take_away()

    def translate(self):
        if self.new_phone:
            _merge_phones(self.old_phone, self.new_phone, self._account)
        else:
            self.old_phone.number = self.new_phone_number

        if self._account.phones.by_number(self.new_phone_number) is self._account.phones.secure:
            if self._aliasification:
                self._aliasification.take_old_alias_away()
                self._aliasification.give_out()

    def _build_aliasification(self):
        return Aliasification(
            account=self._account,
            phone_number=self.new_phone_number,
            consumer=self._consumer,
            blackbox=self._blackbox,
            statbox=self._statbox,
            should_check_secure_phone=False,
            should_check_if_alias_allowed=False,
            enable_search=self._account.phonenumber_alias.enable_search,
        )

    @staticmethod
    def _get_account_by_uid(blackbox, uid):
        userinfo = blackbox.userinfo(**add_phone_arguments(uid=uid))
        return Account().parse(userinfo)

    @staticmethod
    def _find_phone_translation_rule(phone_number):
        for rule in PhoneNumbering.RULES:
            if rule.e164_is_old(phone_number.e164):
                return rule
        raise PhoneNumberTranslator.InapplicablePhoneTranslationRule()

    class InapplicablePhoneTranslationRule(Exception):
        pass

    class PhoneNotFoundError(Exception):
        pass

    class UnableToTranslatePhoneNumber(Exception):
        pass

    UnknownUid = UnknownUid
