# -*- coding: utf-8 -*-

from passport.backend.core.builders.blackbox import BLACKBOX_FIND_BY_PHONE_ALIAS_FORCE_ON
from passport.backend.core.conf import settings
from passport.backend.core.dbmanager.exceptions import DBError
from passport.backend.core.exceptions import UnknownUid
from passport.backend.core.mailer.utils import (
    get_tld_by_country,
    login_shadower,
    MailInfo,
    make_email_context,
    send_mail_for_account,
)
from passport.backend.core.models.account import (
    get_preferred_language,
    PhonenumberAlias,
)
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.types.phone_number.phone_number import get_alt_phone_numbers_of_phone_number
from passport.backend.core.utils.blackbox import build_account


def is_phonenumber_alias_as_email_allowed(account):
    return (
        # Почтовый цифровой алиас (ПЦА) невозможно включить без логинного
        # цифрового алиаса (ЛЦА), т.к. сейчас наличие алиаса phonenumber в БД
        # означает, что включен ЛЦА. А, если кроме алиаса, есть ещё и
        # атрибут account.enable_search_by_phone_alias, то это значит, что на
        # аккаунте включены ЛЦА и ПЦА.
        is_phonenumber_alias_as_login_allowed(account) and

        account.is_normal and
        bool(account.emails.native)
    )


def is_phonenumber_alias_as_login_allowed(account):
    return (
        account.is_lite or
        account.is_normal or
        account.is_social or
        account.is_neophonish
    )


class PhoneAliasManager(object):
    ALIAS_DELETE_REASON_OFF = u'off'
    ALIAS_DELETE_REASON_OWNER_CHANGE = u'owner_change'

    def __init__(self, consumer, environment, statbox=None, need_update_tx=False):
        self._consumer = consumer
        self._env = environment
        self._statbox = statbox
        self._need_update_tx = need_update_tx

    def is_alias_allowed(self, account,  enable_search=True):
        if enable_search:
            return is_phonenumber_alias_as_email_allowed(account)
        else:
            return is_phonenumber_alias_as_login_allowed(account)

    def create_alias(self, account, number, language=None, enable_search=True):
        # TODO: Вынести UPDATE в место вызова метода.
        if self._need_update_tx:
            with UPDATE(
                account,
                self._env,
                {
                    u'action': u'phone_alias_add',
                    u'consumer': self._consumer,
                },
            ):
                account.phonenumber_alias = PhonenumberAlias(
                    parent=account,
                    number=number,
                    enable_search=enable_search,
                )

            if enable_search and is_phonenumber_alias_as_email_allowed(account):
                # Есть почта на Яндексе
                self.send_mail_about_alias_as_login_and_email_enabled(account, language, number)
            else:
                # Нет почты на Яндексе
                self.send_mail_about_alias_as_login_enabled(account, language, number)
        else:
            account.phonenumber_alias = PhonenumberAlias(
                parent=account,
                number=number,
                enable_search=enable_search,
            )

    def delete_alias(self, account, reason=None, language=None):
        alias_number = account.phonenumber_alias.number
        # Запоминаем состояние включённости, чтобы после удаления ЦА понимать,
        # отключился ли ПЦА вместе с удалённым алиасом или нет.
        alias_as_email_turned_off = bool(account.phonenumber_alias.enable_search)

        # TODO: Вынести UPDATE в место вызова метода.
        if self._need_update_tx:
            with UPDATE(
                account,
                self._env,
                {
                    u'action': u'phone_alias_delete',
                    u'consumer': self._consumer,
                },
            ):
                account.phonenumber_alias = None
        else:
            account.phonenumber_alias = None

        if reason == self.ALIAS_DELETE_REASON_OFF:
            if alias_as_email_turned_off and is_phonenumber_alias_as_email_allowed(account):
                self.send_mail_about_alias_as_login_and_email_disabled(account, language, alias_number)
            else:
                self.send_mail_about_alias_as_login_disabled(account, language, alias_number)
        elif reason == self.ALIAS_DELETE_REASON_OWNER_CHANGE:
            if alias_as_email_turned_off and is_phonenumber_alias_as_email_allowed(account):
                self.send_mail_about_alias_as_login_and_email_owner_changed(
                    account,
                    language,
                    alias_number,
                )
            else:
                self.send_mail_about_alias_as_login_owner_changed(account, language, alias_number)

    def create_alias_with_logging(self, account, number, language=None,
                                  validation_period=0, enable_search=True):
        self._statbox.bind(
            operation=u'aliasify',
            validation_period=validation_period,
        )
        try:
            self.create_alias(
                account, number,
                language=language,
                enable_search=enable_search,
            )
            self._statbox.log()
        except DBError:
            self._statbox.log(error=u'alias.isnt_created')
            raise

    def delete_alias_with_logging(self, owner, reason, language=None):
        self._statbox.bind(
            operation=u'dealiasify',
            uid=owner.uid,
            reason=reason,
        )
        try:
            self.delete_alias(owner, reason=reason, language=language)
            self._statbox.log()
        except DBError:
            self._statbox.log(error=u'alias.isnt_deleted')
            raise

    def disable_as_email(self, account, language):
        if account.phonenumber_alias.enable_search:
            account.phonenumber_alias.enable_search = False
            self.send_mail_about_disable_alias_as_email(
                account,
                language,
                account.phonenumber_alias.number,
            )

    def enable_as_email(self, account, language):
        assert is_phonenumber_alias_as_email_allowed(account)

        if not account.phonenumber_alias.enable_search:
            account.phonenumber_alias.enable_search = True
            self.send_mail_about_enable_alias_as_email(
                account,
                language,
                account.phonenumber_alias.number,
            )

    def send_mail_about_alias_as_login_and_email_enabled(self, account, language, phone_number):
        if not is_phonenumber_alias_as_email_allowed(account):
            # TODO: Правильно было бы не включать ПЦА на аккаунте без почты (и
            # соответственно не отсылать такое письмо), но мы не можем так
            # делать пока пользователь не может управлять ПЦА из GUI.
            self.send_mail_about_alias_as_login_enabled(account, language, phone_number)
            return

        context = {
            'e164_phone': phone_number.e164,
            'digital_email': phone_number.digital + '@' + account.emails.default.domain,
            'e164_email': phone_number.e164 + '@' + account.emails.default.domain,
        }
        self._send_mail_notifications(
            account=account,
            language=language,
            subject_key='phone_alias.as_login_and_email_enabled_subject',
            template='mail/phone_alias_as_login_and_email_enabled.html',
            context=context,
        )

    def send_mail_about_alias_as_login_enabled(self, account, language, phone_number):
        language = language or get_preferred_language(account)
        phrases = settings.translations.NOTIFICATIONS[language]

        context = {
            'phone_alias_explanation_key': 'phone_alias.on.explanation.no_mailbox',
            'login_alias': phone_number.digital,
            'domain': '',
            'FORMATTED_PHONE_NUMBER': phone_number.international,
            'LOGIN_ALIAS_VALUE': phone_number.digital,
            'YASMS_VALIDATOR_URL': phrases[u'validator_url'],
            'HELP_URL': phrases[u'phone_alias.help_url'],
        }

        self._send_mail_notifications(
            account=account,
            language=language,
            subject_key='phone_alias.on.subject',
            template='mail/phone_alias_notification.html',
            context=context,
        )

    def send_mail_about_enable_alias_as_email(self, account, language, phone_number):
        if not account.emails.default:
            return

        context = {
            'digital_email': phone_number.digital + '@' + account.emails.default.domain,
            'e164_email': phone_number.e164 + '@' + account.emails.default.domain,
        }

        self._send_mail_notifications(
            account=account,
            language=language,
            subject_key='phone_alias.enable_as_email_subject',
            template='mail/phone_alias_as_email_enabled.html',
            context=context,
        )

    def send_mail_about_disable_alias_as_email(self, account, language, phone_number):
        if not account.emails.default:
            return

        context = {
            'digital_email': phone_number.digital + '@' + account.emails.default.domain,
            'e164_email': phone_number.e164 + '@' + account.emails.default.domain,
        }

        self._send_mail_notifications(
            account=account,
            language=language,
            subject_key='phone_alias.disable_as_email_subject',
            template='mail/phone_alias_as_email_disabled_by_you.html',
            context=context,
        )

    def send_mail_about_alias_as_login_and_email_disabled(self, account, language, phone_number):
        if not account.emails.default:
            return

        context = {
            'e164_phone': phone_number.e164,
            'digital_email': phone_number.digital + '@' + account.emails.default.domain,
            'e164_email': phone_number.e164 + '@' + account.emails.default.domain,
        }
        self._send_mail_notifications(
            account=account,
            language=language,
            subject_key='phone_alias.as_login_disabled_subject',
            template='mail/phone_alias_as_login_and_email_disabled_by_you.html',
            context=context,
        )

    def send_mail_about_alias_as_login_disabled(self, account, language, phone_number):
        context = {'e164_phone': phone_number.e164}
        self._send_mail_notifications(
            account=account,
            language=language,
            subject_key='phone_alias.as_login_disabled_subject',
            template='mail/phone_alias_as_login_disabled_by_you.html',
            context=context,
        )

    def send_mail_about_alias_as_login_and_email_owner_changed(self, account, language, phone_number):
        if not account.emails.default:
            return

        context = {
            'e164_phone': phone_number.e164,
            'digital_email': phone_number.digital + '@' + account.emails.default.domain,
            'e164_email': phone_number.e164 + '@' + account.emails.default.domain,
        }
        self._send_mail_notifications(
            account=account,
            language=language,
            subject_key='phone_alias.as_login_disabled_by_somebody_subject',
            template='mail/phone_alias_as_login_and_email_disabled_by_somebody.html',
            context=context,
        )

    def send_mail_about_alias_as_login_owner_changed(self, account, language, phone_number):
        context = {'e164_phone': phone_number.e164}
        self._send_mail_notifications(
            account=account,
            language=language,
            subject_key='phone_alias.as_login_disabled_by_somebody_subject',
            template='mail/phone_alias_as_login_disabled_by_somebody.html',
            context=context,
        )

    def _send_mail_notifications(self, account, language, subject_key, template,
                                 context, sender='email_sender_display_name'):
        language = language or get_preferred_language(account)

        context = make_email_context(
            language=language,
            account=account,
            context=context,
        )

        phrases = settings.translations.NOTIFICATIONS[language]

        info = MailInfo(
            subject=phrases[subject_key],
            from_=phrases[sender],
            tld=get_tld_by_country(account.person.country),
        )

        send_mail_for_account(template, info, context, account, login_shadower)


class Aliasification(object):
    # TODO Писать в event.log:
    #
    #   Для пользователя у которого отбирается алиас, идентификатор
    #   отобравшего.
    #
    #   Для пользователя, которому передаётся алиас, идентификатор прошлого
    #   владельца.
    """
    Создаёт телефонный алиас на данной учётной записи, удаляет телефонный
    алиас с прежней учётной записи и извещает пользователей.

    Входные параметры
        account
            Учётная запись для которой создаётся алиас
        phone_number
            Номер телефона (объект PhoneNumber) из которого образуется алиас
        language
            Язык который понимает пользователь учётной записи account

    Исключения
        AliasNotAllowed
        PhoneNotSecure
        InvalidState
        BlackboxInvalidResponseError
        BlackboxTemporaryError
        BlackboxUnknownError
        DBError
    """

    _UNKNOWN = object()

    class AliasNotAllowed(Exception):
        pass

    class PhoneNotSecure(Exception):
        pass

    class InvalidState(Exception):
        pass

    def __init__(self, account, phone_number, consumer, blackbox, statbox,
                 language=None, should_check_secure_phone=True,
                 phone_alias_manager_cls=None,
                 should_check_if_alias_allowed=True, enable_search=True):
        self._phone_alias_manager = self._build_phone_alias_manager(
            phone_alias_manager_cls or PhoneAliasManager,
            consumer,
            statbox,
        )

        if (should_check_if_alias_allowed and
                not self._phone_alias_manager.is_alias_allowed(account, enable_search=enable_search)):
            raise self.AliasNotAllowed()

        if should_check_secure_phone:
            self._check_secure_phone(account, phone_number)

        self._account = account
        self._phone_number = phone_number
        self._prev_owner = self._UNKNOWN
        self._language = language
        self._blackbox = blackbox
        self._statbox = statbox
        self._is_alias_given_out = False
        self._enable_search = enable_search

    def _check_secure_phone(self, account, phone_number):
        secure_phone = account.phones.secure
        if not secure_phone:
            raise self.PhoneNotSecure()
        if secure_phone.number != phone_number:
            raise self.PhoneNotSecure()

    def _build_phone_alias_manager(self, cls, consumer, statbox):
        return cls(
            consumer=consumer,
            environment=None,
            statbox=statbox,
            need_update_tx=False,
        )

    def give_out(self, validation_period=0):
        """
        Выдать алиас на данную учётную запись.
        """
        if self._prev_owner is self._UNKNOWN:
            raise self.InvalidState('Get previous owner')
        elif (self._prev_owner is not None and
              self._prev_owner.phonenumber_alias is not None and
              self._prev_owner.phonenumber_alias.number == self._phone_number):
            raise self.InvalidState('Take new alias away')

        self._check_secure_phone(self._account, self._phone_number)
        if not self._phone_alias_manager.is_alias_allowed(self._account, enable_search=self._enable_search):
            raise self.AliasNotAllowed()

        alias = self._account.phonenumber_alias
        if alias and alias.number:
            if alias.number == self._phone_number:
                return
            else:
                raise self.InvalidState('Take old alias away')

        with self._statbox.make_context(
            is_owner_changed=self._prev_owner is not None,
            number=self._phone_number.masked_format_for_statbox,
            uid=self._account.uid or '-',
            login=self._account.login,
        ):
            self._phone_alias_manager.create_alias_with_logging(
                self._account,
                self._phone_number,
                self._language,
                validation_period=validation_period,
                enable_search=self._enable_search,
            )
        self._is_alias_given_out = True

    def notify(self):
        if self._is_alias_given_out:
            if self._account.phonenumber_alias.enable_search:
                self._phone_alias_manager.send_mail_about_alias_as_login_and_email_enabled(
                    self._account,
                    self._language,
                    self._phone_number,
                )
            else:
                self._phone_alias_manager.send_mail_about_alias_as_login_enabled(
                    self._account,
                    self._language,
                    self._phone_number,
                )

    def take_old_alias_away(self):
        """
        Забирает старый алиас с учётной записи.
        """
        # В одной транзакции нельзя удалить алиас и создать новый, поэтому
        # удаление старого алиаса и создание нового разделены.
        alias = self._account.phonenumber_alias
        if alias and alias.number and alias.number != self._phone_number:
            with self._statbox.make_context(
                number=alias.number.masked_format_for_statbox,
                uid=self._account.uid,
            ):
                self._phone_alias_manager.delete_alias_with_logging(
                    self._account,
                    self._phone_alias_manager.ALIAS_DELETE_REASON_OFF,
                    self._language,
                )

    def take_away(self):
        """
        Забрать алиас у прошлого владельца.
        """
        if self._prev_owner is self._UNKNOWN:
            raise self.InvalidState('Get previous owner')
        if self._prev_owner is not None:
            with self._statbox.make_context(
                number=self._phone_number.masked_format_for_statbox,
                uid=self._prev_owner.uid,
            ):
                self._phone_alias_manager.delete_alias_with_logging(
                    self._prev_owner,
                    self._phone_alias_manager.ALIAS_DELETE_REASON_OWNER_CHANGE,
                )

    def get_owner(self):
        """
        Узнать, кто предыдущий владелец алиаса.
        """
        alias = self._account.phonenumber_alias.number
        if alias and alias == self._phone_number:
            self._prev_owner = None
            return

        all_alt_phone_numbers = [self._phone_number] + get_alt_phone_numbers_of_phone_number(self._phone_number)
        i_am_owner = None

        while (
            all_alt_phone_numbers and
            (self._prev_owner is None or self._prev_owner is self._UNKNOWN) and
            not i_am_owner
        ):
            alt_phone_number = all_alt_phone_numbers.pop(0)
            try:
                self._prev_owner = build_account(
                    self._blackbox.userinfo(
                        login=alt_phone_number.digital,
                        need_aliases=True,
                        emails=True,
                        find_by_phone_alias=BLACKBOX_FIND_BY_PHONE_ALIAS_FORCE_ON,
                    ),
                )
            except UnknownUid:
                # Телефонный алиас пока никому не назначен.
                self._prev_owner = None
            else:
                if self._prev_owner.uid == self._account.uid:
                    self._prev_owner = None
                    i_am_owner = True

        return self._prev_owner
