# -*- coding: utf-8 -*-

import contextlib
from datetime import datetime
import logging

from passport.backend.core.conf import settings
from passport.backend.core.dbmanager.exceptions import DBError
from passport.backend.core.logging_utils.loggers import DummyLogger
from passport.backend.core.models.phones.phones import MarkOperation
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.serializers.logs import log_events
from passport.backend.core.utils.decorators import cached_property
from passport.backend.core.yasms.notifications import notify_user_by_email_that_phone_unbound
from passport.backend.core.yasms.phonenumber_alias import PhoneAliasManager
from passport.backend.core.yasms.utils import get_many_accounts_with_phones_by_uids
from passport.backend.utils.common import generate_random_code


log = logging.getLogger('passport.yasms')


def unbind_old_phone(subject_phone, blackbox_builder, statbox, consumer,
                     event_timestamp, environment, threshold_timestamp=None):
    """
    Смотри документацию к UnbindOldPhone

    Исключения
        BlackboxInvalidResponseError
        BlackboxTemporaryError
        BlackboxUnknownError
    """
    unbinder = UnbindOldPhone(
        subject_phone=subject_phone,
        blackbox_builder=blackbox_builder,
        statbox=statbox,
        consumer=consumer,
        event_timestamp=event_timestamp,
        environment=environment,
        threshold_timestamp=threshold_timestamp,
    )

    accounts = unbinder.load_accounts_to_expel()

    for account in accounts:
        with unbinder.notification(account):
            unbinder.unbind_phone(account)


class UnbindOldPhone(object):
    """
    Отвязывает телефон от других учётных записей, если привязка произошла
    раньше привязки данного телефона и число привязок достигло предела.

    Входные параметры
        subject_phone
            Объект Phone на учётной записи к которой он был привязан

        blackbox_builder
            Клиент ЧЯ

        statbox
            Статбокс

        consumer
            Потребитель

        event_timestamp
            Время, которое нужно записать в событие привязывающего аккаунта
            об отвязке.

        threshold_timestamp
            Отвязать телефоны, которые были привязаны раньше данного времени.
            Если не указано, то используется время привязки subject_phone.
    """
    def __init__(
        self,
        subject_phone,
        blackbox_builder,
        statbox,
        consumer,
        event_timestamp,
        environment,
        threshold_timestamp=None,
    ):
        self._blackbox_builder = blackbox_builder
        self._consumer = consumer
        self._environment = environment
        self._event_timestamp = event_timestamp
        self._statbox = statbox
        self._subject_phone = subject_phone

        if threshold_timestamp is None:
            threshold_timestamp = self._subject_phone.bound
        self._threshold_timestamp = threshold_timestamp

        self._subject_account = subject_phone.parent.parent

    def load_accounts_to_expel(self):
        """
        Список аккаунтов, от которых нужно отвязать телефон из-за того, что он
        привязан к слишком большому числу аккаунтов.

        Ходит в ЧЯ!

        Исключения
            BlackboxInvalidResponseError
            BlackboxTemporaryError
            BlackboxUnknownError
        """
        current_bindings = self._blackbox_builder.phone_bindings(
            need_current=True,
            need_history=False,
            need_unbound=False,
            phone_numbers=[self._subject_phone.number.e164],
            should_ignore_binding_limit=False,
        )

        current_bindings_count = len(current_bindings)
        subject_in_bindings = any(b[u'uid'] == self._subject_account.uid for b in current_bindings)
        if (not subject_in_bindings and
                not self._subject_phone.binding.should_ignore_binding_limit):
            current_bindings_count += 1

        if current_bindings_count <= settings.YASMS_PHONE_BINDING_LIMIT:
            return list()
        uids_to_unbind = {
            b[u'uid'] for b in current_bindings
            if (b[u'uid'] != self._subject_account.uid and
                datetime.fromtimestamp(b[u'binding_time']) < self._threshold_timestamp)
        }

        not_filtered_accounts, _ = get_many_accounts_with_phones_by_uids(uids_to_unbind, self._blackbox_builder)

        accounts = list()
        for account in not_filtered_accounts:
            phone = account.phones.by_number(self._subject_phone.number.e164)
            if not (phone and
                    phone.bound and
                    phone.bound < self._threshold_timestamp and
                    not phone.operation):
                continue
            if account.phones.secure == phone and (account.totp_secret.is_set or account.sms_2fa_on):
                continue
            accounts.append(account)

        # Среди аккаунтов нет subject_account, поэтому прибавим единицу.
        if len(accounts) + 1 <= settings.YASMS_PHONE_BINDING_LIMIT:
            return list()

        accounts.sort(key=self._get_phone_confirmation_from_account)
        return accounts[:len(accounts) - settings.YASMS_PHONE_BINDING_LIMIT + 1]

    def unbind_phone(self, account):
        """
        Отвязывает телефон от данного аккаунта

        Записывает в логи, что причина отвязки это слишком большое число
        аккаунтов, с которыми связан телефон (номер телефона и последний
        привязавший аккаунт передаются в конструктор).

        Ничего не делает, если отказывает БД
        """
        # NOTE: В account находится учётная запись с телефоном, который нужно
        # отвязать.
        phone = account.phones.by_number(self._subject_phone.number.e164)
        try:
            with UPDATE(
                account,
                self._environment,
                {u'action': u'acquire_phone', u'consumer': self._consumer},
                initiator_uid=self._subject_account.uid,
            ):
                # Заблокируем телефон с помощю операции-флажка, чтобы не
                # случилось гонок.
                #
                # Из-за гонки можно отвязать номер с операцией или
                # начать операцию на отвязанном номере.
                #
                # Дано: привязан простой номер
                # П1: Получает телефонные данные
                # П1: Проверяет, что на номере нет операции
                # П1: Отвязывает номер
                # П2: Получает телефонные данные
                # П2: Проверяет, что номер привязан как простой
                # П2: Начинает операцию защиты
                # П1: Фиксирует изменения
                # П2: Фиксирует изменения
                mark_op = MarkOperation.create(
                    account.phones,
                    phone.id,
                    generate_random_code(settings.SMS_VALIDATION_CODE_LENGTH),
                    DummyLogger(),
                )
            with UPDATE(
                account,
                self._environment,
                {
                    u'action': u'unbind_phone_from_account',
                    u'consumer': self._consumer,
                    u'reason_uid': str(self._subject_account.uid),
                },
                initiator_uid=self._subject_account.uid,
            ):
                mark_op.apply()
                account.phones.unbound_phone(phone)
                if (account.phonenumber_alias.number and
                        account.phonenumber_alias.number == phone.number):
                    self._phone_alias_manager.delete_alias(
                        account,
                        self._phone_alias_manager.ALIAS_DELETE_REASON_OFF,
                    )

            log_events(
                {
                    self._subject_account.uid: {
                        # Укажем аккаунт из-за которого произошла отвязка.
                        #
                        # Если от аккаунта в одну секунду отвязали 2 номера
                        # из-за привязки этих номеров к 2-ум другим аккаунтам,
                        # то произойдут 2 события с именем
                        # "unbind_phone_from_account" и для того чтобы они
                        # не склеились к ним добавляется суффикс ".uid".
                        #
                        # Увы, но это не решает проблему, когда от аккаунта
                        # отвязывается 2 номера из-за их привязки к одному
                        # аккаунту.
                        u'unbind_phone_from_account.%d' % account.uid: self._subject_phone.number.e164,
                        u'consumer': self._consumer,
                    },
                },
                self._environment.user_ip,
                self._environment.user_agent,
                self._environment.cookies.get('yandexuid'),
                datetime_=self._event_timestamp,
                initiator_uid=self._subject_account.uid,
            )

        except DBError:
            # Действия по отвязке необязательны поэтому, чтобы не терять
            # возможность отвязать номер от других учётных записей, игнорируем
            # ошибку базы данных.
            log.warning(
                u'Database error occured while phone with id=%s was being '
                u'unbounded on account with uid=%s' % (phone.id, account.uid),
            )
        else:
            self._statbox.dump_stashes()
            self._statbox.log(
                action=self._statbox.Link(u'phone_unbound'),
                uid=account.uid,
                number=phone.number.masked_format_for_statbox,
                phone_id=phone.id,
            )

    @contextlib.contextmanager
    def notification(self, account):
        """
        Высылает уведомительное письмо о произошедшей отвязке телефона (но
        только, если она произошла)
        """
        phone = account.phones.by_number(self._subject_phone.number)
        is_bound = phone and phone.bound
        is_secure = phone and phone == account.phones.secure

        yield

        if is_bound:
            phone = account.phones.by_number(self._subject_phone.number)
            if not (phone and phone.bound):
                notify_user_by_email_that_phone_unbound(
                    account,
                    self._subject_phone.number,
                    is_secure,
                )

    def _find_accounts_with_earliest_confirmation(self, accounts, count):
        """
        Возвращает список, в котором находится не более count аккаунтов, с самой
        старой датой подтверждения телефона.
        """
        accounts.sort(key=lambda a: a.phones.by_number(self._subject_phone.number.e164).confirmed)
        return accounts[:count]

    @cached_property
    def _phone_alias_manager(self):
        return PhoneAliasManager(
            self._consumer,
            environment=self._environment,
            statbox=self._statbox,
            need_update_tx=False,
        )

    def _get_phone_confirmation_from_account(self, account):
        phone = account.phones.by_number(self._subject_phone.number.e164)
        return (phone.confirmed, phone.id)
