# -*- coding: utf-8 -*-
"""
Классы запросов. На данный момент есть существенное отличие от алхимии:
методы запросов (where, values, join, итд.), возвращают объект запроса только
для method chaining синтаксиса, но это всегда один и тот же объект,
клонирование не производится, переиспользовать "неполные" объекты не следует.
"""

from collections import (
    defaultdict,
    OrderedDict,
)
import re

from passport.backend.core.ydb.declarative.elements import (
    and_,
    DescOrderBy,
    Expression,
    FromElement,
    Label,
    ParametrizableValue,
    Table,
)
from passport.backend.core.ydb.declarative.errors import ProgrammingError
from passport.backend.core.ydb.ydb import YdbQuery
import six


class RenderContext(object):
    def __init__(
        self,
        namespace,
        get_table_alias,
        alias_to_selectable,
        version=None,
    ):
        self.parameter_declarations = []
        self.parameters = {}
        self.namespace = namespace
        self.names_counter = defaultdict(lambda: 0)
        self.get_table_alias = get_table_alias
        if alias_to_selectable is None:
            self.alias_to_selectable = None
        else:
            self.alias_to_selectable = dict(alias_to_selectable)
        self.version = version

    def bind_parameter(self, base, value, param_type):
        if self.namespace is None:
            param_name = '$%s' % base
        else:
            param_name = '$%s_%s' % (self.namespace, base)
        if self.names_counter[base] > 0:
            param_name = '%s_%s' % (param_name, self.names_counter[base])
        self.names_counter[base] += 1
        self.parameter_declarations.append((param_name, param_type))
        self.parameters[param_name] = value
        return param_name


class CompiledQuery(YdbQuery):
    def __init__(self, sql, context, has_result):
        super(CompiledQuery, self).__init__(sql, context)
        self._sql = sql
        self._context = context
        self.has_result = has_result

    def parse_query_result(self, result):
        new_result = {}
        for alias, expression_obj in self._context.alias_to_selectable.items():
            new_result[alias] = expression_obj.to_pyval(result[alias])
        return new_result

    def get_raw_statement(self):
        return self._sql

    def get_parameters(self):
        return self._context.parameters


class Query(object):
    RE_COLUMN = re.compile('^[a-zA-Z_][a-zA-Z0-9_]*$')
    v1 = False

    @staticmethod
    def _render_declarations(context):
        return '\n'.join(
            'DECLARE %s AS %s;' % (name, param_type)
            for name, param_type in context.parameter_declarations
        )

    @staticmethod
    def _render_v1_prefix(context):
        return '--!syntax_v1\n' if context.version == 1 else ''

    def _protect_col_name(self, value):
        if not self.RE_COLUMN.match(value):
            raise ValueError('Invalid column name %s' % value)
        return value

    def _prerender(self):
        raise NotImplementedError()

    def compile(self):
        """
        Компилировать запрос

        :return: объект скомпилированного запроса
        :rtype: CompiledQuery
        """
        sql_body, context = self._prerender()
        sql = '%s%s\n%s' % (
            self._render_v1_prefix(context),
            self._render_declarations(context),
            sql_body,
        )
        return CompiledQuery(
            sql=sql,
            context=context,
            has_result=False,
        )


# noinspection PyProtectedMember
class Select(Query):
    def __init__(self, columns, optimizer_index=None):
        self._selectables = []
        self._selectable_aliases = []
        self._alias_to_selectable = {}
        self._tables = []
        self._joins = {}
        self._table_to_alias = {}
        self._column_to_table_alias = {}
        self._alias_counters = defaultdict(lambda: defaultdict(lambda: 0))

        self._order_by_clause = None
        self._group_by_clause = None
        self._where_clause = None

        # Check if columns are iterable
        try:
            (x for x in columns)
        except TypeError:
            columns = [columns]

        for element in columns:
            self._process_column_argument(element)

        self._optimizer_index = optimizer_index

    def _process_column_argument(self, argument):
        """
        Обработать аргумент-колонку селекта, он может быть таблицей
        или выражением, содержащим или не содержащим табличную колонку
        """
        if isinstance(argument, Label):
            # На аргументе своя метка (алиас), снять её
            label = argument.name
            argument = argument.element
        else:
            label = None

        if isinstance(argument, FromElement):
            if label is not None:
                raise TypeError('Cannot label table or table-like elements')
            table_alias = self._add_table_get_alias(argument)
            # В таблице все элементы - это табличные колонки, добавить их в
            # селект и связать с табличным алиасом
            for column in argument._extract_columns():
                self._add_selectable(column, None)
                self._bind_column_to_table(column, table_alias)
        elif isinstance(argument, Expression):
            # Выражение само по себе добавляется в селект, а также из него
            # достаются рекурсивно все колонки для связки с табличными алиасами
            self._add_selectable(argument, label)
            self._process_extracted_columns(argument)
        else:
            raise TypeError('Cannot use %r as from clause' % argument)

    def _add_table_get_alias(self, table_object):
        """ Добавить таблицу в FROM и/или получить её алиас """
        try:
            alias = self._table_to_alias[table_object.objid]
        except KeyError:
            self._tables.append(table_object)
            alias = self._generate_alias('table', table_object.name)
            self._table_to_alias[table_object.objid] = alias

        return alias

    def _add_selectable(self, selectable, alias):
        """ Добавить selectable в SELECT """
        if alias is None:
            alias = self._generate_alias('selectable', selectable.name)
        self._selectables.append(selectable)
        self._selectable_aliases.append(alias)
        if alias in self._alias_to_selectable:
            raise ProgrammingError(
                'Trying to reuse alias %s (aleady used for %s) '
                % (alias, self._alias_to_selectable[alias]),
            )
        self._alias_to_selectable[alias] = selectable

    def _process_extracted_columns(self, element):
        """
        Найти в элементе колонки, добавить их таблицу в FROM или просто
        найти алиас, связать колонки с алиасом
        """
        for column in element._extract_columns():
            table_alias = self._add_table_get_alias(column.table)
            self._bind_column_to_table(column, table_alias)

    def _bind_column_to_table(self, column, table_alias):
        """ Связать колонку с алиасом её таблицы """
        self._column_to_table_alias[column.objid] = table_alias

    def _generate_alias(self, alias_type, base):
        """
        Сгенерировать алиас. Первый алиас данного типа - <base>, а дальше
        <base>_1, <base>_2, ...
        """
        counter_pos = self._alias_counters[alias_type][base]
        self._alias_counters[alias_type][base] += 1
        if counter_pos > 0:
            return '%s_%s' % (base, counter_pos)
        else:
            return base

    def _get_table_alias(self, table):
        try:
            return self._table_to_alias[table.objid]
        except KeyError:
            raise KeyError('Cannot find alias for table %s' % table)

    def _process_table(self, table_obj, context):
        if not isinstance(table_obj, FromElement):
            raise ProgrammingError('Cannot use %r as table' % table_obj)
        rendered = table_obj._render(context, True)
        if self._optimizer_index:
            rendered = '%s view %s' % (rendered, self._optimizer_index)
        alias = self._get_table_alias(table_obj)
        return '%s AS %s' % (rendered, alias)

    @staticmethod
    def _process_join(context, rendered_table, clause, join_type):
        return '%sJOIN %s ON %s' % (
            '%s ' % join_type.upper() if join_type is not None else '',
            rendered_table,
            clause._render(context, True),
        )

    def _render_tables(self, context):
        from_l = []
        join_l = []
        for table_obj in self._tables:
            rendered = self._process_table(table_obj, context)
            if table_obj.objid in self._joins:
                join_l.append(
                    self._process_join(
                        context,
                        rendered,
                        *self._joins[table_obj.objid]
                    ),
                )
            else:
                from_l.append(rendered)
        if not from_l:
            raise ProgrammingError('No FROM tables in select')

        return ', '.join(from_l), '\n'.join(join_l)

    def _render_selectables(self, context):
        if not self._selectables:
            raise ProgrammingError('No columns in select')
        elements_and_aliases = zip(self._selectables, self._selectable_aliases)
        return ', '.join(
            '%s AS %s' % (selectable._render(context, True), alias)
            for selectable, alias in elements_and_aliases
        )

    def _render_where(self, context):
        if self._where_clause is None:
            return None
        return self._where_clause._render(context, True)

    @staticmethod
    def _render_expr_list(expressions, context):
        if expressions is None:
            return None
        return ', '.join(
            expression._render(context, True)
            for expression in expressions
        )

    def _prerender(self, namespace='q'):
        context = RenderContext(
            namespace,
            self._get_table_alias,
            self._alias_to_selectable,
            version=1 if self._optimizer_index else None,
        )
        froms_s, joins_s = self._render_tables(context)
        columns_s = self._render_selectables(context)
        where_s = self._render_where(context)
        order_by_s = self._render_expr_list(self._order_by_clause, context)
        group_by_s = self._render_expr_list(self._group_by_clause, context)

        sql = 'SELECT %s\nFROM %s' % (
            columns_s,
            froms_s,
        )
        if joins_s:
            sql = '%s\n%s' % (sql, joins_s)
        if where_s:
            sql = '%s\nWHERE %s' % (sql, where_s)
        if order_by_s:
            sql = '%s\nORDER BY %s' % (sql, order_by_s)
        if group_by_s:
            sql = '%s\nGROUP BY %s' % (sql, group_by_s)

        return sql, context

    def _parse_list_clause(self, clause):
        if not clause:
            return None
        try:
            (x for x in clause)
        except TypeError:
            clause = [clause]

        for expression in clause:
            if not isinstance(expression, (Expression, DescOrderBy)):
                raise TypeError(
                    'Cannot use %r as order by clause' % clause,
                )
            self._process_extracted_columns(expression)
        return clause

    def order_by(self, clause):
        """
        Добавить сортировку ORDER BY.
        Передача новых аргументов добавляет выражение для сортировки.
        Передача None сбрасывает выражения

        например::
            query.order_by(table.c.column1)
            query.order_by(table.c.column2.desc())
        будет аналогично::
            query.order_by([table.c.column1, table.c.column2.desc()])

        :param clause: выражение или список выражений для сортировки
        :return: объект запроса
        :rtype: Select
        """
        clause = self._parse_list_clause(clause)
        if clause is None:
            self._order_by_clause = None
        else:
            if self._order_by_clause is None:
                self._order_by_clause = clause
            else:
                self._order_by_clause += clause

        return self

    def group_by(self, clause):
        """
        Добавить группировку ORDER BY.
        аргументы аналогичны order_by

        :param clause: выражение или список выражений для группировки
        :return: объект запроса
        :rtype: Select
        """
        clause = self._parse_list_clause(clause)
        if clause is None:
            self._group_by_clause = None
        else:
            if self._group_by_clause is None:
                self._group_by_clause = clause
            else:
                self._group_by_clause += clause

        return self

    def where(self, clause):
        """
        Добавить фильтр WHERE
        Передача новых аргументов добавит их в запрос через AND

        например:
            query.where(table.c.col1 == 1)
            query.where(table.c.col2 == 2)
        аналогично:
            query.where(and_(table.c.col1 == 1, table.c.col2 == 2))

        :param clause: выражение для фильтрации
        :return: объект запроса
        """
        self._process_extracted_columns(clause)
        if self._where_clause is None:
            self._where_clause = clause
        else:
            self._where_clause = and_(self._where_clause, clause)
        return self

    def join(self, table, clause, join_type=None):
        """
        Добавить таблицу в join

        например:
            query.join(table2, table2.col1 == table1.col2, 'left outer')

        :param table: табличный объект
        :param clause: условие джойна
        :param join_type: тип джойна, строка 'left', 'left outer' итд. Если
            не указана, используется тип по умолчанию из БД
        :return: объект запроса
        """
        if not isinstance(table, FromElement):
            raise TypeError('Cannot use %r as join table' % table)
        if not isinstance(clause, Expression):
            raise TypeError('Cannot use %r as join clause' % clause)

        self._add_table_get_alias(table)
        self._joins[table.objid] = (clause, join_type)

        return self


def select(columns, where_clause=None, order_by=None, optimizer_index=None):
    """
    SELECT

    :param columns: колонки, iterable колонок или одна колонка
    :param where_clause: условие WHERE, одно выражение. Если нужно много -
        совмещать через and_()
    :param order_by: выражение для сортировки, одно или iterable
    :param optimizer_index: имя вторичного индекса, используемого в WHERE
    :return: объект Select
    """
    query = Select(columns=columns, optimizer_index=optimizer_index)
    if where_clause is not None:
        query = query.where(where_clause)
    if order_by is not None:
        query = query.order_by(order_by)

    return query


# noinspection PyProtectedMember
class Insert(Query):
    def __init__(self, table, values=None, insert_type='INSERT'):
        if not isinstance(table, Table):
            raise TypeError('Cannot insert into %r' % table)
        self._table = table

        self._columns = None
        self._values = None

        if insert_type.upper() not in ('INSERT', 'UPSERT', 'REPLACE'):
            raise ValueError('Unsupported insert type %s' % insert_type)
        self._insert_type = insert_type.upper()
        if values:
            self._add_values(values)

    def _add_values(self, *args, **kwargs):
        if args and kwargs:
            raise ValueError('Cannot set values by args and kwargs at once')

        if kwargs:
            args = [kwargs]

        self._values = []
        for arg in args:
            if not isinstance(arg, (dict, OrderedDict)):
                raise TypeError('Cannot use %r as values' % arg)
            if self._columns is None:
                self._columns = list(map(self._protect_col_name, arg.keys()))
                if six.PY2 and not isinstance(arg, OrderedDict):
                    self._columns = sorted(self._columns)
            try:
                self._values.append([arg[column] for column in self._columns])
            except KeyError as err:
                raise ValueError('%s. Values dicts must have same keys' % err)

    def values(self, *args, **kwargs):
        """
        Добавить значения.

        Есть три варианта передачи аргументов:
        - kwargs:
            values(col1=value1, col2=value2)
        - args dict(s):
            values({"col1": value1, "col2": value2})
            values(
                {"col1": value1_1, "col2": value2_1},
                {"col1": value1_2, "col2": value2_2},
            )

        :return: объект запроса
        """
        self._add_values(*args, **kwargs)
        return self

    def _process_insert_value_block(self, context, block):
        return '(%s)' % ', '.join(
            self._table.column_dict[column]._wrap_opposite_value(
                ParametrizableValue(
                    column,
                    *self._table.column_dict[column].from_pyval(value)
                )._render(context)
            )
            for column, value in zip(self._columns, block)
        )

    def _process_insert_values(self, context):
        if not self._values:
            raise ProgrammingError('Insert query has no values')

        return 'VALUES\n%s' % ',\n'.join(
            self._process_insert_value_block(context, value_block)
            for value_block in self._values
        )

    def _prerender(self):
        context = RenderContext(None, None, None)
        sql = '%s INTO %s\n(%s)\n%s' % (
            self._insert_type,
            self._table.name,
            ', '.join(self._columns),
            self._process_insert_values(context),
        )

        return sql, context


def insert(table, values=None, insert_type='INSERT'):
    """
    INSERT

    :param table: таблица
    :param values: значения, См. документацию метода Insert.values()
    :param insert_type: 'INSERT', 'UPSERT', 'REPLACE', см. документацию YQL
    :return: объект Insert
    """
    return Insert(table=table, values=values, insert_type=insert_type)


# noinspection PyProtectedMember
class Delete(Query):
    def __init__(self, table, where=None, optimizer_index=None):
        if not isinstance(table, Table):
            raise TypeError('Cannot delete from %r' % table)
        self._table = table
        self._where_clause = None

        if optimizer_index and self._table.first_primary_key is None:
            raise ValueError(
                'Cannot use table without primary keys '
                'with optimizer_index argument in delete',
            )
        self._optimizer_index = optimizer_index

        if where is not None:
            self._add_where(where)

    def _add_where(self, clause):
        if clause is not None:
            if not isinstance(clause, Expression):
                raise TypeError('Cannot use %r as delete where' % clause)
        if self._where_clause is None:
            self._where_clause = clause
        else:
            self._where_clause = and_(self._where_clause, clause)
        return self

    def where(self, clause):
        """
        Добавить фильтр WHERE
        Передача новых аргументов добавит их в запрос через AND

        например:
            query.where(table.c.col1 == 1)
            query.where(table.c.col2 == 2)
        аналогично:
            query.where(and_(table.c.col1 == 1, table.c.col2 == 2))

        :param clause: выражение для фильтрации
        :return: объект запроса
        """
        self._add_where(clause)
        return self

    def _get_table_alias(self, table):
        if table.objid != self._table.objid:
            raise ProgrammingError('Table %s is unknown in delete' % table)
        return None

    def _render_where(self, context):
        if self._where_clause is None:
            return None
        return self._where_clause._render(context, True)

    def _prerender(self):
        version = 1 if self._optimizer_index else None
        context = RenderContext(None, self._get_table_alias, None, version)

        sql = 'DELETE FROM %s' % self._table.name

        if self._optimizer_index:
            sql = '%s ON\nSELECT %s FROM %s view %s' % (
                sql,
                self._table.first_primary_key_name,
                self._table.name,
                self._optimizer_index,
            )

        where_s = self._render_where(context)
        if where_s:
            sql = '%s\nWHERE %s' % (sql, where_s)

        return sql, context


def delete(table, where=None, optimizer_index=None):
    """
    DELETE

    :param table: таблица
    :param where: условие where, см. документацию Delete.where
    :param optimizer_index: имя вторичного индекса, используемого в WHERE
    :return: объект Delete
    """
    return Delete(table=table, where=where, optimizer_index=optimizer_index)
