# -*- coding: utf-8 -*-

import datetime
from functools import partial
import json
import time

from passport.backend.core.ydb.declarative.errors import SchemaDeclarationError
from passport.backend.utils.time import unixtime_to_datetime
import six


class DataType(object):
    _name = 'Unknown'

    @property
    def name(self):
        return self._name

    def get_type_annotation(self):
        return self.name

    def to_pyval(self, value):
        return value

    def from_pyval(self, value):
        return value

    def _wrap_opposite_value(self, value):
        return value


class Bool(DataType):
    _name = 'Bool'

    def from_pyval(self, value):
        return bool(value)

    def to_pyval(self, value):
        return bool(value)


class Integer(DataType):
    _sizes = (8, 16, 32, 64)

    def __init__(self, size=64, signed=True):
        if size not in self._sizes:
            raise SchemaDeclarationError(
                'Wrong integer size %s, %s supported' %
                (size, self._sizes),
            )
        self.size = size
        self.signed = signed

    @property
    def name(self):
        return '%s%s' % ('Int' if self.signed else 'Uint', self.size)

    def from_pyval(self, value):
        if isinstance(value, six.integer_types):
            return value
        else:
            return int(value)


Int8 = partial(Integer, 8, True)
Int16 = partial(Integer, 16, True)
Int32 = partial(Integer, 32, True)
Int64 = partial(Integer, 64, True)
Uint8 = partial(Integer, 8, False)
Uint16 = partial(Integer, 16, False)
Uint32 = partial(Integer, 32, False)
Uint64 = partial(Integer, 64, False)


class Float(DataType):
    def __init__(self, double=False):
        self.double = double

    def from_pyval(self, value):
        return float(value)

    @property
    def name(self):
        return 'Double' if self.double else 'Float'


Double = partial(Float, double=True)


class String(DataType):
    _name = 'String'

    def from_pyval(self, value):
        if isinstance(value, six.binary_type):
            return value
        elif isinstance(value, six.text_type):
            return value.encode('utf8')
        else:
            return six.text_type(value).encode()


class Utf8(DataType):
    _name = 'Utf8'

    def from_pyval(self, value):
        if isinstance(value, six.text_type):
            return value
        elif isinstance(value, six.binary_type):
            return value.decode('utf8')
        else:
            return six.text_type(value)


class DateTime(DataType):
    _name = 'Datetime'

    def from_pyval(self, value):
        if isinstance(value, datetime.datetime):
            return int(time.mktime(value.timetuple()))
        elif isinstance(value, six.integer_types):
            return value
        elif isinstance(value, float):
            return int(value)
        else:
            raise TypeError('Datetime value must be number or datetime')

    def to_pyval(self, value):
        return unixtime_to_datetime(value)


class Date(DataType):
    _name = 'Date'
    _epoch = unixtime_to_datetime(0).date()

    def from_pyval(self, value):
        if isinstance(value, datetime.date):
            if isinstance(value, datetime.datetime):
                value = value.date()
            return (value - self._epoch).days
        elif isinstance(value, int):
            return value
        else:
            raise TypeError('Date value must be date or int')

    def to_pyval(self, value):
        return self._epoch + datetime.timedelta(days=value)


class Timestamp(DataType):
    _name = 'Timestamp'

    def __init__(self, as_float=True):
        self.as_float = as_float

    def from_pyval(self, value):
        if isinstance(value, int):
            return value*1000000
        elif isinstance(value, float):
            return int(value*1000000)
        elif isinstance(value, datetime.datetime):
            return int(time.mktime(value.timetuple())*1000000)
        else:
            raise TypeError('Timestamp value must be number or datetime')

    def to_pyval(self, value):
        value = value/1000000

        if self.as_float:
            return value
        else:
            return unixtime_to_datetime(value)


class Json(DataType):
    _name = 'Json'

    def from_pyval(self, value):
        if isinstance(value, six.string_types):
            return value
        else:
            return json.dumps(value)

    def to_pyval(self, value):
        return json.loads(value)


class PyTypeMap(object):
    default = Utf8()

    def __init__(self):
        self.map = {}

    def add(self, py_types, data_type):
        for py_type in py_types:
            self.map[py_type] = data_type


PY_TYPE_MAP = PyTypeMap()
PY_TYPE_MAP.add((bool,), Bool())
PY_TYPE_MAP.add((six.text_type,), Utf8())
PY_TYPE_MAP.add((six.binary_type,), String())
PY_TYPE_MAP.add(six.integer_types, Int64())
PY_TYPE_MAP.add((float,), Double())
PY_TYPE_MAP.add((datetime.date,), Date())
PY_TYPE_MAP.add((datetime.datetime,), DateTime())
PY_TYPE_MAP.add((dict, list), Json())


def pytype_to_data_type(pytype):
    try:
        return PY_TYPE_MAP.map[pytype]
    except KeyError:
        return PY_TYPE_MAP.default


def init_type(data_type):
    if isinstance(data_type, DataType):
        return data_type
    elif isinstance(data_type, type) and issubclass(data_type, DataType):
        return data_type()
    elif isinstance(data_type, partial) and issubclass(data_type.func, DataType):
        return data_type()
    else:
        raise TypeError('Data type must be DataType class or instance')
