# -*- coding: utf-8 -*-
"""
Family invites module

Main table DDL:
--!syntax_v1
CREATE TABLE family_invites (
    invite_id String,
    family_id Uint64,
    issuer_uid Uint64,
    send_method Uint8,
    create_time Timestamp,
    contact Utf8,
    PRIMARY KEY (invite_id),
    INDEX family_id_index GLOBAL ON (family_id),
    INDEX create_time_index GLOBAL ON (create_time)
);
"""


from passport.backend.core.exceptions import BaseCoreError
from passport.backend.core.models.family import FamilyInvite
from passport.backend.core.ydb.declarative import (
    delete,
    insert,
    select,
)
from passport.backend.core.ydb.exceptions import (
    YdbNoResultFound,
    YdbPreconditionError,
)
from passport.backend.core.ydb.schemas import family_invites_table as fit
from passport.backend.core.ydb.ydb import (
    get_ydb_family_invite,
    YdbQueryExecutor,
)


def get_executor():
    return YdbQueryExecutor(get_ydb_family_invite())


def insert_family_invite(invite):
    query = insert(fit, dict(
        invite_id=invite.invite_id,
        family_id=invite.family_id,
        issuer_uid=invite.issuer_uid,
        send_method=invite.send_method,
        create_time=invite.create_time,
        contact=invite.contact,
    ))
    query_c = query.compile()
    ydb_executor = get_executor()
    try:
        ydb_executor.execute_queries([query_c])
    except YdbPreconditionError:
        raise FamilyInviteExistsError()


def find_family_invite(invite_id):
    query = select(fit, fit.c.invite_id == invite_id)
    ydb_executor = get_executor()
    query_c = query.compile()
    result = ydb_executor.execute_queries([query_c])[0]
    try:
        row = result.one()
    except YdbNoResultFound:
        return None

    return FamilyInvite().parse(row)


def find_invites_for_family(family_id, exclude_invite=None):
    query = select(
        fit,
        fit.c.family_id == family_id,
        optimizer_index='family_id_index',
    )
    if exclude_invite is not None:
        query = query.where(fit.c.invite_id != exclude_invite)
    ydb_executor = get_executor()
    query_c = query.compile()
    result = ydb_executor.execute_queries([query_c])[0]
    # TODO можно сделать здесь ленивый парсер вместо list'a
    return [FamilyInvite().parse(row) for row in result]


def delete_family_invite(invite_id):
    query = delete(
        fit,
        fit.c.invite_id == invite_id,
    )
    query_c = query.compile()
    ydb_executor = get_executor()
    ydb_executor.execute_queries([query_c])


def delete_family_invites_until(timestamp):
    query = delete(
        fit,
        fit.c.create_time <= timestamp,
        optimizer_index='create_time_index',
    )
    query_c = query.compile()
    ydb_executor = get_executor()
    ydb_executor.execute_queries([query_c])


def delete_family_invites_by_family_id(family_id):
    query = delete(
        fit,
        fit.c.family_id == family_id,
        optimizer_index='family_id_index',
    )
    query_c = query.compile()
    ydb_executor = get_executor()
    ydb_executor.execute_queries([query_c])


class FamilyInviteExistsError(BaseCoreError):
    pass
