# -*- coding: utf-8 -*-

import logging
import textwrap

from passport.backend.core.crypto.signing import get_yandex_and_yandex_team_signing_registry
from passport.backend.core.models.support_code import SupportCode
from passport.backend.core.serializers.ydb.exceptions import DataCorruptedYdbSerializerError
from passport.backend.core.serializers.ydb.public import (
    from_ydb_rows,
    to_ydb_rows,
)
from passport.backend.core.serializers.ydb.support_code import (
    hash_support_code_value,
    SupportCodeSerializerConfiguration,
)
from passport.backend.core.ydb.exceptions import YdbPreconditionError
from passport.backend.core.ydb.ydb import (
    get_ydb_support_code,
    YdbOnlineReadOnlyTxMode,
    YdbQuery,
    YdbQueryExecutor,
)
from passport.backend.utils.string import smart_bytes
from passport.backend.utils.time import datetime_to_integer_unixtime


log = logging.getLogger(__name__)


def insert_support_code(support_code):
    signing_registry, cur_secret, old_secret = get_signing_things()
    serializer_config = SupportCodeSerializerConfiguration(
        signing_registry=signing_registry,
        cur_secret=cur_secret,
        old_secret=old_secret,
    )
    row1, row2 = to_ydb_rows(support_code, serializer_config)

    ydb = YdbQueryExecutor(get_ydb_support_code())

    query = InsertSupportCodeYdbQuery(row1, row2)
    try:
        ydb.execute_queries(
            [query],
        )
    except YdbPreconditionError:
        raise SupportCodeExistsError()


def find_support_code(code_value):
    signing_registry, cur_secret, old_secret = get_signing_things()

    old_hash = hash_support_code_value(code_value, old_secret)
    cur_hash = hash_support_code_value(code_value, cur_secret)
    query = FindSupportCodeYdbQuery(cur_hash, old_hash)

    ydb = YdbQueryExecutor(get_ydb_support_code())
    result_sets = ydb.execute_queries(
        [query],
        tx_mode=YdbOnlineReadOnlyTxMode(),
    )

    serializer_config = SupportCodeSerializerConfiguration(
        signing_registry=signing_registry,
        cur_secret=cur_secret,
        old_secret=old_secret,
    )
    try:
        support_code = from_ydb_rows(SupportCode, result_sets[0], serializer_config)
    except DataCorruptedYdbSerializerError as e:
        log.debug(str(e))
        return
    if support_code:
        support_code.value = code_value
    return support_code


def delete_support_codes_expired_before_timestamp(timestamp):
    timestamp = datetime_to_integer_unixtime(timestamp)
    query = DeleteSupportCodesExpiredBeforeTimestampYdbQuery(timestamp)
    ydb = YdbQueryExecutor(get_ydb_support_code())
    ydb.execute_queries([query])


def get_signing_things():
    signing_registry = get_yandex_and_yandex_team_signing_registry()
    cur_secret = signing_registry.get()
    old_secret_id = signing_registry.get_prev_version_id(smart_bytes(cur_secret.id))
    old_secret = signing_registry.get(smart_bytes(old_secret_id))
    return signing_registry, cur_secret, old_secret


class InsertSupportCodeYdbQuery(YdbQuery):
    def __init__(self, row1, row2):
        raw_statement = textwrap.dedent("""
            DECLARE $code_hash1 AS Utf8;
            DECLARE $code_hash2 AS Utf8;
            DECLARE $value1 AS Utf8;
            DECLARE $value2 AS Utf8;
            DECLARE $expires_at1 AS Datetime;
            DECLARE $expires_at2 AS Datetime;

            INSERT INTO support_codes(code_hash, expires_at, value)
            VALUES
                ($code_hash1, $expires_at1, $value1),
                ($code_hash2, $expires_at2, $value2);
        """)
        parameters = {
            '$code_hash1': row1['code_hash'],
            '$code_hash2': row2['code_hash'],
            '$value1': row1['value'],
            '$value2': row2['value'],
            '$expires_at1': row1['expires_at'],
            '$expires_at2': row2['expires_at'],
        }
        super(InsertSupportCodeYdbQuery, self).__init__(
            raw_statement=raw_statement,
            parameters=parameters,
        )


class FindSupportCodeYdbQuery(YdbQuery):

    def __init__(self, code_hash1, code_hash2):
        raw_statement = textwrap.dedent("""
            DECLARE $code_hash1 AS Utf8;
            DECLARE $code_hash2 AS Utf8;

            SELECT * FROM support_codes
            WHERE code_hash = $code_hash1 OR code_hash = $code_hash2;
        """)
        parameters = {
            '$code_hash1': code_hash1,
            '$code_hash2': code_hash2,
        }

        super(FindSupportCodeYdbQuery, self).__init__(
            raw_statement=raw_statement,
            parameters=parameters,
        )


class DeleteSupportCodesExpiredBeforeTimestampYdbQuery(YdbQuery):
    """
    Удаляет support_code из БД, которые протухают раньше чем данный timestamp.
    """

    def __init__(self, timestamp):
        raw_statement = textwrap.dedent("""
            DECLARE $timestamp AS Datetime;

            DELETE FROM support_codes WHERE expires_at <= $timestamp;
        """)
        parameters = {'$timestamp': timestamp}
        super(DeleteSupportCodesExpiredBeforeTimestampYdbQuery, self).__init__(
            raw_statement=raw_statement,
            parameters=parameters,
        )


class SupportCodeError(Exception):
    pass


class SupportCodeExistsError(SupportCodeError):
    pass
