# -*- coding: utf-8 -*-

from __future__ import (
    absolute_import,
    unicode_literals,
)

from datetime import datetime
import logging

from passport.backend.core.builders.blackbox.blackbox import Blackbox
from passport.backend.core.builders.blackbox.utils import add_phone_arguments
from passport.backend.core.builders.social_api import SocialApi
from passport.backend.core.builders.social_api.exceptions import (
    SocialApiRequestError,
    SocialApiTemporaryError,
)
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers.statbox import (
    to_statbox,
    AccountModificationInfosecLogger,
)
from passport.backend.core.models.account import ACCOUNT_DISABLED_ON_DELETION
from passport.backend.core.models.delete_tasks import PhoneBindingsHistoryDeleteTask
from passport.backend.core.models.family import FamilyInfo
from passport.backend.core.runner.context_managers import (
    CREATE,
    DELETE,
    UPDATE,
)
from passport.backend.core.services import (
    REAL_SERVICES,
    Service,
)
from passport.backend.core.subscription import (
    can_be_unsubscribed,
    delete_subscription,
    UnsubscribeBlockingServiceError,
    UnsubscribeProtectedServiceError,
    user_has_contract_with_yandex,
)
from passport.backend.core.utils.blackbox import get_many_accounts_by_uids
from passport.backend.core.ydb.processors.family_invite import delete_family_invites_by_family_id
from passport.backend.dbscripts import context


log = logging.getLogger('passport.backend.dbscripts.account_deleter')


def delete_accounts(uids, environment):
    userinfo_args = {
        'emails': True,
        'email_attributes': 'all',
        'get_family_info': True,
        'force_show_mail_subscription': True,
    }
    userinfo_args = add_phone_arguments(**userinfo_args)

    accounts, removed_uids = get_many_accounts_by_uids(uids, Blackbox(), userinfo_args)
    log.debug('Removed accounts: %r', removed_uids)
    retval = [{'uid': uid, 'status': 'not_found'} for uid in removed_uids]
    social_api = SocialApi()

    for account in accounts:
        with context.set({'uid': account.uid}):
            try:
                is_deleted = delete_account(account, environment, social_api)
            except Exception:
                # Любой отказ на аккаунте не должен быть причиной пропуска других
                # аккаунтов.
                log.error('Error occurred while processing account (uid=%d)' % account.uid, exc_info=True)
                retval.append({'uid': account.uid, 'status': 'error'})
            else:
                log.debug('Account %d is processed', account.uid)
                status = 'deleted' if is_deleted else 'not_deleted'
                retval.append({'uid': account.uid, 'status': status})

    return retval


def delete_account(account, environment, social_api):
    return _AccountDeleter(account, environment, social_api).delete()


class _AccountDeleter(object):
    def __init__(self, account, environment, social_api):
        self._account = account
        self._env = environment
        self._social_api = social_api
        self._blackbox = Blackbox()

        self._family_deleter = _FamilyDeleter(self._blackbox, self._env)
        self._family_deleter.account = self._account

    def delete(self):
        """
        Удаляет всё что можно удалить про аккаунт.

        В самом благоприятном случае аккаунт удаляется полностью и программа
        возвращает True.
        """
        if self._account.disabled_status != ACCOUNT_DISABLED_ON_DELETION:
            log.debug('account.disabled_status = %s' % self._account.disabled_status)
            if self._account.deletion_operation:
                with UPDATE(self._account, self._env, {'action': 'remove_deletion_operation'}):
                    self._account.deletion_operation = None
            return
        elif not self._account.deletion_operation:
            log.debug('Deletion operation not found')
            return

        mail_sid = Service.by_slug('mail').sid
        if self._account.has_sid(mail_sid):
            self._unsubscribe_from_not_blocking_sids(sids=[mail_sid])

        if self._account.has_family:
            self._family_deleter.load_family()
            if self._account.uid == self._account.family_info.admin_uid:
                self._family_deleter.delete_kids()
            self._family_deleter.delete_family_records()

        self._delete_apple_tokens()

        if not self._is_quarantine_finished():
            log.debug('It is too early for deletion (started at %s)' % self._account.deletion_operation.started_at)
            return

        self._unsubscribe_from_not_blocking_sids()

        if (
            self._is_personal_data_must_be_deleted() and
            not user_has_contract_with_yandex(self._account)
        ):
            self._delete_personal_data()

        if user_has_contract_with_yandex(self._account):
            return

        try:
            self._delete_all_data()
        except (SocialApiTemporaryError, SocialApiRequestError):
            log.warning('Failed to delete account because of Socialism failed')
            return

        self._log_deletion()
        return True

    def _is_quarantine_finished(self):
        return datetime.now() - self._account.deletion_operation.started_at > settings.ACCOUNT_DELETION_QUARANTINE_PERIOD

    def _unsubscribe_from_not_blocking_sids(self, sids=None):
        REAL_SIDS = sids or REAL_SERVICES.values()
        with UPDATE(self._account, self._env, {'action': 'delete_account'}):
            for sid in list(self._account.subscriptions.keys()):
                if sid in REAL_SIDS:
                    service = Service.by_sid(sid)
                    try:
                        can_be_unsubscribed(self._account, service)
                    except (UnsubscribeBlockingServiceError, UnsubscribeProtectedServiceError):
                        pass
                    else:
                        delete_subscription(self._account, service)

    def _delete_personal_data(self):
        with UPDATE(self._account, self._env, {'action': 'delete_account'}):
            self._account.person.firstname = None
            self._account.person.lastname = None
            self._account.person.display_name = None
            self._account.person.default_avatar = None
            self._account.person.birthday = None
            self._account.person.country = None
            self._account.person.city = None
            self._account.person.timezone = None
            self._account.person.gender = None

    def _delete_all_data(self):
        self._social_api.delete_all_profiles_by_uid(self._account.uid)
        with DELETE(self._account, self._env, {'action': 'delete_account'}):
            self._create_deletion_tasks()

    def _delete_family_records(self):
        family_data = self._blackbox.family_info(self._account.family_info.family_id)
        family_info = FamilyInfo().parse(family_data)
        if str(self._account.uid) == str(self._account.family_info.admin_uid):
            events = {'action': 'delete_family_admin_account'}
            with DELETE(family_info, self._env, events):
                pass
        else:
            events = {'action': 'delete_family_member_account'}
            with UPDATE(family_info, self._env, events):
                family_info.remove_member_uid(self._account.uid)

    def _log_deletion(self):
        to_statbox({
            'mode': 'account_delete',
            'step': 'account_deleter',
            'action': 'deleted',
            'uid': self._account.uid,
        })
        AccountModificationInfosecLogger(
            uid=self._account.uid,
            mode='account_delete',
            step='account_deleter',
            action='deleted',
        ).log()

    def _is_personal_data_must_be_deleted(self):
        return datetime.now() - self._account.deletion_operation.started_at >= settings.PERSONAL_DATA_MUST_BE_DELETED_PERIOD

    def _create_deletion_tasks(self):
        phone_history_deletion_task = PhoneBindingsHistoryDeleteTask(
            uid=self._account.uid,
            deletion_started_at=datetime.now(),
        )
        with CREATE(phone_history_deletion_task, self._env, {}):
            pass

    def _delete_apple_tokens(self):
        try:
            self._social_api.delete_tokens_from_account(uid=self._account.uid, provider_name='apl', revoke=True)
        except (SocialApiRequestError, SocialApiTemporaryError) as e:
            log.error('Failed to delete Apple tokens: %s' % e)
        except Exception:
            log.error('Failed to delete Apple tokens', exc_info=True)


class _FamilyDeleter(object):
    def __init__(self, blackbox, env):
        self._blackbox = blackbox
        self._env = env
        self.account = None
        self.family_info = None

    def load_family(self):
        family_data = self._blackbox.family_info(
            self.account.family_info.family_id,
            get_members_info='all',
        )
        self.family_info = FamilyInfo().parse(family_data)

    def delete_family_records(self):
        if str(self.account.uid) == str(self.account.family_info.admin_uid):
            delete_family_invites_by_family_id(self.account.family_info.family_id)
            with DELETE(self.family_info, self._env, {'action': 'delete_family_admin_account'}):
                pass
        elif self.account.uid in self.family_info:
            with UPDATE(self.family_info, self._env, events={'action': 'delete_family_member_account'}):
                self.family_info.remove_member_uid(self.account.uid)

    def delete_kids(self):
        kid_uids = self.family_info.kids.members.keys()
        kids, _ = get_many_accounts_by_uids(
            kid_uids,
            self._blackbox,
            userinfo_args=dict(get_family_info=True),
        )

        for kid in kids:
            with DELETE(kid, self._env, events={'action': 'kiddish_delete'}):
                pass
