# -*- coding: utf-8 -*-

from __future__ import (
    absolute_import,
    unicode_literals,
)

from datetime import datetime
import json
import logging

from passport.backend.core.builders.blackbox.blackbox import Blackbox
from passport.backend.core.conf import settings
from passport.backend.core.env import Environment
from passport.backend.dbscripts.account_deleter import settings as local_settings
from passport.backend.dbscripts.account_deleter.base import delete_accounts
from passport.backend.dbscripts.utils import EntryPoint
from passport.backend.utils.common import chop


log = logging.getLogger('passport.backend.dbscripts.account_deleter')
output = logging.getLogger('passport.backend.dbscripts.account_deleter.output')


def _uids_from_blackbox(conf_name):
    conf = settings.ACCOUNT_DELETION[conf_name]
    now = datetime.now()
    rv = Blackbox().deletion_operations(
        deleted_after=now - conf['max_age'],
        deleted_before=now - conf['min_age'],
        chunk_no=0,
        chunk_count=1,
    )
    return [d['uid'] for d in rv['deletion_operations']]


def run(conf_name, dynamic=False, uids=None):
    uids = uids or []
    if dynamic:
        uids = _uids_from_blackbox(conf_name)

    uids = list(set(uids))
    uids_count = len(uids)

    log.info('Unprocessed accounts left: %d', uids_count)

    global_status = []
    for chunk in chop(uids, 100):
        local_status = delete_accounts(chunk, environment=Environment())
        global_status.extend(local_status)

        uids_count -= len(chunk)
        log.info('Unprocessed accounts left: %d', uids_count)

    output.info(json.dumps(global_status))


class Main(EntryPoint):
    SETTINGS = local_settings

    def run(self, args):
        run(args.conf_name, args.dynamic, args.uids)

    def get_lock_name(self, args):
        conf = settings.ACCOUNT_DELETION[args.conf_name]
        return conf['lock_name']

    def get_arg_parser(self):
        parser = super(Main, self).get_arg_parser()
        parser.add_argument('--uids', nargs='*', type=int, default=[], metavar='uid')
        parser.add_argument('--dynamic', action='store_true')
        parser.add_argument('conf_name')
        return parser


main = Main()
