""" Синхронизатор почтовых отписок в рассылятор """
import argparse
import logging
import os
import sys

import passport.backend.api.settings.settings as api_settings
from passport.backend.core.conf import settings
from passport.backend.core.sync_utils.chunk_loader import ChunkLoader
from passport.backend.core.sync_utils.readers import FileReader
from passport.backend.dbscripts.load_mail_unsubscriptions.loader import Loader


log = logging.getLogger('passport.backend.dbscripts.load_mail_unsubscriptions')


def file_path(path):
    if not os.path.exists(os.path.dirname(path)):
        raise argparse.ArgumentTypeError("Not a valid path")
    path = os.path.abspath(path)
    if os.path.exists(path) and not os.path.isfile(path):
        raise argparse.ArgumentTypeError("Not a file")
    return path


def load(args):
    with FileReader(path=args.uids_file) as reader:
        chunk_loader = ChunkLoader(
            reader=reader,
            state_file_path=args.state_file,
            load_state=args.restart,
        )
        loader = Loader(chunk_loader, args.dry_run)
        loader.run()
        log.info('Finished loading')


def setup_settings():
    settings.configure(api_settings)


def setup_logger(log_path):
    main_logger = logging.getLogger('passport')
    main_logger.setLevel('DEBUG')

    formatter = logging.Formatter('%(asctime)s: %(levelname)s: %(message)s')

    file_handler = logging.FileHandler(filename=log_path)
    file_handler.setLevel('DEBUG')
    file_handler.setFormatter(formatter)

    stdout_handler = logging.StreamHandler(stream=sys.stdout)
    stdout_handler.setLevel('DEBUG')
    stdout_handler.setFormatter(formatter)

    main_logger.addHandler(file_handler)
    main_logger.addHandler(stdout_handler)


def main():
    parser = argparse.ArgumentParser(description='Load mail unsubscriptions')
    parser.add_argument('--dry-run', action='store_true')
    parser.add_argument('--state-file', required=True, type=file_path)
    parser.add_argument('--uids-file', required=True, type=file_path)
    parser.add_argument('--restart', action='store_true')
    parser.add_argument('--log-path', required=True)
    args = parser.parse_args()
    setup_settings()
    setup_logger(args.log_path)
    load(args)
