# -*- coding: utf-8 -*-

from datetime import (
    datetime,
    timedelta,
)
import logging

from passport.backend.core.builders.blackbox import Blackbox
from passport.backend.core.conf import settings
from passport.backend.core.db.schemas import (
    phone_bindings_history_delete_tasks_table as pbh_delete_table,
    phone_bindings_history_table as pbh_table,
)
from passport.backend.core.dbmanager.manager import (
    find_sharded_dbm,
    get_dbm,
    safe_execute,
)
from passport.backend.core.exceptions import UnknownUid
from passport.backend.core.models.account import Account
from passport.backend.core.models.delete_tasks import PhoneBindingsHistoryDeleteTask
from passport.backend.core.runner.context_managers import DELETE
from passport.backend.core.utils.decorators import cached_property
from passport.backend.dbscripts.utils import Throttler


log = logging.getLogger(__name__)


class PhoneBindingsHistoryDeleter(object):
    def __init__(self, env):
        self._env = env
        self.throttler = Throttler(rps=settings.DATABASE_WRITE_PER_SECOND)

    @cached_property
    def blackbox(self):
        return Blackbox()

    @staticmethod
    def delete_phone_bindings_history(uid):
        dbm = find_sharded_dbm(pbh_table, uid)
        delete_query = pbh_table.delete(pbh_table.c.uid == uid)
        safe_execute(
            dbm.get_engine(),
            executable=lambda engine_: engine_.execute(delete_query),
            retries=settings.DATABASE_DELETE_RETRIES,
        ).close()

    def check_account_exists(self, uid):
        response = self.blackbox.userinfo(uid=uid)
        try:
            Account().parse(response)
            return True
        except UnknownUid:
            return False

    def check_task(self, task):
        if self.check_account_exists(task.uid):
            log.debug('Ignore task {}: account {} still exists'.format(task, task.uid))
            return False
        return True

    def check_and_execute_task(self, task):
        result = self.check_task(task)
        if result:
            log.debug('Deleting phone bindings for uid {}. Task {}'.format(task.uid, task))
            self.delete_phone_bindings_history(task.uid)
        return result

    @staticmethod
    def _chunked_query(query, id_column, db_name='passportdbcentral'):
        last_id = 0
        while True:
            chunk_query = (
                query
                .where(id_column > last_id)
                .order_by(id_column)
                .limit(settings.TASKS_QUERY_CHUNK_SIZE)
            )
            result = safe_execute(
                engine=get_dbm(db_name).get_engine(),
                executable=lambda _engine: _engine.execute(chunk_query),
            )
            i = 0
            for i, row in enumerate(result, 1):
                last_id = getattr(row, id_column.key)
                yield row
            if i < settings.TASKS_QUERY_CHUNK_SIZE:
                break

    def run(self):
        delete_until = datetime.now() - timedelta(days=settings.PHONE_BINDING_DELETION_THRESHOLD_DAYS)
        tasks_query = pbh_delete_table.select(
            pbh_delete_table.c.deletion_started_at <= delete_until,
        )

        result = self._chunked_query(tasks_query, pbh_delete_table.c.task_id)
        for task_data in result:
            task = PhoneBindingsHistoryDeleteTask().parse(dict(task_data.items()))
            deleted = False
            try:
                deleted = self.check_and_execute_task(task)
            except Exception:
                log.exception('Unhandled exception while processing task {}'.format(task))

            try:
                with DELETE(task, self._env, {}):
                    pass
            except Exception:
                log.exception('Unhandled exception while deleting task {}'.format(task))

            self.throttler.throttle(2 if deleted else 1)
