# -*- coding: utf-8 -*-

from __future__ import (
    absolute_import,
    unicode_literals,
)

from collections import defaultdict
from datetime import datetime
import logging

from passport.backend.core.builders.blackbox.blackbox import Blackbox
from passport.backend.core.builders.yasms import YaSms as YasmsBuilder
from passport.backend.core.dbmanager.exceptions import DBError
from passport.backend.core.eav_type_mapping import get_attr_type
from passport.backend.core.logging_utils.loggers import StatboxLogger
from passport.backend.core.models.phones.phones import (
    PhoneChangeSet,
    RemoveSecureOperation,
)
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.serializers.eav.exceptions import EavDeletedObjectNotFound
from passport.backend.core.serializers.eav.query import (
    EavDeleteAttributeQuery,
    EavDeletePhoneOperationCreatedQuery,
)
from passport.backend.core.utils.blackbox import get_many_accounts_by_userinfo_list
from passport.backend.core.yasms.notifications import notify_about_phone_changes
from passport.backend.core.yasms.phonenumber_alias import PhoneAliasManager
from passport.backend.core.yasms.unbinding import unbind_old_phone
from passport.backend.dbscripts import context
from passport.backend.dbscripts.phone_harvester.db import (
    EavDeleteAllPhoneBindingsQuery,
    EavDeleteAllPhoneExtendedAttributesQuery,
    EavDeleteAllPhoneOperationsQuery,
    execute_in_transaction,
)
from passport.backend.dbscripts.utils import get_many_userinfo_with_phones_by_uids_inconsistency_safe


log = logging.getLogger('passport.backend.dbscripts.phone_harvester')


def harvest_uids(uids, environment):
    userinfo_list, removed_uids = get_many_userinfo_with_phones_by_uids_inconsistency_safe(uids, Blackbox())

    try:
        collect_garbage(userinfo_list, removed_uids)
    except Exception:
        log.error('Error occurred while collecting garbage', exc_info=True)

    accounts, _ = get_many_accounts_by_userinfo_list(userinfo_list)
    for account in accounts:
        with context.set({'uid': account.uid}):
            harvest_account(account, environment)


def harvest_account(account, environment):
    log.debug('Start processing of account %d', account.uid)

    external_events = {'action': 'harvest_expired_phone_operations'}
    statbox = StatboxLogger(is_harvester=True)
    yasms_builder = YasmsBuilder()
    event_timestamp = datetime.now()

    try:
        with notify_about_phone_changes(
            account=account,
            yasms_builder=yasms_builder,
            statbox=statbox,
            consumer=None,
        ), UPDATE(
            account,
            environment,
            external_events,
            datetime_=event_timestamp,
        ):
            change_set = harvest_expired_phone_operations(account, statbox)

        statbox.dump_stashes()

        # TODO: Создать алиас после замены защищённого телефона на новый
        # телефон с алиасным флажком.

        for phone_number in change_set.bound_numbers:
            phone = account.phones.by_number(phone_number)
            unbind_old_phone(
                subject_phone=phone,
                blackbox_builder=Blackbox(),
                statbox=statbox,
                consumer=None,
                event_timestamp=event_timestamp,
                environment=environment,
            )
    except EavDeletedObjectNotFound:
        # Ошибка может произойти, если с момента чтения из ЧЯ объект
        # удалили (гонка), или если ЧЯ сообщил, что объект существует, но его
        # нет (отстала реплика).
        log.warning(
            'Unexpected state of data found while processing account (uid=%d)',
            account.uid,
        )
    except Exception as e:
        # Любой отказ на аккаунте не должен быть причиной пропуска других
        # аккаунтов.
        log.error(
            'Error occurred while processing account (uid=%d)',
            account.uid,
            exc_info=e,
        )
    else:
        log.debug('Account %d is processed', account.uid)


def harvest_expired_phone_operations(account, statbox):
    total_change_set = PhoneChangeSet()

    for logical_op in account.phones.get_logical_operations(statbox):
        is_removal_of_secure_phone = type(logical_op) is RemoveSecureOperation
        forbid_removal_of_secure_phone = account.totp_secret.is_set or account.sms_2fa_on

        if (
            logical_op.in_quarantine and
            logical_op.is_expired and
            not (is_removal_of_secure_phone and forbid_removal_of_secure_phone)
        ):
            _, change_set = logical_op.apply(is_harvester=True)
            total_change_set += change_set
            log.debug('Logical operation is applied: %r', logical_op)

        elif logical_op.is_expired:
            logical_op.cancel(is_harvester=True)
            log.debug('Logical operation is cancelled: %r', logical_op)

        else:
            log.debug('Logical operation is skipped: %r', logical_op)

    if (account.phonenumber_alias.number and
            account.phonenumber_alias.number.e164 in total_change_set.unbound_numbers):
        phone_alias_manager = PhoneAliasManager(
            statbox=statbox,
            consumer=None,
            environment=None,
            need_update_tx=False,
        )
        phone_alias_manager.delete_alias(account, phone_alias_manager.ALIAS_DELETE_REASON_OFF)

    # TODO: Обелить учётную запись после привязки чистого номера.

    return total_change_set


def collect_garbage(userinfo_list, removed_uids):
    uid_to_eav_queries = defaultdict(list)

    for uid in removed_uids:
        uid_to_eav_queries[uid] += [
            EavDeleteAllPhoneExtendedAttributesQuery(uid),
            EavDeleteAllPhoneBindingsQuery(uid),
            EavDeleteAllPhoneOperationsQuery(uid),
            EavDeleteAttributeQuery(
                uid,
                map(
                    get_attr_type,
                    [
                        'phones.secure',
                        'phones.default',
                    ],
                ),
            ),
        ]

    if removed_uids:
        formatted_removed_uids = ', '.join(map(str, removed_uids))
        log.info('Collect phone operations on removed accounts: %s' % formatted_removed_uids)

    # Вычищаем операции без телефона
    phone_operations = list()
    phones = dict()
    for userinfo in userinfo_list:
        phone_operations += userinfo.get('phone_operations', dict()).values()
        uid = userinfo['uid']
        for phone_id, phone in userinfo.get('phones', dict()).items():
            phones.update({(uid, phone_id): phone})

    orphaned_phone_operations = list()
    for phone_op in phone_operations:
        phone_key1 = (phone_op['uid'], phone_op['phone_id'])
        phone1 = phone2 = phones.get(phone_key1)
        if phone_op['phone_id2'] is not None:
            phone_key2 = (phone_op['uid'], phone_op['phone_id2'])
            phone2 = phones.get(phone_key2)
        if phone1 is None or phone2 is None:
            orphaned_phone_operations.append(phone_op)

    for phone_op in orphaned_phone_operations:
        uid = phone_op['uid']
        uid_to_eav_queries[uid] += [
            EavDeletePhoneOperationCreatedQuery(uid, phone_op['id']),
        ]

    if orphaned_phone_operations:
        formatted_orphaned_phone_operations = list()
        for phone_op in orphaned_phone_operations:
            formatted_orphaned_phone_operations.append(
                '(uid=%s, operation_id=%s)' % (phone_op['uid'], phone_op['id']),
            )
        formatted_orphaned_phone_operations = ', '.join(formatted_orphaned_phone_operations)
        log.info('Collect orphaned phone operations: %s' % formatted_orphaned_phone_operations)

    for uid, eav_queries in uid_to_eav_queries.items():
        try:
            execute_in_transaction(eav_queries)
        except DBError:
            log.error('Database failed while collecting garbage from deleted account (uid=%d)', uid)
