# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from passport.backend.core.dbmanager.manager import safe_execute_queries
from passport.backend.core.dbmanager.transaction_manager import full_transaction
from passport.backend.core.eav_type_mapping import EXTENDED_ATTRIBUTES_PHONE_TYPE
from passport.backend.core.serializers.eav.query import (
    EavExtendedAttributeQuery,
    EavPhoneBindingCreatedQuery,
    EavPhoneOperationCreatedQuery,
)
from sqlalchemy.sql.expression import (
    and_,
    or_,
)


@full_transaction
def execute_in_transaction(eav_queries):
    safe_execute_queries(eav_queries)


class _EavDeleteExtendedAttributesQuery(EavExtendedAttributeQuery):
    """
    Удаляет атрибуты пользователя по данным критериям или все атрибуты.
    """
    def __init__(self, uid, ext_attrs_data):
        super(_EavDeleteExtendedAttributesQuery, self).__init__(uid)
        self.ext_attrs_data = ext_attrs_data

    def query_params(self):
        return self.ext_attrs_data

    def to_query(self):
        t = self.get_table()

        del_conditions = []
        for item in self.ext_attrs_data:
            cond = []
            for column in {'entity_type', 'entity_id', 'type'}:
                if column in item:
                    cond.append(getattr(t.c, column) == item[column])
            del_conditions.append(and_(*cond))

        return t.delete().where(and_(t.c.uid == self.uid, or_(*del_conditions)))


class EavDeleteAllPhoneExtendedAttributesQuery(_EavDeleteExtendedAttributesQuery):
    def __init__(self, uid):
        super_class = super(EavDeleteAllPhoneExtendedAttributesQuery, self)
        super_class.__init__(
            uid=uid,
            ext_attrs_data=[{'entity_type': EXTENDED_ATTRIBUTES_PHONE_TYPE}],
        )


class _EavDeletePhoneBindingsQuery(EavPhoneBindingCreatedQuery):
    """
    Удаляет записи из таблицы phone_bindings для данного пользователя (все или
    для данных телефонов).
    """
    def __init__(self, uid, phone_ids):
        super(_EavDeletePhoneBindingsQuery, self).__init__(uid)
        self._phone_ids = phone_ids

    def to_query(self):
        t = self.get_table()
        cond = and_(t.c.uid == self.uid)
        if self._phone_ids is not None:
            cond = and_(cond, t.c.phone_id.in_(self._phone_ids))
        return t.delete().where(cond)


class EavDeleteAllPhoneBindingsQuery(_EavDeletePhoneBindingsQuery):
    def __init__(self, uid):
        super_class = super(EavDeleteAllPhoneBindingsQuery, self)
        super_class.__init__(uid=uid, phone_ids=None)


class _EavDeletePhoneOperationsQuery(EavPhoneOperationCreatedQuery):
    """
    Удаляет записи из таблицы phone_operations для данного пользователя (все или
    заданные).
    """
    def __init__(self, uid, operation_ids):
        super(_EavDeletePhoneOperationsQuery, self).__init__(uid)
        self._operation_ids = operation_ids

    def to_query(self):
        t = self.get_table()
        cond = and_(t.c.uid == self.uid)
        if self._operation_ids is not None:
            cond = and_(cond, t.c.phone_id.in_(self._operation_ids))
        return t.delete().where(cond)


class EavDeleteAllPhoneOperationsQuery(_EavDeletePhoneOperationsQuery):
    def __init__(self, uid):
        super_class = super(EavDeleteAllPhoneOperationsQuery, self)
        super_class.__init__(uid=uid, operation_ids=None)
