# -*- coding: utf-8 -*-

from __future__ import (
    absolute_import,
    unicode_literals,
)

import contextlib
import datetime
import functools
import json
import logging
import uuid

from passport.backend.core.logging_utils.loggers.dummy import DummyLogger
from passport.backend.core.models.phones.phones import (
    Operation as PhoneOperation,
    PhoneBinding,
)
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.types.phone_number.phone_number import PhoneNumber
from passport.backend.core.undefined import Undefined
from passport.backend.core.yasms.phone_numbering import PhoneNumberTranslator
from passport.backend.utils.time import datetime_to_unixtime


log = logging.getLogger(__name__)


@contextlib.contextmanager
def DRY_RUN_AWARE_UPDATE(instances, environment, events, dry_run=False, *args, **kwargs):
    if dry_run:
        yield
    else:
        with UPDATE(instances, environment, events, *args, **kwargs):
            yield


def translate_phone_number(blackbox, dry_run, env, phone_number, statbox, uid):
    try:
        translator = PhoneNumberTranslator.build_from_uid(
            uid=uid,
            blackbox=blackbox,
            consumer=None,
            old_phone_number=phone_number,
            statbox=statbox if not dry_run else DummyLogger(),
        )
    except PhoneNumberTranslator.UnknownUid:
        log.info('Account not found: %s' % uid)
        return
    except PhoneNumberTranslator.InapplicablePhoneTranslationRule:
        log.info('No rule to translate phone: %s' % phone_number.e164)
        return

    try:
        translator.check_all()
    except translator.PhoneNotFoundError:
        log.info('Phone %s not found on %s' % (phone_number.e164, uid))
        return
    except translator.UnableToTranslatePhoneNumber:
        log.info('Unable to translate phone %s on %s' % (phone_number.e164, uid))
        return

    UPDATE = functools.partial(DRY_RUN_AWARE_UPDATE, dry_run=dry_run)

    with TransactionLogger.build(log) as txlog:
        if translator.old_phonenumber_alias_owner:
            txlog.old_phonenumber_alias_owner(translator)
            with UPDATE(
                translator.old_phonenumber_alias_owner,
                env,
                dict(
                    action='phone_alias_delete',
                    initiator_uid=str(uid),
                ),
            ):
                translator.take_phonenumber_alias_away_from_old_owner()

        txlog.account(translator)
        with UPDATE(translator.account, env, dict(action='apply_phone_translation_rule')):
            translator.translate()

    message = 'Successfully translated %s to %s on %s' % (
        translator.old_phone_number.e164,
        translator.new_phone_number.e164,
        translator.account.uid,
    )
    if dry_run:
        message += ' (dry_run)'
    log.info(message)


class TransactionLogger(object):
    def __init__(self):
        self.logger = None
        self.tx_id = uuid.uuid4().hex

    @classmethod
    def build(cls, logger):
        self = TransactionLogger()
        self.logger = logger
        return self

    def __enter__(self):
        self.info('Begin transaction')
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        if exc_type is None and exc_val is None and exc_tb is None:
            self.info('Transaction commited')
        else:
            self.info('Transaction failed')
        return False

    def old_phonenumber_alias_owner(self, translator):
        alias_owner = translator.old_phonenumber_alias_owner
        self.info(
            self.build_checkpoint(
                **dict(
                    alias_owner.phonenumber_alias,
                    checkpoint_id='old_phonenumber_alias_owner',
                    uid=alias_owner.uid,
                )
            ),
        )

    def account(self, translator):
        self.info(
            self.build_checkpoint(
                checkpoint_id='account',
                new_phone=dict(translator.new_phone) if translator.new_phone else None,
                old_phone=dict(translator.old_phone),
                phonenumber_alias=dict(translator.account.phonenumber_alias),
                secure_phone_id=translator.account.phones.secure_id,
                uid=translator.account.uid,
            ),
        )

    def info(self, message):
        self.logger.debug('tx_id=%s %s' % (self.tx_id, message))

    def build_checkpoint(self, **kwargs):
        return json.dumps(
            kwargs,
            ensure_ascii=False,
            separators=(',', ':'),
            default=self.default_json_encode,
        )

    @staticmethod
    def default_json_encode(o):
        if o is Undefined:
            return 'Undefined'
        type_of_o = type(o)
        if type_of_o is PhoneNumber:
            return o.e164
        if type_of_o is datetime.datetime:
            return datetime_to_unixtime(o)
        if type_of_o is PhoneBinding:
            return dict(o)
        if type_of_o is PhoneOperation:
            return o.id
        raise TypeError('%r is not JSON serializable' % o)
