# -*- coding: utf-8 -*-

from __future__ import (
    absolute_import,
    unicode_literals,
)

import argparse
import logging

from passport.backend.core.builders.blackbox import Blackbox
from passport.backend.core.env import Environment
from passport.backend.core.logging_utils.loggers import StatboxLogger
from passport.backend.core.types.phone_number.phone_number import (
    InvalidPhoneNumber,
    PhoneNumber,
)
from passport.backend.dbscripts.utils import EntryPoint

from .base import translate_phone_number


log = logging.getLogger(__name__)


def escaped(s):
    return s.replace('\n', r'\n').replace('\r', r'\r')


def parse_line(line):
    bits = line.strip().split()
    if len(bits) < 2:
        raise ParsingError(line)

    try:
        uid = int(bits[0])
    except ValueError:
        raise ParsingError(line)

    phone_number = bits[1]
    if phone_number[0] != '+':
        phone_number = '+' + phone_number
    try:
        phone_number = PhoneNumber.parse(phone_number)
    except InvalidPhoneNumber:
        raise ParsingError(line)

    return uid, phone_number


def read_file(_file):
    for line in _file:
        try:
            yield parse_line(line)
        except ParsingError:
            log.error('Failed to parse line: %s' % escaped(line))


def run(dry_run, _file):
    blackbox = Blackbox()
    env = Environment()
    statbox = StatboxLogger()

    for uid, phone_number in read_file(_file):
        try:
            translate_phone_number(
                blackbox=blackbox,
                dry_run=dry_run,
                env=env,
                phone_number=phone_number,
                statbox=statbox,
                uid=uid,
            )
        except Exception as e:
            log.error('Unhandled exception: %s, %s, %s' % (uid, phone_number.e164, type(e).__name__), exc_info=True)


class Main(EntryPoint):
    LOCK_NAME = '/passport/phone_number_translator/global_lock'

    def run(self, args):
        run(args.dry_run, args.file)

    def get_arg_parser(self):
        parser = super(Main, self).get_arg_parser()
        parser.add_argument('--dry-run', action='store_const', const=True, dest='dry_run')
        parser.add_argument('--from', dest='file', metavar='path', required=True, type=argparse.FileType('r'))
        return parser


class ParsingError(Exception):
    pass


main = Main()
