# -*- coding: utf-8 -*-
import logging
from random import SystemRandom
from time import sleep

from passport.backend.core.conf import settings
from passport.backend.core.db.read_api.runner import EavReadingActionRunner
from passport.backend.dbscripts.randoms_generator import settings as local_settings
from passport.backend.dbscripts.randoms_generator.db import (
    CleanTableQuery,
    GetKeyspacesQuery,
    InsertKeyQuery,
)
from passport.backend.dbscripts.utils import EntryPoint
from passport.backend.utils.string import smart_text


log = logging.getLogger('passport.backend.dbscripts.randoms_generator')


def generate_key(alphabet, size):
    rnd = SystemRandom()
    return ''.join(rnd.choice(alphabet) for _ in range(size)).encode()


def run(process_randoms, process_lrandoms):
    if not (process_randoms or process_lrandoms):
        log.info('No action requested - exiting')
        return

    runner = EavReadingActionRunner()

    keyspaces = settings.KEYSPACES_AND_TABLES_TO_PROCESS
    if not keyspaces:
        keyspaces = runner.execute(GetKeyspacesQuery())

    for keyspace, table_name in keyspaces.items():
        table_name = smart_text(table_name)
        keyspace = smart_text(keyspace)
        if (
            (keyspace == 'cookiel' and not process_lrandoms) or
            (keyspace != 'cookiel' and not process_randoms)
        ):
            continue

        keyspace_settings = settings.KEYSPACE_SETTINGS.get(keyspace, settings.DEFAULT_KEYSPACE_SETTINGS)
        runner.execute(InsertKeyQuery(
            table_name=table_name,
            key=generate_key(alphabet=keyspace_settings['alphabet'], size=keyspace_settings['size']),
            ttl=keyspace_settings['ttl'],
        ))
        runner.execute(CleanTableQuery(
            table_name=table_name,
        ))

    # Делаем паузу, чтобы не не успели исполниться скрипты на соседних машинах
    # (на случай расхождения времени между машинами)
    sleep(settings.MAX_TIME_DISCREPANCY)


class Main(EntryPoint):
    LOCK_NAME = '/passport/randoms_generator/global_lock'
    SETTINGS = local_settings

    def run(self, args):
        run(args.randoms, args.lrandoms)

    def get_arg_parser(self):
        parser = super(Main, self).get_arg_parser()
        parser.add_argument('--randoms', action='store_true')
        parser.add_argument('--lrandoms', action='store_true')
        return parser


main = Main()
