# -*- coding: utf-8 -*-
from datetime import datetime

from passport.backend.core.db.query import DbQuery
from passport.backend.core.db.schemas import central_metadata
from passport.backend.core.db.types import TIMESTAMP
from sqlalchemy.schema import (
    Column,
    Table,
)
from sqlalchemy.sql import select
from sqlalchemy.sql.functions import current_timestamp
from sqlalchemy.types import (
    DateTime,
    Integer,
    VARBINARY,
    VARCHAR,
)


keyspaces_table = Table(
    'keyspaces',
    central_metadata,
    Column('groupid', Integer, primary_key=True, nullable=False, autoincrement=True),
    Column('domainsuff', VARCHAR(50), nullable=False, default='', unique=True),
    Column('tablename', VARCHAR(30), nullable=False, default=''),
    Column('addpref', Integer, nullable=False, default=0),
    Column('inuse', Integer, nullable=False, default=1),
)

lrandoms_table = Table(
    'lrandoms',
    central_metadata,
    Column('id', Integer, primary_key=True, nullable=False, autoincrement=True),
    Column('keybody', VARBINARY(255), nullable=False, default=b''),  # ключ для подписи
    Column('start', DateTime, nullable=False),  # дата создания ключа
    Column('valid', DateTime, nullable=False),  # дата протухания ключа
)


def get_randoms_table(table_name):
    # lrandoms отличается от остальных таблиц типом временных полей (DATETIME vs TIMESTAMP)
    # TODO: когда-нибудь устранить различие и/или слить все randoms в одну таблицу
    if table_name == 'lrandoms':
        return lrandoms_table
    return Table(
        table_name,
        central_metadata,
        Column('id', Integer, primary_key=True, nullable=False, autoincrement=True),
        Column('keybody', VARBINARY(255), nullable=False, default=''),  # ключ для подписи
        Column('start', TIMESTAMP(), server_default=current_timestamp(), nullable=False),  # дата создания ключа
        Column('valid', TIMESTAMP(), nullable=False),  # дата протухания ключа
        extend_existing=True,
    )


class GetKeyspacesQuery(DbQuery):
    def get_table(self):
        return keyspaces_table

    def to_query(self):
        t = self.get_table()
        return select([t.c.domainsuff, t.c.tablename])

    def parse_query_result(self, query_result):
        return {keyspace: table_name for keyspace, table_name in query_result.fetchall()}


class InsertKeyQuery(DbQuery):
    def __init__(self, table_name, key, ttl):
        super(InsertKeyQuery, self).__init__()
        self._table_name = table_name
        self._key = key
        self._created = datetime.now()
        self._expires = datetime.now() + ttl

    def get_table(self):
        return get_randoms_table(self._table_name)

    def query_params(self):
        return {
            'keybody': self._key,
            'start': self._created,
            'valid': self._expires,
        }

    def to_query(self):
        t = self.get_table()
        return t.insert().values(**self.query_params())

    def parse_query_result(self, query_result):
        return


class CleanTableQuery(DbQuery):
    def __init__(self, table_name):
        super(CleanTableQuery, self).__init__()
        self._table_name = table_name

    def get_table(self):
        return get_randoms_table(self._table_name)

    def to_query(self):
        t = self.get_table()
        return t.delete().where(t.c.valid < datetime.now())

    def parse_query_result(self, query_result):
        return
