# -*- coding: utf-8 -*-

from collections import namedtuple
from datetime import datetime
import logging

from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.subscription import user_has_contract_with_yandex
from passport.backend.core.types.birthday import Birthday
from passport.backend.core.types.gender import Gender
from passport.backend.core.utils.blackbox import get_many_accounts_by_uids


log = logging.getLogger(__name__)

_MIN_SUBNAME_LENGTH = 4
_DataFromProvider = namedtuple(
    '_DataFromProvider',
    [
        'birthdays',
        'emails',
        'genders',
        'names',
        'profile_ids',
        'uids',
    ],
)


def delete_data_obtained_from_social_provider(provider_code, userid, blackbox,
                                              social_api, environment):
    data_from_provider = _get_social_data_obtained_from_provider(social_api, provider_code, userid)

    userinfo_args = dict(email_attributes='all')
    accounts, _ = get_many_accounts_by_uids(
        data_from_provider.uids,
        blackbox,
        userinfo_args,
    )

    i = 0
    while i < len(accounts):
        if user_has_contract_with_yandex(accounts[i]):
            log.error(
                'Unable to delete account (%s) data, because user has contract '
                'with Yandex' % accounts[i].uid,
            )
            del accounts[i]
        else:
            i += 1

    log.info(
        'Delete social data obtained from (%s, %s) on accounts %s' %
        (provider_code, userid, ', '.join(str(a.uid) for a in accounts)),
    )

    timestamp = datetime.now()
    for account in accounts:
        with UPDATE(
            account,
            environment=environment,
            events={'action': 'delete_data_obtained_from_social_provider'},
            datetime_=timestamp,
        ):
            _delete_passport_data(account, data_from_provider)
            account.global_logout_datetime = timestamp

    social_api.delete_social_data(data_from_provider.profile_ids)


def _delete_passport_data(account, data_from_provider):
    names = data_from_provider.names or []
    birthdays = data_from_provider.birthdays or []
    genders = data_from_provider.genders or []
    emails = data_from_provider.emails or []

    account.person.default_avatar = None
    _delete_firstname_if_match(account, names)
    _delete_lastname_if_match(account, names)
    _delete_display_name_if_match(account, names)
    _delete_birthday_if_match(account, birthdays)
    _delete_gender_if_match(account, genders)
    _delete_emails_if_match(account, emails)


def _delete_firstname_if_match(account, names):
    if not account.person.firstname:
        return
    if any(_names_match(n, account.person.firstname) for n in names):
        account.person.firstname = None


def _delete_lastname_if_match(account, names):
    if not account.person.lastname:
        return
    if any(_names_match(n, account.person.lastname) for n in names):
        account.person.lastname = None


def _delete_display_name_if_match(account, names):
    if not account.person.display_name.name:
        return
    if any(_names_match(n, account.person.display_name.name) for n in names):
        account.person.display_name.name = None


def _delete_birthday_if_match(account, birthdays):
    if not account.person.birthday:
        return
    if any(b == account.person.birthday for b in birthdays):
        account.person.birthday = None


def _delete_gender_if_match(account, genders):
    if not account.person.gender:
        return
    if any(g == account.person.gender for g in genders):
        account.person.gender = None


def _delete_emails_if_match(account, emails):
    for email_address in emails:
        email = account.emails.find(email_address)
        if email and email.is_unsafe:
            del account.emails[email_address]


def _names_match(s1, s2):
    s1 = s1.lower().strip()
    s2 = s2.lower().strip()
    return (
        s1 == s2 or
        len(s1) >= _MIN_SUBNAME_LENGTH and s1 in s2 or
        len(s2) >= _MIN_SUBNAME_LENGTH and s2 in s1
    )


def _get_social_data_obtained_from_provider(social_api, provider_code, userid):
    profiles = social_api.get_profiles(userid, provider_code, person=True)

    profile_ids = set()
    uids = set()
    names = set()
    birthdays = set()
    genders = set()
    emails = set()

    for profile in profiles:
        profile_id = profile.get('profile_id')
        if profile_id:
            profile_ids.add(profile_id)

        uid = profile.get('uid')
        if uid:
            uids.add(uid)

        username = profile.get('username')
        if username:
            names.add(username)

        person = profile.get('person', dict())
        for key in ['firstname', 'lastname', 'nickname']:
            name = person.get(key)
            if name:
                names.add(name)

        try:
            birthday = Birthday.parse(person.get('birthday'))
        except ValueError:
            pass
        else:
            birthdays.add(birthday)

        gender = Gender.from_char(person.get('gender'))
        genders.add(gender)

        email = person.get('email')
        if email:
            emails.add(email)

    return _DataFromProvider(
        profile_ids=profile_ids,
        uids=uids,
        names=names,
        birthdays=birthdays,
        genders=genders,
        emails=emails,
    )
