# -*- coding: utf-8 -*-

from __future__ import (
    absolute_import,
    unicode_literals,
)

from argparse import ArgumentParser
import functools
import itertools
import logging
import multiprocessing.pool
import operator
import os
from signal import (
    SIG_IGN,
    SIGINT,
    signal,
)
import sys
from time import (
    sleep,
    time,
)

from passport.backend.core.builders.blackbox.exceptions import BlackboxTemporaryError
from passport.backend.core.builders.blackbox.utils import add_phone_arguments
from passport.backend.core.conf import settings
from passport.backend.core.dbmanager.manager import get_dbm
from passport.backend.core.dbmanager.sharder import (
    build_range_shard_function,
    get_sharder,
)
from passport.backend.core.utils.blackbox import (
    get_many_accounts_by_userinfo_list,
    get_many_userinfo_by_uids,
)
from passport.backend.dbscripts import (
    settings as local_settings,
    templating,
)
from passport.backend.dbscripts.logging import (
    bootstrap_logging__first_step,
    initialize_logging,
)
from passport.backend.utils.lock import (
    lock,
    LockError,
)
import yenv


log = logging.getLogger('passport.backend.dbscripts')


def get_many_accounts_with_phones_by_uids_inconsistency_safe(uids, blackbox):
    userinfo_list, removed_uids = get_many_userinfo_with_phones_by_uids_inconsistency_safe(uids, blackbox)
    accounts, _ = get_many_accounts_by_userinfo_list(userinfo_list)
    return accounts, removed_uids


def get_many_userinfo_with_phones_by_uids_inconsistency_safe(uids, blackbox):
    # Почта нужна чтобы высылать уведомления
    userinfo_args = dict(emails=True)
    userinfo_args = add_phone_arguments(**userinfo_args)
    splitter = _BadUidsSplitter(uids, blackbox, userinfo_args)
    _, bad = splitter.split()
    if bad:
        log.error('Unparseble uids: %r' % bad)
    return splitter.userinfo_list, splitter.removed_uids


def round_robin_iter(iterables):
    """
    Обход списка iterables по алгоритму round-robin
    """
    end = object()
    is_not_end = functools.partial(operator.is_not, end)
    return filter(
        is_not_end,
        itertools.chain.from_iterable(
            itertools.zip_longest(*iterables, fillvalue=end),
        ),
    )


def _initialize(is_quiet, local_settings):
    settings.configure(local_settings)

    initialize_logging(settings.LOG_BASE_DIR, is_quiet)

    # Создаем подключения к базе
    for db_name, config in settings.DB_CONFIG.items():
        get_dbm(db_name).configure(config)

    shard_function = build_range_shard_function(settings.DB_SHARDING_RANGES)
    for table_name, config in settings.DB_SHARDING_CONFIG.items():
        get_sharder(table_name).configure(config, shard_function)

    template_base_dir = os.path.join(
        os.path.dirname(os.path.abspath(__file__)),
        'templates',
    )
    templating.initialize(template_base_dir)


class Envs(object):
    def __init__(self):
        self._env_list = list()

    @classmethod
    def from_tuples(cls, ts):
        """
        Входные параметры

        ts - Список пар (yenv.name, yenv.type)

        Вместо любого элемента пары можно использовать '*', если на месте этого
        элемента допускается любое значение.
        """
        self = Envs()
        self._env_list.extend(ts)
        return self

    def match(self, yenv):
        for allowed_env_name, allowed_env_type in self._env_list:
            if (
                (allowed_env_name == '*' or allowed_env_name == yenv.name) and
                (allowed_env_type == '*' or allowed_env_type == yenv.type)
            ):
                return True
        return False


class EntryPoint(object):
    ALLOWED_ENVS = Envs.from_tuples([('*', '*')])
    LOCK_NAME = 'default'
    SETTINGS = local_settings
    WORKER_POOL_SIZE = 0

    def __init__(self):
        self._worker_pool = None

    def get_arg_parser(self):
        parser = ArgumentParser()
        parser.add_argument('--quiet', action='store_true')
        return parser

    def get_lock_name(self, args):
        return self.LOCK_NAME

    def get_settings(self):
        return self.SETTINGS

    def is_allowed_env(self, reason_list=None):
        if self.ALLOWED_ENVS.match(yenv):
            return True

        if reason_list is not None:
            reason_list[0] = 'forbidden environment: name=%s, type=%s' % (yenv.name, yenv.type)
        return False

    def run(self, args):
        raise NotImplementedError()

    def _get_args(self):
        parser = self.get_arg_parser()
        return parser.parse_args(sys.argv[1:])

    def _build_worker_pool(self, args, local_settings):
        if self.WORKER_POOL_SIZE > 0:
            log.info('Create workers')
            self._worker_pool = _WorkerPool(
                processes=self.WORKER_POOL_SIZE,
                initializer=self._initialize_worker,
                initargs=(args, local_settings),
            )

    def _initialize_worker(self, args, local_settings):
        # Чтобы подпроцессы не убивались по CTRL-C.
        # Останавливайте процессы в родителе.
        signal(SIGINT, SIG_IGN)
        _initialize(args.quiet, local_settings)

    def _close_worker_pool(self):
        if self._worker_pool:
            log.info('Terminate workers...')
            signal(SIGINT, lambda s, f: log.info('Still terminate workers. Please wait.'))
            self._worker_pool.graceful_terminate()
            self._worker_pool = None

    def _run_and_close(self, args):
        try:
            self.run(args)
        finally:
            self._close_worker_pool()

    def __call__(self):
        self._worker_pool = None

        bootstrap_logging__first_step()

        try:
            args = self._get_args()
            local_settings = self.get_settings()
            # Важно создать пул процессов как можно раньше, чтобы не шарить
            # файловые дескрипторы, сокеты и т.п.
            self._build_worker_pool(args, local_settings)
            _initialize(args.quiet, local_settings)

            reason = ['']
            if not self.is_allowed_env(reason):
                log.info('Script `%s` is disabled: %s', sys.argv[0], reason[0])
                return

            lock_name = '/%s%s' % (settings.ZOOKEEPER_LOCK_PREFIX, self.get_lock_name(args))
            with lock(
                ylock_config=settings.YLOCK_CONFIG,
                lock_name=lock_name,
            ) as acquired:
                if acquired:
                    self._run_and_close(args)
        except BlackboxTemporaryError:
            log.warning('Blackbox is unavailable')
            sys.exit(1)
        except LockError:
            log.error('Zookeeper is unavailable')
            sys.exit(1)
        except Exception as e:
            log.error('Unhandled exception', exc_info=e)
            sys.exit(1)
        finally:
            self._close_worker_pool()


class FooBarSplitter(object):
    def __init__(self, candidates):
        self._candidates = candidates

    def split(self):
        return self._split(self._candidates)

    def _is_bar_exists(self, candidates):
        return any(c % 2 == 0 for c in candidates)

    def _split(self, candidates):
        if self._is_bar_exists(candidates):
            if len(candidates) > 1:
                half = len(candidates) // 2
                foos1, bars1 = self._split(candidates[:half])
                foos2, bars2 = self._split(candidates[half:])
                foos = foos1 + foos2
                bars = bars1 + bars2
            else:
                foos = []
                bars = candidates
        else:
            foos = candidates
            bars = []
        return (foos, bars)


class Throttler(object):
    def __init__(self, rps):
        self._rps = float(rps)
        self._latest_moment = None

    def _evaluate_throttle_time(self, requests_passed):
        if self._latest_moment is not None:
            time_passed = self._get_time() - self._latest_moment
            throttle_time = requests_passed * (1. / self._rps) - time_passed
            if throttle_time < 0:
                throttle_time = 0
        else:
            throttle_time = 0
        return throttle_time

    def _get_time(self):
        return time()  # pragma: no cover

    def _sleep(self, seconds):
        sleep(seconds)  # pragma: no cover

    def throttle(self, events_passed=1):
        throttle_time = self._evaluate_throttle_time(events_passed)
        if throttle_time > 0:
            self._sleep(throttle_time)

        self._latest_moment = self._get_time()

    def decrease_rps(self):
        self._rps *= 0.9
        log.info('Throttler: rps decreased to %.2f' % self._rps)

    def increase_rps(self):
        self._rps *= 1.1
        log.info('Throttler: rps increased to %.2f' % self._rps)


class ThrottlerControl(object):
    def __init__(self, throttler, decrease_signum, increase_signum):
        self._throttler = throttler
        self._decrease_signum = decrease_signum
        self._increase_signum = increase_signum

    def _signal(self, signum, handler):
        signal(signum, handler)  # pragma: no cover

    def setup(self):
        self._signal(self._decrease_signum, self.handle_signal)
        self._signal(self._increase_signum, self.handle_signal)

    def handle_signal(self, signum, frame):
        if signum == self._decrease_signum:
            self._throttler.decrease_rps()
        elif signum == self._increase_signum:
            self._throttler.increase_rps()


class _BadUidsSplitter(FooBarSplitter):
    def __init__(self, candidates, blackbox, userinfo_args=None):
        super(_BadUidsSplitter, self).__init__(candidates)
        self.userinfo_list = list()
        self.removed_uids = set()
        self._blackbox = blackbox
        self._userinfo_args = userinfo_args

    def _is_bar_exists(self, candidates):
        try:
            userinfo_list, removed_uids = get_many_userinfo_by_uids(candidates, self._blackbox, self._userinfo_args)
        except BlackboxTemporaryError:
            raise
        except Exception:
            return True
        else:
            self.userinfo_list += userinfo_list
            self.removed_uids |= removed_uids
            return False


class _WorkerPool(multiprocessing.pool.Pool):
    def graceful_terminate(self):
        """
        Перед остановом, даёт воркерам закончить уже выполняемые задачи.
        """

        # Гарантируем, что новые задачи не будут добавляться в очередь
        self.close()

        # В очереди может быть очень много задач запланированных на
        # выполнение, вычистим их все чтобы не ждать их обработку.
        while not self._inqueue.empty():
            self._inqueue.get()

        # Даём сигнал на завершение процессу, который следит за воркерами.
        # Он в свою очередь косвенно даёт сигнал на завершение воркеров.
        self._worker_handler._state = multiprocessing.pool.TERMINATE

        # Ждём пока воркеры обработают последние задачи и завершатся
        for p in self._pool:
            p.join()

        # Терминируем весь пул как следует.
        # Нельзя вызывать только этот метод, потому что он завершает
        # воркеры принудительно.
        self.terminate()

        # Ждём полного останова пула
        self.join()

    def _maintain_pool(self):
        # Отключаю восстановление упавших воркеров, потому что это может
        # привести к опасным гонками, из-за того что форкнется "грязный"
        # процесс (с сокетами, файлами и т.п.).
        pass

    def workers_count(self):
        return len(self._pool)
