#!/bin/bash
set -e

RUN_TESTS_BIN="passport/backend/qa/autotests/run_tests.sh"
RUN_TESTS_ARGS="--no-allure-server"
API_BIN_PATH="passport/backend/api/bin"
AUTOTESTS_PORT=10361
AUTOTESTS_HOSTNAME="autotests.dev.passport.yandex.ru"
AUTOTESTS_ENV="development"
CONTAINER_REPO="passport-api-dev-run"
TAG_PREFIX="passport-autotests-api"

say() {
    echo -e "\033[0;36m[AUTOTESTS]\033[0m $@"
}

say_err() {
    echo -e "\033[0;31m[AUTOTESTS: CRITICAL] $@\033[0m"
}

say_note() {
    echo -e "\033[0;33m[AUTOTESTS: NOTICE]\033[0m $@"
}


usage() {
    if [ "$1" ]; then
        echo $1
    fi
    echo "Usage: passport-autotests-api [--rebuild-api|--rebuild-all] [ya make args]"
    echo "  Run passport autotests"
    echo "  --rebuild-api: force rebuild only api bin"
    echo "  --rebuild-all: force rebuild container including base image (e.g. secrets/grants update required)"
    echo
    echo "Usage: passport-autotests-api --attach"
    echo "  Attach to a currently running autotests container for current revision"
    echo
    echo "Usage: passport-autotests-api --logs ['docker logs' options]"
    echo "  Show logs of a currently running autotests container"
    echo "  for list of options help run 'docker logs --help'"
    echo
    echo "Usage: passport-autotests-api --console"
    echo "  Open console inside a currently running autotests container for current revision"
    exit 255
}

REBUILD_API_ARG=0
REBUILD_ALL_ARG=0
ACTION='run'
parse_args() {
    while [ $# -gt 0 ]; do
        case "$1" in
            --help)
                usage
                ;;
            --attach)
                ACTION=attach
                if [ $# -gt 1 ]; then
                    usage "--attach doesn't require args"
                fi
                ;;
            --logs)
                ACTION=logs
                shift
                DOCKER_LOGS_ARGS="$@"
                break
                ;;
            --console|--shell)
                ACTION=console
                if [ $# -gt 1 ]; then
                    usage "--console doesn't require args"
                fi
                ;;
            --rebuild-api)
                REBUILD_API_ARG=1
                ;;
            --rebuild-all)
                REBUILD_API_ARG=1
                REBUILD_ALL_ARG=1
                ;;
            *)
                YA_MAKE_ARGS="$@"
                break
                ;;
        esac
        shift
    done
}


parse_args $@

function get_rev_and_dir() {
    set +e
    REV=$(arc rev-parse HEAD 2>&1)
    REV=${REV: -8}
    VCS_ROOT=$(arc rev-parse --show-toplevel 2>&1)
    if [ $? -ne 0 ]; then
        REV=$(svn info . --show-item revision 2>&1)
        VCS_ROOT=$(svn info . --show-item wc-root --no-newline 2>&1)
        if [ $? -ne 0 ]; then
            say_err "Cannot determine current revision and repo path. Please stay under arc/svn root."
            exit 255
        fi
    fi
    set -e
    CONTAINER_TAG="${TAG_PREFIX}-${REV}"
    CONTAINER_NAME="${CONTAINER_REPO}:${CONTAINER_TAG}"
}


build_api() {
    say "Building passport API"
    (cd "${VCS_ROOT}/${API_BIN_PATH}" && ya make)
}


build_container() {
    BUILD_ARGS=""
    if [ $REBUILD_ALL_ARG -eq 1 ]; then
        BUILD_ARGS="-r"
        say "Totally rebuilding docker container"
    else
        say "Building docker container"
    fi
    passport-docker-api -t "$CONTAINER_TAG" -d -f "${VCS_ROOT}/${API_BIN_PATH}/passport-api" -e "$AUTOTESTS_ENV" -h "$AUTOTESTS_HOSTNAME" -p "$AUTOTESTS_PORT" $BUILD_ARGS bin
    say "Docker container started."
    say_note "Use \"passport-autotests-api --attach\" if you want to attach to test docker, for example, from another console"
 }


find_container_id() {
    CONTAINER_ID=$(docker ps | grep -w "$CONTAINER_NAME" | tail -1 | awk '{print $1}')
}

prune_old_containers() {
    # Clean all stopped containers with autotests prefix
    docker container ls --all --filter status=exited | grep -w "${CONTAINER_REPO}:${TAG_PREFIX}" | while read line; do
        say $(echo "$line" | awk '{ print "Removing old stopped container "$1" "$2}')
        docker container rm $(echo $line | awk '{ print $1 }') > /dev/null
    done
    # Clean all running containers with autotests prefix except current
    docker container ps | grep -w "${CONTAINER_REPO}:${TAG_PREFIX}" | grep -v "$CONTAINER_NAME" | while read line; do
        say $(echo "$line" | awk '{ print "Removing old running container from another revision "$1" "$2}')
        docker kill $(echo $line | awk '{ print $1 }') > /dev/null
        docker container rm $(echo $line | awk '{ print $1 }') > /dev/null
    done
    # Delete current running container if rebuild requested
    if [ $REBUILD_API_ARG -eq 1 ]; then
        find_container_id
        if [ "$CONTAINER_ID" ]; then
            say "Removing current running container: $CONTAINER_ID due to rebuild task"
            docker kill "$CONTAINER_ID" > /dev/null
            docker container rm "$CONTAINER_ID" > /dev/null
        fi
    fi
}


start_container() {
    docker container start "$CONTAINER_ID"
}


wait_for_api() {
    if [ -z "$CONTAINER_ID" ]; then
        say_err "Container seems to have stopped unexpectedly."
        exit 255
    fi
    say "Waiting for API to initialize:"
    echo "-----------------logs-----------------------"
    # Аргумент since нужен чтобы кильнуть тупой докер, не реагирующий на sigpipe в режиме -f
    docker logs -n 20 -f "$CONTAINER_ID" | while read line; do
        echo "$line"
        if [[ $line =~ Running.*:$AUTOTESTS_PORT ]]; then
            exit
        fi
    done&
    WAITER_PID=$!
    while true; do
        ps -p $WAITER_PID > /dev/null 2>&1 || {
            jobs -p | while read _pid; do
                kill $_pid > /dev/null 2>&1 || true;
            done
            break
        }
        sleep 0.1
    done
    find_container_id
    if [ -z "$CONTAINER_ID" ]; then
        say_err "Container seems to have stopped unexpectedly."
        exit 255
    fi
    echo "--------------------------------------------"
    say "API has initialized"
}

run_build_tasks() {
    build_api
    build_container
    find_container_id
    wait_for_api
}


get_rev_and_dir

if [ "$ACTION" = "run" ]; then
    prune_old_containers

    if [ $REBUILD_API_ARG -eq 1 -o $REBUILD_ALL_ARG -eq 1 ]; then
        say "Starting rebuild tasks"
        run_build_tasks
        find_container_id
    else
        find_container_id
        if [ "$CONTAINER_ID" ]; then
            say "Found running container for rev $REV: $CONTAINER_ID"
        else
            find_container_id --all
            if [ "$CONTAINER_ID" ]; then
                say "Found stopped container for rev $REV: $CONTAINER_ID. Starting"
                start_container
            else
                say "Container for rev $REV not found. Preparing"
                run_build_tasks
                find_container_id
            fi
        fi
    fi
    say "Starting passport autotests"
    YTEST_PASSPORT_API_URL="http://127.0.0.1:${AUTOTESTS_PORT}/" \
    YTEST_PASSPORT_API_USE_PROXY= \
    YTEST_ENV="$AUTOTESTS_ENV" \
    "${VCS_ROOT}/${RUN_TESTS_BIN}" $RUN_TESTS_ARGS $YA_MAKE_ARGS
    say "You may open http://allure-passport-autotests.${USER}-dev.passport.yandex.net" to view allure report
elif [ "$ACTION" = "attach" ]; then
    find_container_id
    if [ "$CONTAINER_ID" ]; then
        say "Found container for rev $REV: $CONTAINER_ID. Showing last logs"
        docker logs -n 20 "$CONTAINER_ID"
        say_note "Use Ctrl+[ to detach"
        docker attach --detach-keys 'ctrl-[' "$CONTAINER_ID"
    else
        say_err "Running container for rev $REV not found"
    fi
elif [ "$ACTION" = "logs" ]; then
    find_container_id
    if [ "$CONTAINER_ID" ]; then
        docker logs $DOCKER_LOGS_ARGS "$CONTAINER_ID"
    else
        say_err "Running container for rev $REV not found"
    fi
elif [ "$ACTION" = "console" ]; then
    find_container_id
    if [ "$CONTAINER_ID" ]; then
        docker exec -it "$CONTAINER_ID" /bin/bash
    else
        say_err "Running container for rev $REV not found"
    fi
fi
