#!/bin/bash
set -e

BASE_DIR="$(dirname $(realpath $0))/docker/api"

BASE_IMAGE="passport-api-dev-base:1"
RUN_IMAGE_NAME="passport-api-dev-run"

for executable in rsync docker; do
    which $executable > /dev/null || {
        echo "Please install $executable first"
        exit 1
    }
done

usage() {
    if [ "$1" ]; then
        echo $1
    fi
    echo "Usage: passport-docker-api [-r] [-s|-d] [-p port=4000] [-e yenv_type] [-h hostname] [-t TAG] <-f PATH> <bin>"
    echo "Usage: passport-docker-api [-r] [-s|-d] [-p port=4000] [-e yenv_type] [-h hostname] [-t TAG] <-f PATH> <package>"
    echo "Usage: passport-docker-api [-r] [-s|-d] [-p port=4000] [-e yenv_type] [-h hostname] [-t TAG] <base>"
    echo "  -r: force rebuilding of containers"
    echo "  -d: run conatiner in detached mode"
    echo "  -s: start bash inside container instead of API"
    echo "  -p: start api on specified port"
    echo "  -f: specify package/binary path"
    echo "  -t: set specific image tag, ${RUN_IMAGE_NAME}:TAG, otherwise ${RUN_IMAGE_NAME}:bin/package will be used"
    echo "  -h: set specific YENV_TYPE"
    echo "  -h: set specific hostname for a container"
    echo "  bin: run from binary"
    echo "  package: install deb package and run from it"
    echo "  base: run from base image"
    exit 255
}

START_SHELL_ARG=0
FORCE_REBUILD_ARG=0
DETACH_ARG=0
API_PORT=4000
RUN_IMAGE_TAG=""
FILE_PATH=""
SET_YENV_TYPE=""
SET_HOSTNAME=""
RUN_TYPE=""
function get_next_arg() {
    if [ $# -lt 2 ]; then
        usage "parameter required"
    fi
    if [ "$2" == -* ]; then
        usage "parameter required"
    fi
    echo $2
}
parse_args() {
    if [ "$1" = "--help" ]; then usage; fi
    while [ $# -gt 1 ]; do
        case "$1" in
            --help)
                usage
                ;;
            -r)
                FORCE_REBUILD_ARG=1
                ;;
            -d)
                DETACH_ARG=1
                ;;
            -s)
                START_SHELL_ARG=1
                ;;
            -f)
                FILE_PATH="$(get_next_arg $@)"
                shift
                ;;
            -t)
                RUN_IMAGE_TAG="$(get_next_arg $@)"
                shift
                ;;
            -e)
                SET_YENV_TYPE="$(get_next_arg $@)"
                shift
                ;;
            -h)
                SET_HOSTNAME="$(get_next_arg $@)"
                shift
                ;;
            -p)
                API_PORT="$(get_next_arg $@)"
                shift
                ;;
            *)
                usage "unparsable arg $1"
                ;;
        esac
        shift
    done
    if [ $# -ne 1 ]; then
        usage "run type is required as final arg"
    fi
    if ! [[ "$1" =~ ^(bin|package|base)$ ]]; then
        usage "unknown run type '$1'"
    fi
    if [ $START_SHELL_ARG -eq 1 -a $DETACH_ARG -eq 1 ]; then
        usage "-d conflicts with -s"
    fi
    if [[ "$1" =~ ^(bin|package)$ ]]; then
        if [ -z "$FILE_PATH" ]; then
            usage "$1 requires -f"
        fi
    else
        if [ "$FILE_PATH" ]; then
            usage "-f is excess for $1"
        fi
        if [ "$RUN_IMAGE_TAG" ]; then
            usage "-t is excess for $1"
        fi
    fi
    RUN_TYPE=$1
    RUN_IMAGE_TAG=${RUN_IMAGE_TAG:-$RUN_TYPE}
}


parse_args $@


build() {
    ssh-add -l > /dev/null || {
       echo "Please setup ssh agent forwarding first"
       echo "e.g. 'ssh-agent -s; ssh-add'"
       echo "ssh-add -l should display your key(s)"
       exit 1
    }
    IMAGE_TAG="$1"
    shift
    BUILD_PATH="$1"
    shift
    BUILD_ARGS=""
    if [ $FORCE_REBUILD_ARG -eq 1 ]; then
        BUILD_ARGS="--no-cache"
    fi

    DOCKER_BUILDKIT=1 docker build \
        --ssh default \
        --network host \
        --progress plain \
        --build-arg EXT_USER=${EXT_USER:-${USER}} \
        -t "$IMAGE_TAG" \
        $BUILD_ARGS \
        $@ "$BUILD_PATH"
}


# Build base image
echo "--- Building base image $BASE_IMAGE"
build $BASE_IMAGE "$BASE_DIR/base"


# Build run image
if [ "$RUN_TYPE" = "base" ]; then
    RUN_IMAGE="$BASE_IMAGE"
else
    RUN_IMAGE="$RUN_IMAGE_NAME:$RUN_IMAGE_TAG"
    echo "--- Building run image '$RUN_IMAGE' ($RUN_TYPE)"
    BUILD_DIR="$BASE_DIR/$RUN_TYPE"
    rsync -v -P -a --copy-links "$FILE_PATH" "$BUILD_DIR/tmp/out.swp"
    build $RUN_IMAGE "$BUILD_DIR"
fi


# Run docker
if [ $DETACH_ARG -eq 1 ]; then
    RUN_ARGS="-itd"
    echo "--- Starting in detached mode:"
    echo "Use docker attach --detach-keys='ctrl-[' <id> to attach."
    echo "Use Ctrl+[ to detach."
else
    RUN_ARGS="-it"
    echo "--- Starting interactively:"
    echo "Use Ctrl+[ to detach."
    echo "Use docker attach <id> to re-attach."
fi
if [ "$SET_HOSTNAME" ]; then
    RUN_ARGS="$RUN_ARGS --hostname $SET_HOSTNAME --domainname $SET_HOSTNAME"
fi
RUN_ARGS="$RUN_ARGS $RUN_IMAGE"
if [ $START_SHELL_ARG -eq 1 ]; then
    RUN_ARGS="$RUN_ARGS bash"
fi
docker run \
    -e API_PORT=${API_PORT} \
    -e YENV_TYPE=${SET_YENV_TYPE} \
    --network host \
    --mount type=bind,source=/var/cache/geobase,target=/var/cache/geobase \
    --mount type=bind,source=/var/cache/passport-libs-checker,target=/var/cache/passport-libs-checker \
    --detach-keys="ctrl-[" \
    $RUN_ARGS
