package pgadapter

import (
	"context"
	"database/sql"
	"database/sql/driver"
	"encoding/json"
	"fmt"

	"github.com/jackc/pgconn"
	"github.com/jackc/pgerrcode"
	_ "github.com/jackc/pgx/v4"
	hasql "golang.yandex/hasql/sqlx"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/passport/backend/federal_config_api/internal/core/interfaces"
	"a.yandex-team.ru/passport/backend/federal_config_api/internal/core/models"
)

type LookupType int

const (
	GetByConfigID LookupType = iota
	GetByEntityID
	GetByDomainID
)

func PgErrorCode(err error) (string, bool) {
	var pgErr *pgconn.PgError
	if xerrors.As(err, &pgErr) {
		return pgErr.Code, true
	}

	return "", false
}

type JSONAggDomainIDs []uint64

func (jadi *JSONAggDomainIDs) Scan(val interface{}) error {
	switch v := val.(type) {
	case []byte:
		return json.Unmarshal(v, &jadi)
	case string:
		return json.Unmarshal([]byte(v), &jadi)
	default:
		return fmt.Errorf("unsupported type %T", v)
	}
}

func (jadi *JSONAggDomainIDs) Value() (driver.Value, error) {
	return json.Marshal(jadi)
}

type JSONConfigAttrs models.ConfigBody

func (ca *JSONConfigAttrs) Scan(val interface{}) error {
	switch v := val.(type) {
	case []byte:
		return json.Unmarshal(v, &ca)
	case string:
		return json.Unmarshal([]byte(v), &ca)
	default:
		return fmt.Errorf("unsupported type %T", v)
	}
}

func (ca *JSONConfigAttrs) Value() (driver.Value, error) {
	return json.Marshal(ca)
}

func (pga *postgresqlAdapter) txWrap(ctx context.Context, opts sql.TxOptions, fn func(tx *sql.Tx) error) error {
	var node hasql.Node
	if opts.ReadOnly {
		node = pga.cluster.Standby()
		if node == nil {
			node = pga.cluster.Primary()
		}
	} else {
		node = pga.cluster.Primary()
	}
	tx, err := node.DBx().BeginTx(ctx, &opts)
	if err != nil {
		return xerrors.Errorf("failed to begin tx: %w", err)
	}
	defer tx.Rollback()

	err = fn(tx)
	if err == nil {
		if err := tx.Commit(); err != nil {
			return xerrors.Errorf("failed to commit tx: %w", err)
		}
	} else {
		return xerrors.Errorf("failed in tx function: %w", err)
	}
	return nil
}

func (pga *postgresqlAdapter) txExec(ctx context.Context, tx *sql.Tx, sql string, arguments ...interface{}) (sql.Result, error) {
	res, err := tx.ExecContext(
		ctx,
		sql,
		arguments...,
	)
	if err != nil {
		pgErrorCode, _ := PgErrorCode(err)
		if pgErrorCode == pgerrcode.UniqueViolation /* or just == "23505" */ {
			return res, xerrors.Errorf("failed tx exec: %w", interfaces.ErrDuplicateEntry)
		}
		return res, xerrors.Errorf("failed tx exec: %w", err)
	}
	return res, nil
}

func (pga *postgresqlAdapter) txGetConfigByID(ctx context.Context, tx *sql.Tx, lookupType LookupType, id string) (uint64, string, string, []uint64, models.ConfigBody, error) {
	var (
		configID    uint64
		entityID    string
		namespace   string
		whereClause string
		domainIDs   JSONAggDomainIDs
		configBody  JSONConfigAttrs
	)
	switch lookupType {
	case GetByConfigID:
		whereClause = "where c.config_id = $1"
	case GetByDomainID:
		whereClause = "where c.config_id = (select config_id from domain_id_to_config_id where domain_id = $1)"
	case GetByEntityID:
		whereClause = "where e.entity_id = $1"
	default:
		panic(fmt.Errorf("unsupported lookup type: %d", lookupType))
	}

	configBody.Enabled = true
	err := tx.QueryRowContext(
		ctx,
		fmt.Sprintf(
			"select"+
				"   c.config_id, "+
				"	coalesce(e.entity_id, ''), "+
				"   n.namespace,"+
				"	json_agg(d.domain_id), "+
				"	c.attributes "+
				"from configs c "+
				"left join entity_id_to_config_id e "+
				"	on c.config_id = e.config_id "+
				"left join domain_id_to_config_id d "+
				"	on c.config_id = d.config_id "+
				"left join namespace_to_config_id n"+
				"   on c.config_id = n.config_id "+
				"%s "+ // сюда вставляется условие where
				"group by c.config_id, e.entity_id, n.namespace, c.attributes",
			whereClause,
		),
		id,
	).Scan(&configID, &entityID, &namespace, &domainIDs, &configBody)
	if err == sql.ErrNoRows {
		return configID, entityID, namespace, domainIDs, models.ConfigBody(configBody), xerrors.Errorf("failed to get config: %w", interfaces.ErrNotFound)
	}
	if err != nil {
		return configID, entityID, namespace, domainIDs, models.ConfigBody(configBody), xerrors.Errorf("failed to get config %d: %w", configID, err)
	}
	return configID, entityID, namespace, domainIDs, models.ConfigBody(configBody), nil
}

func (pga *postgresqlAdapter) txListConfigIDs(ctx context.Context, tx *sql.Tx, namespace string, startConfigID uint64, limit uint64) ([]uint64, error) {
	var configIDs []uint64
	rows, err := tx.QueryContext(
		ctx,
		"select c.config_id "+
			"from namespace_to_config_id n "+
			"left join configs c "+
			"on n.config_id = c.config_id "+
			"where"+
			"  n.namespace = $1 and"+
			"  c.config_id >= $2 "+
			"order by c.config_id asc "+
			"limit $3",
		namespace,
		startConfigID,
		limit,
	)
	if err != nil {
		return configIDs, xerrors.Errorf("failed to list configs: %w", err)
	}
	defer rows.Close()

	for rows.Next() {
		var configID uint64
		err := rows.Scan(&configID)
		if err != nil {
			return configIDs, xerrors.Errorf("failed to list configs: %w", err)
		}
		configIDs = append(configIDs, configID)
	}
	return configIDs, nil
}

func (pga *postgresqlAdapter) txAddNamespaceToConfigID(ctx context.Context, tx *sql.Tx, mapping NamespaceToConfigID) error {
	_, err := pga.txExec(
		ctx,
		tx,
		"insert into namespace_to_config_id (namespace, config_id) values ($1, $2)",
		mapping.Namespace,
		mapping.ConfigID,
	)
	if err != nil {
		return xerrors.Errorf("failed to add namespace: %w", err)
	}
	return nil
}

func (pga *postgresqlAdapter) txAddEntityIDToConfigID(ctx context.Context, tx *sql.Tx, mapping EntityIDToConfigID) error {
	_, err := pga.txExec(
		ctx,
		tx,
		"insert into entity_id_to_config_id (entity_id, config_id) values ($1, $2)",
		mapping.EntityID,
		mapping.ConfigID,
	)
	if err != nil {
		return xerrors.Errorf("failed to add entity_id: %w", err)
	}
	return nil
}

func (pga *postgresqlAdapter) txAddDomainIDToConfigID(ctx context.Context, tx *sql.Tx, mapping DomainIDToConfigID) error {
	_, err := pga.txExec(
		ctx,
		tx,
		"insert into domain_id_to_config_id (domain_id, config_id) values ($1, $2)",
		mapping.DomainID,
		mapping.ConfigID,
	)
	if err != nil {
		return xerrors.Errorf("failed to add domain_id: %w", err)
	}
	return nil
}

func (pga *postgresqlAdapter) txAddConfig(ctx context.Context, tx *sql.Tx, attrs Attributes) (uint64, error) {
	var configID uint64

	attrsJSON, err := json.Marshal(attrs)
	if err != nil {
		return configID, err
	}

	err = tx.QueryRowContext(
		ctx,
		"insert into configs (attributes) values ($1) returning config_id",
		string(attrsJSON),
	).Scan(&configID)
	if err != nil {
		pgErrorCode, _ := PgErrorCode(err)
		if pgErrorCode == pgerrcode.UniqueViolation /* or just == "23505" */ {
			return configID, xerrors.Errorf("failed to add config: %w", interfaces.ErrDuplicateEntry)
		}
		return configID, xerrors.Errorf("failed to add config: %w", err)
	}
	return configID, nil
}

func (pga *postgresqlAdapter) txEditConfig(ctx context.Context, tx *sql.Tx, configID uint64, attrs Attributes) error {
	attrsJSON, err := json.Marshal(attrs)
	if err != nil {
		return err
	}

	res, err := pga.txExec(
		ctx,
		tx,
		"update configs set attributes = $1 where config_id = $2",
		string(attrsJSON),
		configID,
	)
	if err != nil {
		pgErrorCode, _ := PgErrorCode(err)
		if pgErrorCode == pgerrcode.UniqueViolation {
			return xerrors.Errorf("failed to edit config: %w", interfaces.ErrDuplicateEntry)
		}
		return xerrors.Errorf("failed to edit config: %w", err)
	}
	ra, err := res.RowsAffected()
	if err != nil {
		return xerrors.Errorf("failed to edit config: %w", err)
	}
	if ra == 0 {
		return xerrors.Errorf("failed to edit config: %w", interfaces.ErrNotFound)
	}
	return nil
}

func (pga *postgresqlAdapter) txGetNamespaceMapping(ctx context.Context, tx *sql.Tx, configID uint64) (string, error) {
	var namespace string
	err := tx.QueryRowContext(
		ctx,
		"select namespace from namespace_to_config_id where config_id = $1",
		configID,
	).Scan(&namespace)
	if err != nil {
		return namespace, xerrors.Errorf("failed to get namespace mapping: %w", err)
	}
	return namespace, nil
}

func (pga *postgresqlAdapter) txGetEntityIDMapping(ctx context.Context, tx *sql.Tx, configID uint64) (string, error) {
	var entityID string
	err := tx.QueryRowContext(
		ctx,
		"select entity_id from entity_id_to_config_id where config_id = $1",
		configID,
	).Scan(&entityID)
	if err != nil {
		return entityID, xerrors.Errorf("failed to get entity_id mapping: %w", err)
	}
	return entityID, nil
}

func (pga *postgresqlAdapter) txDeleteEntityIDMapping(ctx context.Context, tx *sql.Tx, entityID string, configID uint64) error {
	res, err := pga.txExec(
		ctx,
		tx,
		"delete from entity_id_to_config_id where entity_id = $1 and config_id = $2",
		entityID,
		configID,
	)
	if err != nil {
		return xerrors.Errorf("failed to delete entity_id mapping: %w", err)
	}
	ra, err := res.RowsAffected()
	if err != nil {
		return xerrors.Errorf("failed to delete entity_id mapping: %w", err)
	}
	if ra == 0 {
		return xerrors.Errorf("failed to delete entity_id mapping: %w", interfaces.ErrNotFound)
	}
	return nil
}

func (pga *postgresqlAdapter) txUpdateEntityIDMapping(ctx context.Context, tx *sql.Tx, oldEntityID string, newEntityID string, configID uint64) error {
	res, err := pga.txExec(
		ctx,
		tx,
		"update entity_id_to_config_id set entity_id = $1 where entity_id = $2 and config_id = $3",
		newEntityID,
		oldEntityID,
		configID,
	)
	if err != nil {
		return xerrors.Errorf("failed to update entity_id mapping: %w", err)
	}
	ra, err := res.RowsAffected()
	if err != nil {
		return xerrors.Errorf("failed to update entity_id mapping: %w", err)
	}
	if ra == 0 {
		return xerrors.Errorf("failed to update entity_id mapping: %w", interfaces.ErrNotFound)
	}
	return nil
}

func (pga *postgresqlAdapter) txDeleteDomainIDsMapping(ctx context.Context, tx *sql.Tx, domainIDs []uint64, configID uint64) error {
	res, err := pga.txExec(
		ctx,
		tx,
		"delete from domain_id_to_config_id where domain_id = ANY ($1) and config_id = $2",
		domainIDs,
		configID,
	)
	if err != nil {
		return xerrors.Errorf("failed to delete domain_id mapping: %w", err)
	}
	ra, err := res.RowsAffected()
	if err != nil {
		return xerrors.Errorf("failed to delete domain_id mapping: %w", err)
	}
	if ra == 0 {
		return xerrors.Errorf("domain_id mapping not found: %w", interfaces.ErrNotFound)
	}
	return nil
}

func (pga *postgresqlAdapter) txGetDomainIDMapping(ctx context.Context, tx *sql.Tx, configID uint64) ([]uint64, error) {
	var domainIDs []uint64
	rows, err := tx.QueryContext(
		ctx,
		"select domain_id from domain_id_to_config_id where config_id = $1",
		configID,
	)
	if err != nil {
		return domainIDs, xerrors.Errorf("failed to get domain_id mapping: %w", err)
	}

	defer rows.Close()

	for rows.Next() {
		var domainID uint64
		err := rows.Scan(&domainID)
		if err != nil {
			return domainIDs, xerrors.Errorf("failed to get domain_id mapping: %w", err)
		}
		domainIDs = append(domainIDs, domainID)
	}
	return domainIDs, nil
}

func (pga *postgresqlAdapter) txDeleteConfig(ctx context.Context, tx *sql.Tx, configID uint64) error {
	res, err := pga.txExec(
		ctx,
		tx,
		"delete from configs where config_id = $1",
		configID,
	)
	if err != nil {
		return xerrors.Errorf("failed to delete config: %w", err)
	}
	ra, err := res.RowsAffected()
	if err != nil {
		return err
	}
	if ra == 0 {
		return xerrors.Errorf("failed to get domain_id mapping: %w", interfaces.ErrNotFound)
	}
	return nil
}

func (pga *postgresqlAdapter) txGetNamespaceByConfigID(ctx context.Context, tx *sql.Tx, configID uint64) (string, error) {
	var namespace string
	err := tx.QueryRowContext(
		ctx,
		"select namespace from namespace_to_config_id where config_id = $1 limit 1",
		configID,
	).Scan(&namespace)
	if err != nil {
		return namespace, xerrors.Errorf("failed to get namespace by config_id: %w", err)
	}
	return namespace, nil
}
