package controllers

import (
	"context"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/passport/backend/federal_config_api/internal/core/interfaces"
	"a.yandex-team.ru/passport/backend/federal_config_api/internal/core/models"
)

type controller struct {
	federalConfigAdapter interfaces.FederalConfigAdapter
	pddAdapter           interfaces.PDDAdapter
	logger               log.Logger
}

// compile-time проверка, что интерфейс имплементирован корректно
var _ interfaces.FederalConfigController = (*controller)(nil)

type auth struct {
	authAdapter interfaces.AuthAdapter
}

var _ interfaces.AuthController = (*auth)(nil)

func NewAuthController(authAdapter interfaces.AuthAdapter) *auth {
	return &auth{authAdapter}
}

func (a *auth) HasRole(ctx context.Context, role string) (bool, error) {
	return a.authAdapter.HasRole(ctx, role)
}

func (a *auth) HasRoles(ctx context.Context, roles []string) (bool, error) {
	return a.authAdapter.HasRoles(ctx, roles)
}

func NewConfigController(logger log.Logger, federalConfigAdapter interfaces.FederalConfigAdapter, pddAdapter interfaces.PDDAdapter) *controller {
	return &controller{
		federalConfigAdapter: federalConfigAdapter,
		pddAdapter:           pddAdapter,
		logger:               logger,
	}
}

func (c *controller) checkPDDExists(ctx context.Context, domainIDs []uint64) error {
	found, err := c.pddAdapter.Exists(ctx, domainIDs)
	if err != nil {
		return err
	}
	if !found {
		return interfaces.ErrNotFound
	}
	return nil
}

func (c *controller) Create(ctx context.Context, config models.FederationConfig) (models.FederationConfig, error) {
	if err := c.checkPDDExists(ctx, config.DomainIDs); err != nil {
		return config, err
	}
	configID, err := c.federalConfigAdapter.Create(ctx, config)
	if err != nil {
		return config, err
	}
	config.ConfigID = configID
	return config, nil
}

func (c *controller) GetByConfigID(ctx context.Context, namespace string, configID uint64) (models.FederationConfig, error) {
	federalConfig, err := c.federalConfigAdapter.GetByConfigID(ctx, namespace, configID)
	if err != nil {
		return models.FederationConfig{}, err
	}
	return federalConfig, err
}

func (c *controller) GetByEntityID(ctx context.Context, namespace string, entityID string) (models.FederationConfig, error) {
	federalConfig, err := c.federalConfigAdapter.GetByEntityID(ctx, namespace, entityID)
	if err != nil {
		return models.FederationConfig{}, err
	}
	return federalConfig, err
}

func (c *controller) GetByDomainID(ctx context.Context, namespace string, domainID uint64) (models.FederationConfig, error) {
	federalConfig, err := c.federalConfigAdapter.GetByDomainID(ctx, namespace, domainID)
	if err != nil {
		return models.FederationConfig{}, err
	}
	return federalConfig, err
}

func (c *controller) List(ctx context.Context, namespace string, startConfigID uint64, limit uint64) ([]models.FederationConfig, error) {
	var federalConfigs []models.FederationConfig
	federalConfigs, err := c.federalConfigAdapter.List(ctx, namespace, startConfigID, limit)
	if err != nil {
		return federalConfigs, err
	}
	return federalConfigs, nil
}

func (c *controller) Update(ctx context.Context, namespace string, entityID *string, domainIDs *[]uint64, configID uint64, configBody models.ConfigBody) error {
	if domainIDs != nil {
		if err := c.checkPDDExists(ctx, *domainIDs); err != nil {
			return err
		}
	}
	return c.federalConfigAdapter.Update(ctx, namespace, entityID, domainIDs, configID, configBody)
}

func (c *controller) Delete(ctx context.Context, namespace string, configID uint64) error {
	return c.federalConfigAdapter.Delete(ctx, namespace, configID)
}
