package federalcfgapi

import (
	"context"
	"database/sql/driver"
	"fmt"
	"net/http"
	"net/url"
	"strconv"
	"time"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/passport/backend/federal_config_api/internal/adapters/blackboxadapter"
	"a.yandex-team.ru/passport/backend/federal_config_api/internal/adapters/pgadapter"
	"a.yandex-team.ru/passport/backend/federal_config_api/internal/adapters/tvmauthadapter"
	"a.yandex-team.ru/passport/backend/federal_config_api/internal/core/controllers"
	"a.yandex-team.ru/passport/backend/federal_config_api/internal/core/interfaces"
	"a.yandex-team.ru/passport/backend/federal_config_api/internal/logutils"
	"a.yandex-team.ru/passport/shared/golibs/httpdaemon"
	"a.yandex-team.ru/passport/shared/golibs/httpdaemon/httpdtvm"
	"a.yandex-team.ru/passport/shared/golibs/httpdaemon/middlewares"
	"a.yandex-team.ru/passport/shared/golibs/logger"
	"a.yandex-team.ru/passport/shared/golibs/unistat"
)

type AppLoggers struct {
	accessLog   log.Logger
	handlersLog log.Logger
	configLog   log.Logger
	pgLogger    log.Logger
	bbLogger    log.Logger
}

type FederalConfigAPI struct {
	tvm              tvm.Client
	unistat          stats
	configController interfaces.FederalConfigController
	authController   interfaces.AuthController
	AppLoggers
}

type Database struct {
	DatabaseName string   `json:"dbname"`
	Hosts        []string `json:"hosts"`
	Port         int      `json:"port"`
}

type Config struct {
	Tvm         httpdtvm.TvmConfig `json:"tvm"`
	Database    Database           `json:"db"`
	AccessLog   logger.Config      `json:"access_log"`
	HandlersLog logger.Config      `json:"handlers_log"`
	ConfigLog   logger.Config      `json:"config_log"`
	BlackboxLog logger.Config      `json:"blackbox_log"`
	PostgresLog logger.Config      `json:"postgres_log"`
}

type stats struct {
	requestsByPath *unistat.SignalSet
}

type Factory struct{}

type NamespaceOpType string

const NamespaceRoleOpWrite NamespaceOpType = "write"
const NamespaceRoleOpRead NamespaceOpType = "read"

const (
	emptyString       = "-"
	headerETag        = "ETag"
	headerIfNoneMatch = "If-None-Match"
)

func (f *Factory) CreateLoggers(cfg Config) (AppLoggers, error) {
	var loggers AppLoggers

	accessLog, err := logger.CreateLog(cfg.AccessLog)
	if err != nil {
		return loggers, err
	}

	handlersLog, err := logger.CreateLog(cfg.HandlersLog)
	if err != nil {
		return loggers, err
	}

	configLog, err := logger.CreateLog(cfg.ConfigLog)
	if err != nil {
		return loggers, err
	}

	bbLog, err := logger.CreateLog(cfg.BlackboxLog)
	if err != nil {
		return loggers, err
	}

	pgLog, err := logger.CreateLog(cfg.PostgresLog)
	if err != nil {
		return loggers, err
	}

	loggers.accessLog = log.With(accessLog, log.String("loggerName", "access_log"))
	loggers.handlersLog = log.With(handlersLog, log.String("loggerName", "handlers_log"))
	loggers.configLog = log.With(configLog, log.String("loggerName", "config_log"))
	loggers.bbLogger = log.With(bbLog, log.String("loggerName", "blackbox_log"))
	loggers.pgLogger = log.With(pgLog, log.String("loggerName", "pg_log"))

	return loggers, nil
}

func (f *Factory) NewService(config httpdaemon.ServiceConfig) (httpdaemon.Service, error) {
	var cfg Config
	if err := httpdaemon.ParseServiceConfig(config, &cfg); err != nil {
		return nil, err
	}

	tvmClient, err := httpdtvm.InitTvm(cfg.Tvm)
	if err != nil {
		return nil, err
	}

	loggers, err := f.CreateLoggers(cfg)
	if err != nil {
		panic(err)
	}

	var pgConfig DatabaseConfig
	conns := make(map[string]string, len(cfg.Database.Hosts))
	for _, host := range cfg.Database.Hosts {
		pgConfig.ParseFromEnv()
		pgConfig.DatabaseName = cfg.Database.DatabaseName
		pgConfig.Host = host
		pgConfig.Port = cfg.Database.Port
		conns[host] = pgConfig.ToConnString()
	}

	// FIXME надо унести отсюда, иначе приложение не поднимается без доступной бд
	configAdapter, err := pgadapter.NewPostgresqlAdapter(
		context.Background(),
		conns,
		loggers.configLog,
	)
	if err != nil {
		panic(err)
	}
	pddAdapter, err := blackboxadapter.NewBlackboxAdapter(loggers.bbLogger)
	if err != nil {
		panic(err)
	}

	configController := controllers.NewConfigController(loggers.configLog, configAdapter, pddAdapter)

	authAdapter, err := tvmauthadapter.NewTVMAuthAdapter(tvmClient)
	if err != nil {
		panic(err)
	}
	authController := controllers.NewAuthController(authAdapter)

	res := &FederalConfigAPI{
		tvm: tvmClient,
		unistat: stats{
			requestsByPath: unistat.DefaultChunk.CreateSignalSet("in.requests."),
		},
		configController: configController,
		authController:   authController,
	}
	res.accessLog = loggers.accessLog
	res.handlersLog = loggers.handlersLog
	return res, nil
}

func (t *FederalConfigAPI) AddHandlers(e *echo.Echo) {
	e.Pre(
		t.middlewarePopulateRequest(),
		t.middlewareAccessLog(),
	)
	e.Use(
		t.middlewareTvmServiceTicketAuthentication(),
	)

	e.GET(
		"/ping",
		t.HandlePing(),
	)

	e.POST("/1/config/", t.HandleAddConfig())
	e.PUT("/1/config/by_config_id/:config_id/", t.HandleEditConfig())
	e.GET("/1/config/", t.HandleListConfig())
	e.GET("/1/config/by_config_id/:config_id/", t.HandleGetByConfigIDConfig())
	e.GET("/1/config/by_domain_id/:domain_id/", t.HandleGetByDomainIDConfig())
	e.GET("/1/config/by_entity_id/:entity_id/", t.HandleGetByEntityIDConfig())
	e.DELETE("/1/config/by_config_id/:config_id/", t.HandleDeleteConfig())
}

func (t *FederalConfigAPI) HandlePing() echo.HandlerFunc {
	return func(c echo.Context) error {
		l := logutils.AddCommonFromContext(c.Request().Context(), t.handlersLog)
		l.Debugf("Ping: service is up")
		return c.String(http.StatusOK, "")
	}
}

func strToInt(str string) uint64 {
	u, _ := strconv.ParseUint(str, 0, 64)
	// FIXME ошибка замалчивается
	return u
}

func (t *FederalConfigAPI) middlewareAccessLog() echo.MiddlewareFunc {
	nullable := func(val string) string {
		if len(val) == 0 {
			return emptyString
		}
		return val
	}

	return func(next echo.HandlerFunc) echo.HandlerFunc {
		return func(c echo.Context) (err error) {
			c.Response().After(func() {
				startTime := middlewares.ContextStartInstant(c.Request().Context())
				statusCode := c.Response().Status
				msg := fmt.Sprintf(
					"%s\t%s\t%s\t%.1fms\t%d\t%d\t%s\t%s\t%s",
					c.RealIP(),
					c.Request().Method,
					c.Request().URL.Path,
					float64(time.Since(startTime).Microseconds())/1000,
					statusCode,
					c.Response().Size,
					nullable(c.Request().Header.Get(headerIfNoneMatch)),
					nullable(c.Request().URL.RawQuery),
					nullable(c.Response().Header().Get(headerETag)),
				)
				l := logutils.AddCommonFromContext(c.Request().Context(), t.accessLog)
				switch statusCode / 100 {
				case 4:
					l.Warn(msg)
				case 5:
					l.Error(msg)
				default:
					l.Debug(msg)
				}
			})

			return next(c)
		}
	}
}

func (t FederalConfigAPI) middlewarePopulateRequest() echo.MiddlewareFunc {
	// Мидлварь, складывающая в контекст общие для запроса штуки,
	// будь то client_ip или возможно еще чего-то
	return func(next echo.HandlerFunc) echo.HandlerFunc {
		return func(c echo.Context) error {
			ctxRequest := logutils.CtxRequest{
				ClientIP: c.Request().Header.Get("X-Forwarded-For-Y"),
			}
			queryParams, err := url.ParseQuery(c.Request().URL.RawQuery)
			if err == nil {
				ctxRequest.ExternalRequestID = queryParams.Get("request_id")
				if ctxRequest.ExternalRequestID == "" {
					// параметр от наших функциональных тестов
					ctxRequest.ExternalRequestID = queryParams.Get("test_req_id")
				}
			}

			c.SetRequest(c.Request().WithContext(
				logutils.AddRequestToContext(
					c.Request().Context(),
					&ctxRequest,
				),
			))
			return next(c)
		}
	}
}

func (t *FederalConfigAPI) middlewareTvmServiceTicketAuthentication() echo.MiddlewareFunc {
	return func(next echo.HandlerFunc) echo.HandlerFunc {
		return func(c echo.Context) error {
			st := c.Request().Header.Get("X-Ya-Service-Ticket")
			if st == "" {
				return next(c)
			}

			ticket, err := t.tvm.CheckServiceTicket(c.Request().Context(), st)
			if err != nil {
				logutils.AddCommonFromContext(
					c.Request().Context(),
					t.handlersLog,
				).Errorf("tvm error: %s", err.Error())
				return c.String(http.StatusForbidden, err.Error())
			}

			c.SetRequest(c.Request().WithContext(
				tvm.WithServiceTicket(c.Request().Context(), ticket),
			))

			return next(c)
		}
	}
}

func (t *FederalConfigAPI) assertHasRole(c echo.Context, namespace string, op NamespaceOpType) error {
	_, err := t.authController.HasRole(
		c.Request().Context(),
		fmt.Sprintf("/role/service/auth_type/without_user/namespace/%s/access_type/%s/", namespace, op),
	)
	if err != nil {
		logutils.AddCommonFromContext(
			c.Request().Context(),
			t.handlersLog,
		).Warnf("error checking roles: %s", err.Error())
		return err
	}
	return nil
}

func (t *FederalConfigAPI) GenericHandleError(c echo.Context, err error) error {
	if err != nil {
		l := logutils.AddCommonFromContext(c.Request().Context(), t.handlersLog)
		msgForLog := fmt.Sprintf("Responded with error: `%s`", err.Error())
		if xerrors.Is(err, interfaces.ErrNotFound) {
			l.Infof(msgForLog)
			return c.String(http.StatusNotFound, err.Error())
		} else if xerrors.Is(err, interfaces.ErrDuplicateEntry) {
			l.Infof(msgForLog)
			return c.String(http.StatusConflict, err.Error())
		} else if xerrors.Is(err, driver.ErrBadConn) {
			l.Warnf(msgForLog)
			return c.String(http.StatusServiceUnavailable, err.Error())
		} else {
			l.Errorf(msgForLog)
			return c.String(http.StatusInternalServerError, err.Error())
		}
	}
	return nil
}

func (t *FederalConfigAPI) AssertRoleOnNamespace(op NamespaceOpType, fn func(echo.Context) error) func(ctx echo.Context) error {
	return func(c echo.Context) error {
		var namespace string
		if err := c.Request().ParseForm(); err != nil {
			return c.String(http.StatusInternalServerError, fmt.Sprintf("error parsing form: %s", err.Error()))
		}
		queryParams := c.Request().URL.Query()
		if queryParams.Has("namespace") {
			namespace = queryParams.Get("namespace")
		} else {
			return c.String(http.StatusBadRequest, "namespace is required")
		}

		if err := t.assertHasRole(c, namespace, op); err != nil {
			return c.String(http.StatusForbidden, err.Error())
		}

		return fn(c)
	}
}
