# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import logging

import redis
import ujson
from django.conf import settings

from passport_grants_configurator.apps.core.models import (
    Network,
    PreviousResolve,
)
from passport_grants_configurator.apps.core.utils import is_decreased_by_trypo

logger = logging.getLogger(__name__)
redis_connection = redis.Redis(**settings.REDIS_CONFIG)


def redis_retry(count):
    """
    Фабрика декораторов, которые обрабатывают ошибки получения данных из Redis
    При ошибке связи с Redis, заданное количество раз будет выполняться повтор
    """
    def decorator(function):
        def wrapper(*args, **kwargs):
            for i in xrange(count):
                try:
                    return function(*args, **kwargs)
                except redis.RedisError as ex:
                    logger.debug(
                        'Failed [%d/%d] redis-call of %s function with exception %s',
                        i + 1,
                        count,
                        function.__name__,
                        ex,
                    )
            else:
                logger.error('No retries left for function %s', function.__name__)
                raise ex
        return wrapper
    return decorator


def make_default_redis_key(*args, **kwargs):
    """Общая функция форматирования ключей Redis для кеширования вызовов функций"""
    # ключевой параметр _cache_request_prefix не должен дублироваться в ключе
    cache_request_prefix = kwargs.pop('_cache_request_prefix', 'global')
    return '%s:%s:%s%s' % (
        settings.REDIS_KEY_PREFIX,
        cache_request_prefix,
        args,
        kwargs,
    )


def _network_key_formatter(key, network_prefix=None):
    """Функция форматирования ключей Redis при поиске закешированных сетей по строке"""
    network_prefix = network_prefix or settings.REDIS_NETWORK_PREFIX
    return '%s:%s:%s' % (
        settings.REDIS_KEY_PREFIX,
        network_prefix,
        key,
    )


def network_key_getter(string, *args, **kwargs):
    """Функция форматирования ключей для редиса для кеширования сетей"""
    return _network_key_formatter(string)


def macro_key_getter(string, *args, **kwargs):
    """Ключ с потомками макроса для проверки на процент изменения состава"""
    return _network_key_formatter(string, network_prefix=settings.REDIS_MACRO_CHILDREN_PREFIX)


def c_group_key_getter(string, *args, **kwargs):
    """Ключ с потомками кондукторной группы для проверки на процент изменения состава"""
    return _network_key_formatter(string, network_prefix=settings.REDIS_C_GROUP_CHILDREN_PREFIX)


@redis_retry(settings.REDIS_RETRIES)
def cache_structure(formatted_key, structure, timeout=None):
    """Функция, отправляющая данные из питона в редис"""
    if settings.REDIS_CACHE_ENABLED:
        timeout = timeout or settings.REDIS_CACHE_TIMEOUT
        redis_connection.setex(formatted_key, ujson.dumps(structure), timeout)

    return structure


@redis_retry(settings.REDIS_RETRIES)
def get_cached_structure(formatted_key):
    """Функция, получающая данные из редиса в питон"""
    if not settings.REDIS_CACHE_ENABLED:
        return

    structure = redis_connection.get(formatted_key)
    if structure:
        return ujson.loads(structure)


def cache_network(key, meta_network):
    """Функция, отправляющая данные о сетях из питона в редис"""
    formatted_key = _network_key_formatter(key)
    return cache_structure(formatted_key=formatted_key, structure=meta_network)


def get_cached_network(key):
    """Функция, получающия данные о сетях из редиса в питон"""
    formatted_key = _network_key_formatter(key)
    return get_cached_structure(formatted_key=formatted_key)


def get_caching_wrapper(prefix=None, key_getter=None, timeout=None, recache=False, collision_resolver=None):
    """
    Строит декоратор, кеширующий вызовы функций с вниманием к переданным аргументам
    """
    key_getter = key_getter or make_default_redis_key

    def decorator(function):
        # Это множество будет в одном экземпляре для каждой декорированной функции
        # Множество никогда не очищается и существует до конца жизни python-процесса
        used_keys = set()

        def wrapper(*args, **kwargs):
            formatted_key = key_getter(*args, _cache_request_prefix=prefix or function.__name__, **kwargs)
            cached = get_cached_structure(formatted_key=formatted_key)

            # TODO: 'recache' используется только в вызове задачи hourly для раскрытия состава
            # кондукторных групп и макросов
            ignore_cache = kwargs.pop('recache', False) or recache

            if cached and not ignore_cache:
                return cached

            result = function(*args, **kwargs)
            # Результата может и не быть в случае UnknownNetworkTypeError или HostResolvingFailed
            # Тогда и резолвить нечего
            if collision_resolver and result:
                if cached and formatted_key in used_keys:
                    result = collision_resolver(cached, result)

                used_keys.add(formatted_key)

            # Логика по выявлению похудевших макросов и групп
            if key_getter in [macro_key_getter, c_group_key_getter] and args:
                try:
                    # Сети может не быть, если мы обновляем сети не задачей hourly
                    network = Network.objects.get(string=args[0])
                except Network.DoesNotExist:
                    network = None

                # Если результата нет - отдельный случай, в нужных случаях обрабатывается NotFoundError
                if network and network.type in [Network.FIREWALL, Network.CONDUCTOR] and result:
                    prev_resolve, created = PreviousResolve.objects.get_or_create(network=network)
                    if created:
                        prev_resolve.children = ujson.dumps(result)
                        prev_resolve.save()
                    else:
                        prev_children = ujson.loads(prev_resolve.children)
                        decreased_by_trypo = is_decreased_by_trypo(prev_children, result)

                        if len(prev_children) * settings.ANOREXIA_THRESHOLD <= len(result) or decreased_by_trypo:
                            prev_resolve.children = ujson.dumps(result)
                            prev_resolve.save()

            # Перепишем кэш
            cache_structure(formatted_key=formatted_key, structure=result, timeout=timeout)
            return result
        return wrapper

    return decorator

# Кэш-декоратор по умолчанию
cache_call = get_caching_wrapper
