# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import json
import logging
import os
import shutil
import uuid

from difflib import ndiff
from collections import defaultdict
from contextlib import contextmanager
from operator import itemgetter

from django.conf import settings
from lxml import etree
import yenv

from network_apis import NetworkResolver

from passport_grants_configurator.apps.core.exceptions import (
    ExportError,
    ImproperlyConfigured,
    NetworkResolveError,
    NoDiffError,
    ProcessError,
)
from passport_grants_configurator.apps.core.models import (
    ActiveAction,
    ActiveMacros,
    ActiveNetwork,
    Client,
    Consumer,
    Namespace,
    Network,
    PreviousResolve,
)
from passport_grants_configurator.apps.core.permissions import UserPermissions
from passport_grants_configurator.apps.core.utils import (
    check_grants,
    deb_changelog,
    git_add,
    git_diff,
    git_pull,
    git_push,
    git_status,
    grouped,
    request_git_last_commits_info_raw,
)

logger = logging.getLogger(__name__)


class FancyEncoder(json.JSONEncoder):
    def default(self, obj):
        if isinstance(obj, set):
            return sorted(obj)
        return json.JSONEncoder.default(self, obj)


class _YasmsToPassportFileNameNotFound(ImproperlyConfigured):
    """
    Для данного окружения не нашлось имя файла с грантами паспортного Я.Смса.
    """


NETWORK_RESOLVE_MESSAGE_TEMPLATE = '%s - сеть "%s", потребитель "%s": %s'
NETWORK_RESOLVE_ERROR = 'ОШИБКА'
NETWORK_RESOLVE_WARNING = 'Предупреждение'


def format_network_resolve_message(status, network, consumer, message):
    return NETWORK_RESOLVE_MESSAGE_TEMPLATE % (status, network, consumer, message)


def format_checker_script_errors(env, consumer, message):
    return '{} - в окружении {} для потребителя {} при дополнительной валидации были выявлены ошибки: {}'.format(
        NETWORK_RESOLVE_ERROR, env, consumer, message
    )


def format_unresolved_networks(unresolved_list):
    return 'не найдены ip-адреса для имен %s' % ', '.join(unresolved_list)


def format_excluded_ips_and_nets(excluded):
    return 'были пропущены следующие ip-адреса: %s' % ', '.join(excluded)


def format_unresolved_macros():
    return 'не найден макрос'


def format_export_files_out_of_sync(project_name, env_name, env_type):
    return 'В таблице NamespaceEnvironments есть лишнее окружение {en} {et} для проекта {pn}'.format(
        en=env_name,
        et=env_type,
        pn=project_name,
    )


def format_network_decreased_message(network_type, decreased_data):
    return 'При раскрытии {n_type} {n_name} сильно уменьшился состав: \n{diff}'.format(
        n_type=network_type,
        n_name=decreased_data['name'],
        diff='\n'.join(ndiff(decreased_data['old'], decreased_data['new'])),
    )


def get_warnings(unresolved_data, network, consumer_name, force_update=False):
    auto_resolve, manual_resolve = [], []
    auto_resolve_list = unresolved_data['auto_resolve']
    manual_resolve_list = unresolved_data['manual_resolve']
    excluded_list = unresolved_data['excluded']
    decreased_data = unresolved_data['decreased']
    if auto_resolve_list:
        auto_resolve.append(
            format_network_resolve_message(
                NETWORK_RESOLVE_WARNING,
                network.string,
                consumer_name,
                format_unresolved_networks(auto_resolve_list),
            ),
        )
    if manual_resolve_list:
        manual_resolve.append(
            format_network_resolve_message(
                NETWORK_RESOLVE_ERROR,
                network.string,
                consumer_name,
                format_unresolved_macros(),
            ),
        )
    if decreased_data and force_update:
        prev = PreviousResolve.objects.get(network=network)
        prev.children = json.dumps(decreased_data['new'])
        prev.save()

    elif decreased_data:
        manual_resolve.append(
            format_network_resolve_message(
                NETWORK_RESOLVE_ERROR,
                network.string,
                consumer_name,
                format_network_decreased_message(network.type, decreased_data),
            ),
        )
    if excluded_list:
        auto_resolve.append(
            format_network_resolve_message(
                NETWORK_RESOLVE_WARNING,
                network.string,
                consumer_name,
                format_excluded_ips_and_nets(excluded_list),
            ),
        )
    return auto_resolve, manual_resolve


def get_networks(network, consumer_name, force_update=False):
    ip_list, unresolved_data = NetworkResolver.get_ips(network)
    for_auto_resolve, for_manual_resolve = get_warnings(
        unresolved_data,
        network,
        consumer_name,
        force_update=force_update,
    )
    return ip_list, for_auto_resolve, for_manual_resolve


def _format_passport_grants(grants, with_client=True, prefix=None):
    """
    Строит читаемый человеком словарь паспортных грантов
    Возвращает его JSON-представление
    """
    new_grants = {}
    for consumer in grants:
        # Сортируем имена операций у каждого гранта
        consumer_grants = {}
        for grant_name, action_names in grants[consumer]['grants'].items():
            if grant_name != 'subscription':
                grant_key = grant_name
                if prefix:
                    grant_key = '%s%s' % (prefix, grant_name)
                consumer_grants[grant_key] = sorted(action_names)

        # Грант subscription обрабатываем особым образом
        if 'subscription' in grants[consumer]['grants']:
            consumer_grants['subscription'] = new_subscriptions = {}
            subscription_actions = grants[consumer]['grants']['subscription']
            for composite_action in subscription_actions:
                service, action = composite_action.split('.')
                new_subscriptions.setdefault(service, []).append(action)

        # Для каждого потребителя сортируем список сетей
        consumer_networks = sorted(grants[consumer]['networks'])

        consumer_key = consumer
        if prefix:
            consumer_key = '%s%s' % (prefix, consumer)

        new_grants[consumer_key] = {
            'grants': consumer_grants,
            'networks': consumer_networks,
        }
        if with_client:
            new_grants[consumer_key].update(client=grants[consumer].get('client', {}))
    return json.dumps(new_grants, indent=4, sort_keys=True)


def _format_blackbox_by_client_grants(grants):
    new_grants = {}
    for consumer in grants:
        # Сортируем имена операций у каждого гранта
        consumer_grants = {
            grant_name: sorted(action_names)
            for grant_name, action_names in grants[consumer]['grants'].items()
        }

        # Для каждого потребителя сортируем список сетей
        consumer_networks = sorted(grants[consumer]['networks'])

        # Клиент на связку Потребитель-Окружение может быть только один
        new_grants[consumer] = {
            'client': grants[consumer]['client'],
            'grants': consumer_grants,
            'networks': consumer_networks,
        }
    return json.dumps(new_grants, indent=4, sort_keys=True)


def build_default_dict_for_grants():
    return defaultdict(lambda: dict(grants=defaultdict(set), networks=set()))


def build_default_dict_for_grants_with_client():
    return defaultdict(
        lambda: dict(client=defaultdict(), grants=defaultdict(set), networks=set()),
    )


@contextmanager
def error_as_warning(consumer_name, network_name, warnings, error_class=NetworkResolveError):
    try:
        yield
    except error_class as e:
        warnings.append(format_network_resolve_message(
            NETWORK_RESOLVE_ERROR,
            network_name,
            consumer_name,
            e.message,
        ))


def export_passport_like_grants_to_file(environment, file_, namespace=None, force_update=False, prefix=None):
    warnings = []
    manual_resolve = []

    if not namespace:
        namespace = Namespace.objects.get(name='passport')

    grants = build_default_dict_for_grants_with_client()

    clients = Client.objects.select_related('consumer').filter(
        environment=environment,
        consumer__isnull=False,
        namespace=namespace,
    )
    for client in clients:
        grants[client.consumer.name]['client']['client_id'] = client.client_id
        grants[client.consumer.name]['client']['client_name'] = client.name

    queryset = ActiveNetwork.objects.select_related('consumer', 'network')\
        .filter(environment=environment, consumer__namespace=namespace)
    for an in queryset:
        consumer_networks = grants[an.consumer.name]['networks']
        # Строим список адресов для этого потребителя
        with error_as_warning(an.consumer.name, an.network.string, manual_resolve):
            ip_list, for_auto_resolve, for_manual_resolve = get_networks(
                an.network,
                an.consumer.name,
                force_update=force_update,
            )
            consumer_networks.update(ip_list)
            warnings.extend(for_auto_resolve)
            manual_resolve.extend(for_manual_resolve)

    queryset = ActiveAction.objects.select_related('consumer', 'action', 'action__grant')\
        .filter(environment=environment, consumer__namespace=namespace)
    for aa in queryset.order_by('action__name'):
        grants[aa.consumer.name]['grants'][aa.action.grant.name].add(aa.action.name)

    # TODO: Избавиться от вложенного цикла
    queryset = ActiveMacros.objects.select_related('consumer').prefetch_related('macros__action', 'macros__action__grant')\
        .filter(environment=environment, consumer__namespace=namespace)
    for am in queryset:
        for action in am.macros.action.all():
            grants[am.consumer.name]['grants'][action.grant.name].add(action.name)

    formatted_grants = _format_passport_grants(grants, prefix=prefix)
    file_.write(formatted_grants)
    return {'errors': list(), 'warnings': warnings, 'manual_resolve': manual_resolve}


def export_blackbox_by_client_grants_to_file(environment, file_, namespace='blackbox_by_client',
                                             force_update=False, **kwargs):
    warnings = []
    manual_resolve = []

    namespace = Namespace.objects.get(name=namespace)

    grants = build_default_dict_for_grants_with_client()

    clients = Client.objects.select_related('consumer').filter(
        environment=environment,
        consumer__isnull=False,
        namespace=namespace,
    )
    for client in clients:
        grants[client.consumer.name]['client']['client_id'] = client.client_id
        grants[client.consumer.name]['client']['client_name'] = client.name

    queryset = ActiveNetwork.objects.select_related('consumer', 'network') \
        .filter(environment=environment, consumer__namespace=namespace)
    for an in queryset:
        consumer_networks = grants[an.consumer.name]['networks']
        # Строим список адресов для этого потребителя
        with error_as_warning(an.consumer.name, an.network.string, manual_resolve):
            ip_list, for_auto_resolve, for_manual_resolve = get_networks(
                an.network,
                an.consumer.name,
                force_update=force_update,
            )
            consumer_networks.update(ip_list)
            warnings.extend(for_auto_resolve)
            manual_resolve.extend(for_manual_resolve)

    queryset = ActiveAction.objects.select_related('consumer', 'action', 'action__grant') \
        .filter(environment=environment, consumer__namespace=namespace)
    for aa in queryset.order_by('action__name'):
        grants[aa.consumer.name]['grants'][aa.action.grant.name].add(aa.action.name)

    # TODO: Избавиться от вложенного цикла.
    # + Понять, зачем это вообще нужно.
    queryset = ActiveMacros.objects.select_related('consumer').prefetch_related('macros__action',
                                                                                'macros__action__grant') \
        .filter(environment=environment, consumer__namespace=namespace)
    for am in queryset:
        for action in am.macros.action.all():
            grants[am.consumer.name]['grants'][action.grant.name].add(action.name)

    formatted_grants = _format_blackbox_by_client_grants(grants)
    file_.write(formatted_grants)
    return {'errors': list(), 'warnings': warnings, 'manual_resolve': manual_resolve}


def _export_oauth_grants_to_file(environment, file_, namespace, force_update=False, with_client=True):
    warnings = []
    manual_resolve = []

    if with_client:
        grants = build_default_dict_for_grants_with_client()

        clients = Client.objects.select_related('consumer').filter(
            environment=environment,
            consumer__isnull=False,
            namespace=namespace,
        )
        for client in clients:
            grants[client.consumer.name]['client']['client_id'] = client.client_id
            grants[client.consumer.name]['client']['client_name'] = client.name
    else:
        grants = build_default_dict_for_grants()

    consumers = list(Consumer.objects.filter(namespace=namespace).values_list('id', flat=True))

    queryset = ActiveNetwork.objects.select_related('consumer', 'network')\
        .filter(environment=environment, consumer__in=consumers)
    for an in queryset:
        with error_as_warning(an.consumer.name, an.network.string, manual_resolve):
            ip_list, for_auto_resolve, for_manual_resolve = get_networks(
                an.network,
                an.consumer.name,
                force_update=force_update,
            )
            grants[an.consumer.name]['networks'].update(ip_list)
            warnings.extend(for_auto_resolve)
            manual_resolve.extend(for_manual_resolve)

    queryset = ActiveAction.objects.select_related('consumer', 'action', 'action__grant')\
        .filter(environment=environment, consumer__in=consumers)
    for aa in queryset:
        grants[aa.consumer.name]['grants'][aa.action.grant.name].add(aa.action.name)

    # TODO: Избавиться от вложенного цикла
    queryset = ActiveMacros.objects.select_related('macros', 'consumer').prefetch_related('macros__action', 'macros__action__grant')\
        .filter(environment=environment, consumer__in=consumers)
    for am in queryset:
        for action in am.macros.action.all():
            grants[am.consumer.name]['grants'][action.grant.name].add(action.name)

    for meta_consumer in grants.itervalues():
        if 'client' in meta_consumer['grants'] and 'all' in meta_consumer['grants']['client']:
            meta_consumer['grants']['client'] = ['*']

    json.dump(
        grants, file_,
        indent=4, sort_keys=True, cls=FancyEncoder, separators=(',', ': '),
    )
    return {'errors': list(), 'warnings': warnings, 'manual_resolve': manual_resolve}


def export_oauth_grants_to_file(environment, file_, namespace, force_update=False, **kwargs):
    return _export_oauth_grants_to_file(environment, file_, namespace, force_update=force_update)


def export_tvm_api_grants_to_file(environment, file_, namespace, force_update=False, **kwargs):
    return _export_oauth_grants_to_file(environment, file_, namespace, force_update=force_update, with_client=False)


def export_social_grants_to_file(environment, namespace, file_, force_update=False, **kwargs):
    warnings = []
    manual_resolve = []

    # Нужно передавать не unicode-строки для правильной сборки грантов
    grants = dict()

    clients = Client.objects.select_related('consumer').filter(
        environment=environment,
        consumer__isnull=False,
        namespace=namespace,
    )

    for consumer_name in Consumer.objects.filter(namespace=namespace).values_list('name', flat=True):
        grants[str(consumer_name)] = meta_consumer = dict()
        meta_consumer['networks'] = list()
        meta_consumer['grants'] = list()
        meta_consumer['client'] = dict()

    for client in clients:
        grants[str(client.consumer.name)]['client']['client_id'] = client.client_id
        grants[str(client.consumer.name)]['client']['client_name'] = str(client.name)

    queryset = (
        ActiveNetwork
        .objects
        .select_related('consumer', 'network')
        .filter(environment=environment, consumer__namespace=namespace)
    )
    for an in queryset:
        with error_as_warning(an.consumer.name, an.network.string, manual_resolve):
            ip_list, for_auto_resolve, for_manual_resolve = get_networks(
                an.network,
                an.consumer.name,
                force_update=force_update,
            )
            for ip in ip_list:
                grants[str(an.consumer.name)]['networks'].append(str(ip))
            warnings.extend(for_auto_resolve)
            manual_resolve.extend(for_manual_resolve)

    queryset = (
        ActiveAction
        .objects
        .select_related('consumer', 'action', 'action__grant')
        .filter(environment=environment, consumer__namespace=namespace)
    )
    for aa in queryset:
        consumer_grants = grants[str(aa.consumer.name)]['grants']
        grant_name = aa.action.grant.name
        if aa.action.name == '*':
            consumer_grants.append(str(grant_name))
        else:
            name = '%s-%s' % (grant_name, aa.action.name)
            consumer_grants.append(str(name))

    # TODO: Избавиться от вложенного цикла
    queryset = (
        ActiveMacros
        .objects
        .select_related('consumer', 'macros')
        .prefetch_related('macros__action', 'macros__action__grant')
        .filter(environment=environment, consumer__namespace=namespace)
    )
    for am in queryset:
        consumer_grants = grants[str(am.consumer.name)]['grants']
        for action in am.macros.action.all():
            grant_name = action.grant.name
            if action.name == '*':
                consumer_grants.append(str(grant_name))
            else:
                name = '%s-%s' % (grant_name, action.name)
                consumer_grants.append(str(name))

    for consumer_name, meta_consumer in grants.items():
        if meta_consumer['networks'] or meta_consumer['grants']:
            meta_consumer['grants'] = sorted(set(meta_consumer['grants']))
            meta_consumer['networks'] = sorted(set(meta_consumer['networks']))

        # Вычищаем пустых потребителей
        else:
            del grants[consumer_name]

    json.dump(
        grants,
        file_,
        indent=4,
        sort_keys=True,
        cls=FancyEncoder,
        separators=(',', ': '),
    )
    return {'errors': list(), 'warnings': warnings, 'manual_resolve': manual_resolve}


def serialize_attributes(values_list, separator=','):
    """
    Список грантов на атрибуты состоит из чисел в строковом представлении
    Нужно отсортировать числа в натуральном порядке и записать их через запятую
    :param values_list: Список атрибутов -- чисел в строковом представлении
    :return: Строка отсортированных чисел через запятую
    """
    return separator.join(
        sorted(
            values_list,
            key=lambda value: int(value),
        ),
    )


def export_blackbox_grants_to_file(environment, file_, namespace=None, force_update=False, **kwargs):
    warnings = []
    manual_resolve = []

    peers = etree.Element('peers')
    blackbox = Namespace.objects.get(name='blackbox')
    consumer_id_list = Consumer.objects.filter(namespace=blackbox).values_list('id', flat=True)
    logger.info('Going to process %d consumers of blackbox', len(consumer_id_list))

    # В этом запросе выбираем только имена, по которым будем строить "дерево"
    actions_qs = ActiveAction.objects.select_related('consumer', 'action', 'action__grant').\
        filter(consumer__in=consumer_id_list, environment=environment).\
        order_by('consumer__name', 'action__grant__name', 'action__name').\
        values_list('consumer__name', 'action__grant__name', 'action__name')  # Понадобятся только эти имена
    actions_by_consumer_name = dict(
        (consumer_name, list(iterator))
        for consumer_name, iterator in grouped(
            actions_qs,
            key=itemgetter(0),  # Первый элемент это consumer_name
        )
    )

    # В этом запросе нам нужны модели Network чтобы получить все ip-адреса
    networks_qs = ActiveNetwork.objects.select_related('network', 'consumer').\
        filter(consumer__in=consumer_id_list, environment=environment).\
        order_by('consumer__name')

    active_networks_by_consumer_name = grouped(
        networks_qs,
        key=lambda item: item.consumer.name,
    )
    for consumer_name, active_networks in active_networks_by_consumer_name:
        consumer_networks = set()
        for active_network in active_networks:
            with error_as_warning(active_network.consumer.name, active_network.network.string, manual_resolve):
                ip_list, for_auto_resolve, for_manual_resolve = get_networks(
                    active_network.network,
                    active_network.consumer.name,
                    force_update=force_update,
                )
                consumer_networks.update(ip_list)
                warnings.extend(for_auto_resolve)
                manual_resolve.extend(for_manual_resolve)

        if not consumer_networks:
            logger.debug('No active networks for consumer %s', consumer_name)
            continue  # pragma: no cover

        ids = '; '.join(sorted(consumer_networks))
        consumer_tag = etree.SubElement(peers, 'entry', id=ids)
        name_tag = etree.SubElement(consumer_tag, 'name')
        name_tag.text = consumer_name

        # У нас сохранен для этого потребителя список из кортежей вида
        # (consumer_name, grant_name, action_name)
        values_list = actions_by_consumer_name.get(consumer_name)

        # Не для всех потребителей с активными сетями выданы гранты
        if not values_list:
            logger.debug('No active actions for consumer %s', consumer_name)
            continue

        # Сейчас мы получили гранты только для текущего потребителя и хотим
        # отсортировать список по имени гранта - второй элемент кортежа
        values_list = map(
            lambda value_tuple: value_tuple[1:],  # Сначала выбрасываем первый элемент
            values_list,
        )
        # Сгруппируем списки действий по грантам
        grants_and_actions = grouped(values_list, key=itemgetter(0))  # Первый элемент это grant_name
        logger.debug('Processing %d actions for consumer %s', len(values_list), consumer_name)

        for grant_name, grant_and_actions in grants_and_actions:
            # Собираем действия в список
            if grant_name in ['allowed_attributes', 'allowed_phone_attributes']:
                grant_tag = etree.SubElement(consumer_tag, grant_name)
                action_list = map(itemgetter(1), grant_and_actions)  # Берем только имена действий
                grant_tag.text = serialize_attributes(action_list)

            elif grant_name in ['allow_sign', 'allow_check_sign']:
                grant_tag = etree.SubElement(consumer_tag, grant_name)
                action_list = map(itemgetter(1), grant_and_actions)  # Берем только имена действий
                grant_tag.text = ','.join(action_list)

            elif grant_name.startswith('dbfield.'):
                for _, action_name in grant_and_actions:
                    section_id = '%s.%s' % (grant_name.split('.')[1], action_name)
                    etree.SubElement(consumer_tag, 'dbfield', id=section_id)

            else:
                for _, action_name in grant_and_actions:
                    grant_tag = etree.SubElement(consumer_tag, grant_name)
                    if not action_name == '*':
                        grant_tag.text = action_name

    logger.info('Writing Blackbox grants as XML')
    file_.write(etree.tostring(peers, pretty_print=True, encoding='utf-8', xml_declaration=True))
    return {'errors': list(), 'warnings': warnings, 'manual_resolve': manual_resolve}


class BaseExporter(object):
    prefix = None

    def __init__(self, namespaces, user, git_api, env_filenames):
        self.git_api = git_api
        self.export_namespace_environments = list(UserPermissions(user).intersect_grouped(namespaces))
        self.project_name = self.git_api['project']
        self.exporting_entity = 'grants'
        self.last_commit_sha = 'default'
        self.lock_file = 'default.lock'
        self.username = user.username
        self.repository_dir = os.path.join(
            self.git_api['working_dir'],
            yenv.type,
            str(uuid.uuid4()),
        )
        self.env_filenames = env_filenames

    @property
    def is_dry_run(self):
        return yenv.type in ('development', 'testing')

    def check_environment(self):
        """
        Проверим рабочее окружение
          * Была ли завершена предыдущая выгрузка?
          * Есть ли права на запись во временную папку для git-репозитория?
        """
        lock_name = '%s_%s.lock' % (self.project_name, self.last_commit_sha)
        self.lock_file = os.path.join(self.git_api['working_dir'], lock_name)
        if os.path.exists(self.lock_file):
            raise ExportError(['Предыдущая выгрузка грантов еще не завершена'])

        try:
            os.makedirs(self.repository_dir)
        except OSError:
            _error = 'Нет доступа к директории для работы с дистрибутивом %s' % self.repository_dir
            raise ExportError([_error])

    def check_permissions(self):
        """Проверим права пользователя на выгрузку в указанные окружения"""
        if not self.export_namespace_environments:
            raise ExportError(['У Вас нет прав для совершения данной операции'])

    def simulate(self, force_update=False):
        """Симуляция. Сборка грантов без выгрузки в центральный репозиторий"""
        try:
            self.prepare()
            self.import_grants()
            return self.update_grants(force_update=force_update)
        finally:
            self.cleanup()

    def export(self, force_update=False):
        """Непосредственная сборка и выгрузка грантов в центральный репозиторий"""
        # TODO: Покрыть тестами
        if self.is_dry_run:
            logger.debug('It\'s dry run')
            diff = self.simulate(force_update=force_update)
            diff['warnings'].insert(
                0,
                'DRY-RUN: '
                'Эта Грантушка работает в тестовом окружении - '
                'ничего не сохранялось',
            )
            return diff

        try:
            self.prepare()
            self.import_grants()
            diff = self.update_grants(force_update=force_update)
            if diff.get('manual_resolve', []):
                logger.info('Export stopped for %s', self.project_name)
                return diff
            logger.debug('Start {entity} export at {name}'.format(
                entity=self.exporting_entity,
                name=self.last_commit_sha,
            ))
            with open(self.lock_file, 'a'):
                self.export_grants()

            return diff
        finally:
            self.cleanup()

    def prepare(self):
        """Подготовка к выгрузке"""
        commit_info = request_git_last_commits_info_raw(api=self.git_api)
        if commit_info:
            self.last_commit_sha = commit_info['id']

        logger.debug(
            'Prepare to export {repo} {entity} by {user} at {commit}'.format(
                repo=self.git_api['repo'],
                entity=self.exporting_entity,
                user=self.username,
                commit=self.last_commit_sha,
            ),
        )

        self.check_permissions()
        self.check_environment()

    def cleanup(self):
        logger.debug('Cleanup at %s', self.last_commit_sha)
        try:
            shutil.rmtree(self.repository_dir)
            # При вызове self.simulate(), lock-файл не создается
            if os.path.exists(self.lock_file):
                os.remove(self.lock_file)
        except OSError as ex:
            logger.warning('Can\'t finish cleanup: %s', ex)

    def import_grants(self):
        """Метод для загрузки текущих грантов из центрального репозитория"""
        raise NotImplementedError()

    def update_grants(self, force_update=False):
        """Метод для обновления грантов новыми значениями с получением дифа"""
        raise NotImplementedError()

    def export_grants(self):
        """Метод выгрузки обновленных грантов в проект"""
        raise NotImplementedError()


class GitExporter(BaseExporter):
    """
    Выкачивает текущую версию git-репозитория с грантами
    Собирает актуальные гранты и обновляет файлы
    Поднимает версию debian-пакета грантов на единицу
    Фиксирует изменения как новый git-коммит в репозитории
    Отправляет изменения в центральный репозиторий
    """

    def import_grants(self):
        logger.debug('Import {entity} for {name}'.format(
            entity=self.exporting_entity,
            name=self.project_name,
        ))
        try:
            git_pull(
                repository=self.git_api['repo'],
                repository_dir=self.repository_dir,
            )
        except ProcessError as e:
            raise ExportError([e.message])

    def update_grants(self, force_update=False):
        logger.debug('Update {entity} for {name}: {forced}'.format(
            entity=self.exporting_entity,
            name=self.project_name,
            forced='forced' if force_update else 'regular',
        ))
        errors, warnings, manual_resolve = [], [], []
        for namespace, environments in self.export_namespace_environments:
            for environment in environments:
                try:
                    filename = self.env_filenames[(namespace.name, environment.name, environment.type)]
                except KeyError:
                    raise ExportError([
                        format_export_files_out_of_sync(
                            namespace.name,
                            environment.name,
                            environment.type,
                        ),
                    ])
                with open(os.path.join(self.repository_dir, filename), 'w') as file_:
                    exported = self.export_function(
                        namespace=namespace,
                        environment=environment,
                        file_=file_,
                        force_update=force_update,
                        prefix=self.prefix,
                    )
                    errors.extend(exported['errors'])
                    warnings.extend(exported['warnings'])
                    manual_resolve.extend(exported['manual_resolve'])
                    if errors:
                        raise ExportError(errors)

            script_errors = check_grants(namespace, self.repository_dir, settings.GRANTS_CHECKER_FILENAME)
            for env, env_errors in script_errors.iteritems():
                for consumer, message in env_errors.iteritems():
                    manual_resolve.append(
                        format_checker_script_errors(env, consumer, message),
                    )

        if manual_resolve:
            return {'diff': '', 'warnings': warnings, 'manual_resolve': manual_resolve}

        logger.debug('Getting diff for %s', self.project_name)

        # Если было добавлено новое окружение, его нужно добавить и в unstaged.
        if 'Untracked files:' in git_status(self.repository_dir):
            git_add(self.repository_dir)

        diff_out = git_diff(self.repository_dir)
        if not diff_out:
            error_str = 'В экспортированных грантах не найдено изменений по сравнению с грантами из репозитория'
            raise NoDiffError([error_str])

        try:
            logger.debug('Post new entry to changelog')
            deb_changelog(
                repository_dir=self.repository_dir,
                committer=self.git_api['committer'],
                message='%s has exported new grants' % self.username,
            )
            return {
                'diff': diff_out,
                'warnings': warnings,
                'manual_resolve': manual_resolve,
            }

        except OSError as e:
            raise ExportError(['Не удалось обновить версию пакета, вызов dch вернул ошибку: %s' % e.strerror])

    def export_grants(self):
        try:
            git_push(
                repository=self.git_api['repo'],
                repository_dir=self.repository_dir,
                committer=self.git_api['committer'],
                message='%s has exported new grants' % self.username,
            )
        except ProcessError as e:
            logger.debug('Exception at %s', self.last_commit_sha)
            raise ExportError([e.message])

    def export_function(self, *args, **kwargs):
        """Специфичная для проекта функция сериализации грантов в файл"""
        raise NotImplementedError()


class BlackboxExporter(GitExporter):
    export_function = staticmethod(export_blackbox_grants_to_file)


class BlackboxByClientExporter(GitExporter):
    export_function = staticmethod(export_blackbox_by_client_grants_to_file)


class PassportExporter(GitExporter):
    export_function = staticmethod(export_passport_like_grants_to_file)


class MeltingPotExporter(GitExporter):
    export_function = staticmethod(
        export_passport_like_grants_to_file,
    )


class HistoryDBExporter(PassportExporter):
    """Используется формат грантов, аналогичный Паспорту"""


class OAuthExporter(GitExporter):
    export_function = staticmethod(export_oauth_grants_to_file)


class TakeoutExporter(GitExporter):
    export_function = staticmethod(export_passport_like_grants_to_file)


class TVMApiExporter(GitExporter):
    export_function = staticmethod(export_tvm_api_grants_to_file)


class SocialExporter(GitExporter):
    export_function = staticmethod(export_social_grants_to_file)


class YaSMSExporter(GitExporter):
    prefix = 'old_yasms_grants_'

    export_function = staticmethod(export_passport_like_grants_to_file)


PROJECT_TO_EXPORTER = {
    'blackbox': BlackboxExporter,
    'blackbox_by_client': BlackboxByClientExporter,
    'historydb': HistoryDBExporter,
    'oauth': OAuthExporter,
    'passport': PassportExporter,
    'meltingpot': MeltingPotExporter,
    'social': SocialExporter,
    'yasms': YaSMSExporter,
    'takeout': TakeoutExporter,
    'tvm-api': TVMApiExporter,
}


def get_exporter(project_name, **kwargs):
    """
    По переданному имени проекта возвращает подходящий объект для выгрузки грантов
    :param project_name: Имя проекта, например 'blackbox'
    :param kwargs: дополнительные параметры
    :return: Экземпляр наследника BaseExporter
    :raise: ExportError если переданы неверные данные
    """
    is_unknown_project_name = (
        project_name not in PROJECT_TO_EXPORTER or
        project_name not in settings.PROJECTS
    )
    if is_unknown_project_name:
        raise ExportError(['Передано некорректное имя проекта %r' % project_name])

    project_data = settings.PROJECTS[project_name]
    namespace_qs = Namespace.objects.filter(name__in=project_data['namespaces'])
    exporter_class = PROJECT_TO_EXPORTER[project_name]
    return exporter_class(
        namespaces=namespace_qs,
        git_api=project_data['git_api'],
        env_filenames=project_data['file_names'],
        **kwargs
    )
