# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import re

from django import forms
from django.forms.models import modelformset_factory
from django.conf import settings
from django.core.exceptions import ValidationError

from .exceptions import (
    NetworkResolveError,
    UnknownNetworkTypeError,
)
from .models import (
    MAX_LENGTH,
    Action,
    AddNetworks,
    Client,
    Consumer,
    DelNetworks,
    Email,
    Environment,
    Grant,
    Issue,
    IssueNetworks,
    Macros,
    Namespace,
    Network,
)
from .network_apis import (
    looks_like_old_conductor_macro,
    NetworkResolver,
)
from .utils import (
    ascii_string,
    not_empty_string,
    switch_keyboard_layout_to_eng,
)

CONSUMER_NAME_ALLOWED_PATTERN = r"^[a-zA-Z0-9-_.;/'@,:()]+$"


# TODO посмотреть, можно ли упростить логику разделив форму на кейс создания и кейс редактирования
class GetGrantsListForm(forms.Form):
    u"""
    Проверка данных для ручки '/grants/get_grants_list/', отдающей список грантов при их запросе со страницы
    редактирования заявки.

    - Если редактируем issue, нужен его id.
    - Если создания потребителя, то нужен id проекта.
    - Для изменения потребителя нужно поле consumer.
    """
    issue = forms.ModelChoiceField(queryset=Issue.objects.filter(status__in=[Issue.DRAFT]), required=False)
    consumer = forms.ModelChoiceField(queryset=Consumer.objects.all(), required=False)
    namespace = forms.ModelChoiceField(queryset=Namespace.objects.all(), required=False)

    def clean(self):
        cleaned_data = super(GetGrantsListForm, self).clean()

        issue = cleaned_data.get('issue')
        consumer = cleaned_data.get('consumer')
        namespace = cleaned_data.get('namespace')

        if not(issue or consumer or namespace):
            err = u'Должно быть передано хотя бы одно из полей issue, consumer_name, namespace'
            raise ValidationError(err)

        if issue:
            cleaned_data['namespace'] = cleaned_data['namespace'] or issue.namespace
            if issue.type == Issue.TO_MODIFY:
                cleaned_data['consumer'] = cleaned_data['consumer'] or issue.consumer
                cleaned_data['environment'] = issue.environments.all()

        if consumer:
            if namespace:
                raise ValidationError(u'Если передано поле consumer, не должно быть передано namespace')
            cleaned_data['namespace'] = consumer.namespace

        return cleaned_data


class CreateGrantForm(forms.ModelForm):
    def __init__(self, *args, **kwargs):
        super(CreateGrantForm, self).__init__(*args, **kwargs)
        # Соберем поля для создания Гранта
        self.fields.update({
            'grant_%s' % k: v
            for k, v in forms.fields_for_model(Grant).items()
        })
        self.fields.keyOrder = ['grant_namespace', 'grant_name', 'name', 'description', 'dangerous']

        # Добавим валидаторы пустой строки и кирилицы
        self.fields['grant_name'].validators.extend([not_empty_string, ascii_string])
        self.fields['name'].validators.extend([not_empty_string, ascii_string])
        self.fields['description'].validators.append(not_empty_string)

    class Meta(object):
        model = Action
        exclude = ['grant']

    def clean(self):
        """Проверим, не существует ли такой грант в это проекте"""
        response = super(CreateGrantForm, self).clean()
        exists = Action.objects.select_related('grant').filter(
            grant__namespace=self.cleaned_data.get('grant_namespace'),
            grant__name=self.cleaned_data.get('grant_name'),
            name=self.cleaned_data.get('name'),
        ).exists()
        if not exists:
            return response

        raise ValidationError(u'Грант "%s.%s" уже существует' % (
            self.cleaned_data.get('grant_name'),
            self.cleaned_data.get('name'),
        ))

    def save(self, *args, **kwargs):
        grant, created = Grant.objects.get_or_create(
            name=self.cleaned_data['grant_name'],
            namespace=self.cleaned_data['grant_namespace'],
        )
        self.instance.grant = grant
        return super(CreateGrantForm, self).save(*args, **kwargs)


class ValidateNetworkForm(forms.Form):
    network = forms.CharField(max_length=250)
    # FIXME: Если этих полей нет, падает вызов network_validation
    namespace = forms.ModelChoiceField(queryset=Namespace.objects.all(), required=False)
    environment = forms.ModelChoiceField(queryset=Environment.objects.all(), required=False)

    def clean_network(self):
        network_keyword = switch_keyboard_layout_to_eng(self.cleaned_data['network'])
        try:
            if looks_like_old_conductor_macro(network_keyword):
                raise UnknownNetworkTypeError('Old conductor macro used %s' % network_keyword)
            network_type, objects = NetworkResolver.get_type_and_children(network_keyword)
            self.cleaned_data['objects'] = objects
            return Network(string=network_keyword, type=network_type)

        except NetworkResolveError as e:
            raise ValidationError(e.message)


class CreateIssueForm(forms.Form):
    """
    Проверка данных для страницы создания заявки. Для инициализации констант нужен либо номер заявки,
    которую будем редактировать, либо имя и окружение потребителя, для изменения которо мы создадим заявку.
    """
    issue = forms.ModelChoiceField(queryset=Issue.objects.filter(status__in=[Issue.DRAFT, Issue.NEW]), required=False)
    consumer = forms.ModelChoiceField(queryset=Consumer.objects.select_related('namespace').all(), required=False)
    environment = forms.ModelChoiceField(queryset=Environment.objects.all(), required=False)

    def clean(self):
        cleaned_data = super(CreateIssueForm, self).clean()

        issue = cleaned_data.get('issue')
        consumer = cleaned_data.get('consumer')
        environment = cleaned_data.get('environment')

        if issue and (consumer or environment):
            raise ValidationError('issue должно быть единственным переданным полем')

        if bool(consumer) != bool(environment):
            raise ValidationError(u'Поля consumer и environment должны передаваться вместе')

        return cleaned_data


class SubmitIssueForm(forms.ModelForm):
    u"""Проверка данных для заявки на изменение или создание потребителя."""

    REQUIRE_ANY_OF = (
        'add_action',
        'del_action',
        'add_macros',
        'del_macros',
        'clients',
        'consumer_name',
        'consumer_name_new',
        'consumer_description',
    )

    id = forms.ModelChoiceField(queryset=Issue.objects.filter(status=Issue.DRAFT), required=False)
    draft = forms.BooleanField(required=False)
    comment = forms.CharField(required=True, validators=[not_empty_string])
    clients = forms.ModelMultipleChoiceField(
        queryset=Client.objects.all(),
        required=False,
    )

    def __init__(self, data=None, *args, **kwargs):
        super(SubmitIssueForm, self).__init__(data, *args, **kwargs)
        namespace = data.get('namespace')

        if namespace:
            namespace_actions = Action.objects.filter(grant__namespace=namespace)
            self.fields['add_action'].queryset = self.fields['del_action'].queryset = namespace_actions
            namespace_macroses = Macros.objects.filter(namespace=namespace)
            self.fields['add_macros'].queryset = self.fields['del_macros'].queryset = namespace_macroses

    def is_empty(self):
        """Считаем форму пустой, если ни одно из ожидаемых полей не заполнено"""
        return not any(
            self.cleaned_data.get(p) for p in self.REQUIRE_ANY_OF
        )

    def clean(self):
        cleaned_data = super(SubmitIssueForm, self).clean()

        consumer = cleaned_data.get('consumer')
        consumer_name = cleaned_data.get('consumer_name', '').strip()
        consumer_name_new = cleaned_data.get('consumer_name_new', '').strip()
        environments = cleaned_data.get('environments', [])
        expiration = cleaned_data.get('expiration')
        namespace = cleaned_data.get('namespace')
        type_ = cleaned_data.get('type')
        clients = cleaned_data.get('clients', [])

        emails_obj = set()
        raw_emails = self.data.get('emails', [])
        emails_error_message = 'emails: Введите валидный адрес с доменом yandex-team или логин пользователя'
        for address in raw_emails:
            if not address:
                continue
            if '@' not in address:
                login = address
                domain = settings.DEFAULT_EMAIL_DOMAIN
                address = '@'.join([address, settings.DEFAULT_EMAIL_DOMAIN])
            else:
                login, domain = address.split('@', 1)
            if not (login and domain in settings.ALLOWED_NOTIFICATION_EMAIL_DOMAINS):
                raise ValidationError(emails_error_message)
            email, _ = Email.objects.get_or_create(address=address)
            emails_obj.add(email)

        cleaned_data['emails'] = list(emails_obj)

        if type_ in (Issue.TO_MODIFY, Issue.TO_SET_EXPIRATION):
            if not consumer:
                raise ValidationError(u'Не передано обязательное поле consumer')
            if Consumer.objects.filter(name=consumer_name_new, namespace=namespace).exists():
                raise ValidationError('Потребитель с именем "{}" уже существует'.format(consumer_name_new))
            if consumer_name_new and not re.match(CONSUMER_NAME_ALLOWED_PATTERN, consumer_name_new):
                raise ValidationError('Поле consumer_name содержит запрещенные символы')

        if type_ in (Issue.TO_CREATE, Issue.TO_CLONE):
            if not consumer_name:
                raise ValidationError(u'Не передано обязательное поле consumer_name')
            if not re.match(CONSUMER_NAME_ALLOWED_PATTERN, consumer_name):
                raise ValidationError('Поле consumer_name содержит запрещенные символы')
            if Issue.objects.filter(
                    consumer_name=consumer_name, status=Issue.NEW, type__in=[Issue.TO_CREATE, Issue.TO_CLONE], namespace=namespace
            ).exists():
                raise ValidationError(u'Заявка на создание потребителя с таким именем уже существует')
            if Consumer.objects.filter(name=consumer_name, namespace=namespace).exists():
                raise ValidationError(u'Потребитель с таким именем уже существует')
            if consumer_name_new:
                raise ValidationError(u'Невозможно переименовать еще не созданного потребителя')

        if type_ != Issue.TO_SET_EXPIRATION and expiration:
            raise ValidationError(u'Поле expiration должно передаваться только для заявок на временные гранты')
        if type_ == Issue.TO_SET_EXPIRATION and not expiration:
            raise ValidationError(u'Необходимо указать дату истечения срока действия грантов')

        if namespace:
            namespace_environments = namespace.environments.all()
            additional = set(environments) - set(namespace_environments)
            if additional:
                env_str = ', '.join(u'"%s"' % unicode(a) for a in additional)
                raise ValidationError(u'Окружения: %s не входят в окружения проекта %s' % (env_str, namespace))
            if namespace.name in settings.NAMESPACES_WITH_REQUIRED_CLIENT:
                # Проверим, что клиенты есть для всех заявленных окружений
                missing_clients = set(environments) - set([c.environment for c in clients])
                if missing_clients:
                    raise ValidationError(u'ClientID: обязательное поле для %s всех окружений' % namespace.name)

        if clients and consumer:
            for client in clients:
                if client.consumer and client.consumer.id != consumer.id:
                    raise ValidationError(u'ClientID: %s уже привязан к потребителю %s' % (
                        client.client_id,
                        client.consumer.name,
                    ))

        return cleaned_data

    class Meta(object):
        model = Issue
        fields = [
            'add_action',
            'add_macros',
            'consumer',
            'clients',
            'consumer_description',
            'consumer_name',
            'consumer_name_new',
            'del_action',
            'del_macros',
            'environments',
            'expiration',
            'namespace',
            'type',
        ]


class ReviewIssueForm(forms.Form):
    issue = forms.ModelChoiceField(queryset=Issue.objects.filter(status=Issue.NEW))
    result = forms.ChoiceField(choices=(['confirmed', ''], ['rejected', '']))

    def clean(self):
        super(ReviewIssueForm, self).clean()
        if not (self.cleaned_data.get('result') and self.cleaned_data.get('issue')):
            return self.cleaned_data

        if self.cleaned_data['result'] == 'rejected':
            return self.cleaned_data

        # Проверяем, что клиент не привязан к другому потребителю
        issue = self.cleaned_data['issue']
        if issue.type in (Issue.TO_MODIFY, Issue.TO_SET_EXPIRATION):
            clients_with_consumers = issue.clients.filter(consumer__isnull=False).exclude(consumer=issue.consumer)
        else:
            clients_with_consumers = issue.clients.filter(consumer__isnull=False)

        if clients_with_consumers:
            raise ValidationError(
                'Клиенты уже привязаны к потребителям; %s' % '; '.join(
                    ['client_id: %s - consumer: %s' % (c.client_id, c.consumer.name) for c in clients_with_consumers],
                ),
            )
        return self.cleaned_data


class IssueNetworksForm(forms.ModelForm):
    class Meta(object):
        model = IssueNetworks
        exclude = ['issue']

    def clean_network(self):
        network = self.cleaned_data['network']
        if network.type == Network.FIREWALL and looks_like_old_conductor_macro(network.string):
            raise ValidationError('Old conductor macro used %s' % network.string)
        return network

AddNetworksFormSet = modelformset_factory(AddNetworks, form=IssueNetworksForm)
DelNetworksFormSet = modelformset_factory(DelNetworks, form=IssueNetworksForm)


class IssueListForm(forms.Form):
    ALL = 'all'
    PENDING = 'pending'
    MY = 'my'
    DRAFTS = 'drafts'
    display_choices = ([display, ''] for display in [ALL, PENDING, MY, DRAFTS, ''])

    display = forms.ChoiceField(choices=display_choices, required=False, initial=ALL)
    namespace = forms.ModelChoiceField(queryset=Namespace.objects.all(), required=False)
    page = forms.IntegerField(required=False)
    keyword = forms.CharField(required=False)


class SearchConsumersForm(forms.Form):
    namespace = forms.ModelChoiceField(queryset=Namespace.objects.all())
    page = forms.IntegerField(required=False)
    keyword = forms.CharField(min_length=2, max_length=250, required=False)
    environments = forms.ModelMultipleChoiceField(queryset=Environment.objects.all(), required=False)


class ConsumerSuggestForm(forms.Form):
    namespace = forms.ModelChoiceField(queryset=Namespace.objects.all())
    keyword = forms.CharField(min_length=1, max_length=250)

    def clean_keyword(self):
        keyword = self.cleaned_data['keyword']
        keyword = switch_keyboard_layout_to_eng(keyword)
        return keyword.strip('_%')


class ConsumerForm(forms.ModelForm):
    name = forms.CharField(max_length=MAX_LENGTH, validators=[ascii_string])

    class Meta:
        model = Consumer


class GrantForm(forms.ModelForm):
    name = forms.CharField(max_length=MAX_LENGTH, validators=[ascii_string])

    class Meta:
        model = Grant


class ActionForm(forms.ModelForm):
    name = forms.CharField(max_length=MAX_LENGTH, validators=[ascii_string])

    class Meta:
        model = Action


class ClientForm(forms.ModelForm):
    name = forms.CharField(max_length=MAX_LENGTH, validators=[ascii_string], required=False)
    client_id = forms.IntegerField(required=True, min_value=1)
    namespace = forms.ModelChoiceField(queryset=Namespace.objects.all(), required=True)
    environment = forms.ModelChoiceField(queryset=Environment.objects.all(), required=True)

    class Meta:
        model = Client

    def clean(self):
        super(ClientForm, self).clean()
        client_id = self.cleaned_data.get('client_id')
        environment = self.cleaned_data.get('environment')
        namespace = self.cleaned_data.get('namespace')

        try:
            existing_client = Client.objects.select_related('consumer')\
                .get(environment=environment, client_id=client_id, namespace=namespace)
        except Client.DoesNotExist:
            return self.cleaned_data

        if existing_client.consumer:
            raise ValidationError(u'ClientID: %s уже привязан к потребителю %s' % (
                client_id,
                existing_client.consumer.name,
            ))
        self.cleaned_data.update(existing_client=existing_client)
        return self.cleaned_data


class NetworkSuggestForm(forms.Form):
    keyword = forms.CharField(min_length=3, max_length=250, required=True)

    def clean_keyword(self):
        keyword = self.cleaned_data['keyword'].lower()
        keyword = switch_keyboard_layout_to_eng(keyword)
        return keyword


class NotifyCreatorsForm(forms.Form):
    # https://doc.yandex-team.ru/Debian/conductor-guide/task/packages.xml#webhooks_new
    branch = forms.ChoiceField(choices=(['stable', ''],), required=True)
    ticket_tasks_total = forms.IntegerField(required=True)
    ticket_tasks_finished = forms.IntegerField(required=True)

    @classmethod
    def get_project_name(cls, package_name):
        return package_name.replace('yandex-', '').replace('-grants', '').replace('-', '_')

    def clean(self):
        super(NotifyCreatorsForm, self).clean()

        if not self.data.get('packages'):
            raise ValidationError('packages required')

        package_name = self.data['packages'][0]['package']

        project = self.get_project_name(package_name)
        if project not in settings.GRANTS_PROJECTS:
            raise ValidationError('no project for package name: {}'.format(package_name))

        self.cleaned_data['project'] = project
        return self.cleaned_data
