# -*- coding: utf-8 -*-
from textwrap import dedent

from django.conf import settings
from django.contrib.sites.models import get_current_site
from django.core.mail import (
    send_mail,
    send_mass_mail,
)
from django.core.urlresolvers import reverse
from django.template import loader, Context

from passport_grants_configurator.apps.core.models import Issue

ISSUE_SUMMARY_TEMPLATE = loader.get_template('mail/issue_summary.html')


def issue_summary(issue_id):
    issue = Issue.objects.select_related(
        'add_action', 'del_action',
        'add_network', 'del_network',
        'add_macros', 'del_macros',
    ).get(id=issue_id)

    context = Context({'issue': issue})
    summary = ISSUE_SUMMARY_TEMPLATE.render(context)
    return summary


def get_issue_link(request, issue):
    return 'https://%s%s' % (
        get_current_site(request).domain,
        reverse('issue', kwargs={'pk': issue.id}),
    )


def notify_issue_reviewed(request, issue, is_rejected=None, is_approved=None):
    """Отправляем письмо с сообщением о том, что заявка была рассмотрена"""
    issue_link = get_issue_link(request, issue)
    verdict = u'РАССМОТРЕНА'
    if is_rejected:
        verdict = u'ОТКЛОНЕНА'
    elif is_approved:
        verdict = u'ОДОБРЕНА'
    subject = u'%s %s заявка в %s для %s' % (
        get_project_title(),
        verdict,
        issue.namespace,
        issue.get_consumer_name(),
    )
    body = u'Сегодня %s рассмотрел вашу заявку для потребителя %s\n%s' % (
        request.user.username,
        issue.get_consumer_name(),
        issue_link,
    )
    send_mail(
        subject,
        body,
        settings.EMAIL_NOTIFICATION_FROM,
        [issue.creator.email],
        fail_silently=True,
    )


def notify_package_in_stable(request, recipients_and_issues, project):
    """
    Отправляем письмо заказчику с сообщением о том, что пакет с грантами в проде
    :type request: request
    :type recipients_and_issues: defaultdict
    :type project: basestring
    """
    mails_data = []

    subject = u'%s Пакет c нужными вам грантами для %s в проде' % (
        get_project_title(),
        project,
    )

    for email_address, issues in recipients_and_issues.iteritems():
        issues_data = []
        for issue in issues:
            consumer = u'Потребитель: {}'.format(issue.get_consumer_name())
            issue_link = u'Ссылка: {}'.format(get_issue_link(request, issue))
            issues_data.append('\n'.join([consumer, issue_link]))

        body = u'Ваши заявки:\n{}'.format('\n---\n'.join(issues_data))
        mails_data.append(
            (
                subject,
                body,
                settings.EMAIL_NOTIFICATION_FROM,
                [email_address],
            ),
        )
    if mails_data:
        send_mass_mail(
            tuple(mails_data),
            fail_silently=True,
        )


def render_notification_new_issue(context):
    return (
        u'Добрый день, \n\n'
        u'Пользователь %(author)s создал(а) заявку в проекте %(namespace)s '
        u'на изменение прав потребителя %(consumer)s в окружении(ях) %(environments)s. \n'
        u'%(summary)s\n'
        u'Комментарий пользователя: \n'
        u'%(comment)s \n\n'
        u'Подробности заявки доступны по ссылке \n'
        u'%(link)s \n'
        u'--- \n'
        u'Ваша Грантушка'
    ) % context


def notify_new_issue_created(request, issue, comment):
    subject = u'%s Заявка в %s для %s ожидает подтверждения' % (
        get_project_title(),
        issue.namespace,
        issue.get_consumer_name(),
    )
    issue_link = get_issue_link(request, issue)
    environments = ', '.join(map(str, issue.environments.all()))
    consumer_name = issue.consumer.name if issue.consumer else (issue.consumer_name_new or issue.consumer_name)
    context = {
        'author': request.user.username,
        'namespace': issue.namespace,
        'environments': environments,
        'consumer': consumer_name,
        'summary': issue_summary(issue.id),
        'comment': comment.comment,
        'link': issue_link,
    }
    message = render_notification_new_issue(context)
    send_mail(
        subject,
        message,
        settings.EMAIL_NOTIFICATION_FROM,
        settings.EMAIL_GRANT_ADMINS,
        fail_silently=True,
    )


def notify(template=None, recipient_list=None, context=None):
    send_mail(
        subject=template['subject'].format(**context),
        message=template['message'].format(**context),
        from_email=settings.EMAIL_NOTIFICATION_FROM,
        recipient_list=recipient_list,
        fail_silently=True
    )


def get_project_title():
    if settings.GRANTS_MAIL_ENV_NAME:
        env_name = settings.ENV_NAME
        return u'(Grant Configurator, {env_name})'.format(env_name=env_name)
    return u'(Grant Configurator)'


MAIL_TEMPLATES = {
    'expiration': {
        'subject': get_project_title() +
                   u'У потребителя {consumer} истекает срок действия грантов',
        'message': dedent(u'''\
                Добрый день!

                У потребителя "{consumer}" истекает срок действия {objects__genitive}.
            Дата истечения срока действия - {expiration}.
        '''),
    },
    'networks_changed': {
        'subject': get_project_title() +
                   u'В проекте {project} есть невыгруженные изменения в грантах',
        'message': dedent(u'''\
                Добрый день!

                В проекте {project} есть невыгруженные изменения в грантах:
            {warnings}

            {diff}

                Экспортировать: {export_link}
        '''),
    },
    'export_failed': {
        'subject': get_project_title() +
                   u'При ежедневной выгрузке грантов в проекте {project} возникли проблемы',
        'message': dedent(u'''\
                Добрый день!

                При выгрузке грантов в проекте {project} возникли следующие ошибки:
            {errors}
        '''),
    },
    'manual_resolve': {
        'subject': get_project_title() +
                   u'Выгрузка в проект {project} остановлена',
        'message': dedent(u'''\
                Добрый день!

                Следующие сети для проекта {project} требуют ручного обновления:
            {manual_resolve}
        '''),
    },
    'keys_changed': {
        'subject': get_project_title() +
                   u'В проекте {project} есть невыгруженные изменения в ключах',
        'message': dedent(u'''\
                Добрый день!

                В проекте {project} есть невыгруженные изменения в ключах:
            {warnings}

            {diff}

                Экспортировать: {export_link}
        '''),
    },
    'keys_export_failed': {
        'subject': get_project_title() +
                   u'При ежедневной выгрузке ключей в проекте {project} возникли проблемы',
        'message': dedent(u'''\
                Добрый день!

                При выгрузке ключей в проекте {project} возникли следующие ошибки:
            {errors}
        '''),
    },
}
