# -*- coding: utf-8 -*-
from django.contrib.auth.models import (
    Group,
    Permission,
)
from django.contrib.contenttypes.models import ContentType
from django.core.management.base import CommandError
from django_docopt_command import DocOptCommand

from guardian.models import GroupObjectPermission

from passport_grants_configurator.apps.core.models import (
    Environment,
    Namespace,
    NamespaceEnvironments,
)


class Command(DocOptCommand):
    docs = """
    Usage:
        add_namespace (-n | --namespace-name) <namespace> (-e | --environment) <name> <type>...
        add_namespace (-n | --namespace-name) <namespace> (-e | --environment) <name> <type>... --no-update-groups
        add_namespace (-n | --namespace-name) <namespace>
        add_namespace (-n | --namespace-name) <namespace> --del
        add_namespace (-h | --help)

    Options:
        -h --help           Show this screen
        --no-update-groups  Do not update permissions groups
        --del               Delete namespace with its NamespaceEnvironments and permissions
    """
    help = u'Добавляет проект с окружениями и правами'

    def del_groups(self, namespace):
        namespace_ctid = ContentType.objects.get(model='namespace', app_label='core').id
        n_permission_id = Permission.objects.get(content_type_id=namespace_ctid, codename='review_namespace_issue').id
        try:
            n_group = Group.objects.get(name=u'Проект ' + namespace.name)
            GroupObjectPermission.objects.get(
                permission_id=n_permission_id,
                content_type_id=namespace_ctid,
                group_id=n_group.id,
                object_pk=namespace.pk,
            ).delete()
            n_group.delete()
        except (Group.DoesNotExist, GroupObjectPermission.DoesNotExist):
            pass

        namespace_environments = NamespaceEnvironments.objects.filter(namespace=namespace)

        ns_env_ctid = ContentType.objects.get(model='namespaceenvironments', app_label='core').id
        ne_permission_id = Permission.objects.get(content_type_id=ns_env_ctid, codename="namespace_environment").id

        for ne in namespace_environments:
            try:
                ne_group = Group.objects.get(
                    name=u'Права на {} {} {}'.format(
                        ne.namespace.name,
                        ne.environment.name,
                        ne.environment.type,
                    ),
                )
                GroupObjectPermission.objects.get(
                    permission_id=ne_permission_id,
                    content_type_id=ns_env_ctid,
                    group_id=ne_group.id,
                    object_pk=ne.id,
                ).delete()
                ne_group.delete()
            except (Group.DoesNotExist, GroupObjectPermission.DoesNotExist):
                continue

    def add_ns_group(self, namespace):
        # Добавляем права на то, чтобы ревьюить заявки в проект
        namespace_ctid = ContentType.objects.get(model='namespace', app_label='core').id
        n_permission_id = Permission.objects.get(content_type_id=namespace_ctid, codename='review_namespace_issue').id
        n_group, _ = Group.objects.get_or_create(name=u'Проект ' + namespace.name)
        GroupObjectPermission.objects.get_or_create(
            permission_id=n_permission_id,
            content_type_id=namespace_ctid,
            group_id=n_group.id,
            object_pk=namespace.pk,
        )

    def add_ne_group(self, namespace_environments):
        # Добавляем права на выгрузку этого проекта
        ns_env_ctid = ContentType.objects.get(model='namespaceenvironments', app_label='core').id
        ne_permission_id = Permission.objects.get(content_type_id=ns_env_ctid, codename="namespace_environment").id

        for ne in namespace_environments:
            ne_group, _ = Group.objects.get_or_create(
                name=u'Права на {} {} {}'.format(
                    ne.namespace.name,
                    ne.environment.name,
                    ne.environment.type,
                ),
            )
            GroupObjectPermission.objects.get_or_create(
                permission_id=ne_permission_id,
                content_type_id=ns_env_ctid,
                group_id=ne_group.id,
                object_pk=ne.id,
            )

    def handle_docopt(self, arguments):
        if arguments['--del']:
            try:
                namespace = Namespace.objects.get(name=arguments['<namespace>'])
            except Namespace.DoesNotExist:
                print 'No such namespace {}'.format(arguments['<namespace>'])
                return
            self.del_groups(namespace)
            namespace.delete()
            print 'Deleted namespace {}'.format(namespace.name)
            return

        # Create Namespace
        namespace, is_created = Namespace.objects.get_or_create(name=arguments['<namespace>'])
        self.add_ns_group(namespace)

        if is_created:
            print 'Created namespace {}'.format(namespace.name)

        # Create NamespaceEnvironments
        namespace_environments = []
        for type_ in arguments['<type>']:
            try:
                environment = Environment.objects.get(name=arguments['<name>'], type=type_)
                ne, _ = NamespaceEnvironments.objects.get_or_create(
                    environment=environment,
                    namespace=namespace,
                )
                namespace_environments.append(ne)
            except Environment.DoesNotExist:
                raise CommandError('Please provide existing environment name and type in parameters')

        # Update permissions and groups
        self.add_ne_group(namespace_environments)
        print 'Done'
