# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from netaddr import IPRange
from bs4 import BeautifulSoup
from django_docopt_command import DocOptCommand

from passport_grants_configurator.apps.core.network_apis import NetworkResolver
from passport_grants_configurator.apps.core.exceptions import HostResolvingFailed
from passport_grants_configurator.apps.core.models import (
    Action,
    ActiveAction,
    ActiveNetwork,
    Consumer,
    Environment,
    Grant,
    Namespace,
    Network,
)


class Command(DocOptCommand):
    docs = '''
    Usage:
        blackbox_import_grants (-f | --file) <file> (-n | --environment-name) <name>  (-t | --environment-type) <type>
    '''

    def handle_docopt(self, arguments):
        namespace, _ = Namespace.objects.get_or_create(name='blackbox')

        environment = Environment.objects.get(name=arguments['<name>'], type=arguments['<type>'])

        ActiveAction.objects.filter(environment=environment, consumer__namespace=namespace).delete()
        ActiveNetwork.objects.filter(environment=environment, consumer__namespace=namespace).delete()

        with open(arguments['<file>']) as f:
            grants_raw = f.read()
        grants = BeautifulSoup(grants_raw)

        for consumer_tag in grants.peers.find_all('entry'):
            networks = []
            actions = []
            try:
                comment = consumer_tag.find('name').text
            except AttributeError:  # Нет комментария
                comment = consumer_tag['id']

            for grant_tag in consumer_tag.find_all(True, recursive=False):
                # К счастью параметрические гранты login_limits устарели
                # Если в name хранятся имена потребителя, то в comment - комменты; иначе комменты хранятся в name
                if grant_tag.name in ('name', 'comment', 'login_limits') or grant_tag.name.startswith('_'):
                    continue
                elif grant_tag.name == 'dbfield':
                    grant_name, action_name = grant_tag['id'].split('.', 1)
                    grant, _ = Grant.objects.get_or_create(name='dbfield.' + grant_name, namespace=namespace)
                    action, _ = Action.objects.get_or_create(name=action_name, grant=grant)
                    actions.append(action)
                elif grant_tag.name in ['allowed_attributes', 'allowed_phone_attributes']:
                    grant, _ = Grant.objects.get_or_create(name=grant_tag.name, namespace=namespace)
                    action_names = [action_name.strip() for action_name in grant_tag.text.split(',')]
                    for action_name in action_names:
                        action, _ = Action.objects.get_or_create(
                            name=action_name,
                            grant=grant,
                        )
                        actions.append(action)
                else:
                    grant, _ = Grant.objects.get_or_create(name=grant_tag.name, namespace=namespace)
                    action, _ = Action.objects.get_or_create(
                        name=(grant_tag.text if grant_tag.text else '*'),
                        grant=grant,
                    )
                    actions.append(action)

            network_strings = [
                n.strip()
                for n in consumer_tag['id'].split(';')
                if n.strip()
            ]
            for string in network_strings:
                try:
                    network_type_ = NetworkResolver.get_type(string)
                    network, _ = Network.objects.get_or_create(string=string, type=network_type_)
                    networks.append(network)
                except HostResolvingFailed:
                    if '-' in string:
                        for ip in IPRange(*string.split('-')):
                            network, _ = Network.objects.get_or_create(string=str(ip), type=Network.IP)
                            networks.append(network)
                    else:
                        raise HostResolvingFailed('Cannot determine network type for {}'.format(string))

            name = comment
            consumer, _ = Consumer.objects.get_or_create(name=name, namespace=namespace)
            print 'imported "{}"'.format(name)

            for network in networks:
                ActiveNetwork.objects.get_or_create(consumer=consumer, network=network, environment=environment)
            for action in actions:
                ActiveAction.objects.get_or_create(consumer=consumer, action=action, environment=environment)
