# -*- coding: utf-8 -*-

from itertools import (
    chain,
    izip_longest,
    product,
)
from optparse import make_option
from django.core.management.base import BaseCommand

from passport_grants_configurator.apps.core.models import Network
from passport_grants_configurator.apps.core.network_apis import (
    request_firewall_macro_list,
    request_conductor_groups_list,
)


class Command(BaseCommand):
    help = 'Turns firewall macros _С_GROUP into conductor groups %group'
    option_list = BaseCommand.option_list + (
        make_option(
            '--undo',
            action='store_true',
            dest='undo',
            default=False,
            help='Turns conductor groups into firewall macros again',
        ),
    )

    def handle(self, *args, **options):
        if options['undo']:
            # Макросы, которые обновляли вручную - потерялись,
            # поэтому их тоже добавлять руками
            conductor_groups = Network.objects.filter(type=Network.CONDUCTOR)
            for group in conductor_groups:
                firewall_string = group.string.upper().replace('%', '_C_').replace('-', '_') + '_'
                group.type = Network.FIREWALL
                group.string = firewall_string
                group.save()
            print('%d conductor groups has become firewall macros again' % len(conductor_groups))
            return

        # Надо зачистить все заплесневелые группы из БД
        Network.objects.filter(type=Network.CONDUCTOR).delete()

        known_conductor_groups = set(request_conductor_groups_list())
        known_firewall_list = set(request_firewall_macro_list())
        conductor_macros = {n.string for n in Network.objects.filter(string__istartswith='_C_')}
        # Макросы чистятся и так, поэтому игнорим те, которые уже невалидны
        actual_conductor_groups = conductor_macros.intersection(known_firewall_list)

        count_all = len(actual_conductor_groups)
        count_found = 0
        for_manual_resolve = []
        separators = ['-', '_']
        separator_variants = {}
        for conductor_macro in actual_conductor_groups:
            # Перебираем все варианты взаимного расположения "_" и "-" в имени группы
            clean_macro_list = conductor_macro[3:-1].lower().split('_')
            repeat = len(clean_macro_list) - 1
            variants = separator_variants.setdefault(repeat, set(product(separators, repeat=repeat)))

            # Sorry (-_-)
            zipped_variants = [list(chain.from_iterable(izip_longest(clean_macro_list, variant, fillvalue='')))
                               for variant in variants]
            ready_to_check = {'%' + ''.join(v) for v in zipped_variants}
            groups_in = ready_to_check.intersection(known_conductor_groups)
            # Если встречается несколько таких подходящих - отдаем в надежные руки админов
            several_in = len(groups_in) > 1
            all_out = not groups_in
            if several_in or all_out:
                for_manual_resolve.append(conductor_macro)
            else:
                # Подошел ровно один вариант
                macro_object = Network.objects.get(string=conductor_macro)
                macro_object.type = Network.CONDUCTOR
                macro_object.string = groups_in.pop()
                macro_object.save()
                count_found += 1

        print('Task is done.')
        print('Found: %d/%d' % (count_found, count_all))
        print('Unknown groups: %d/%d' % (len(for_manual_resolve), count_all))
        print('\n'.join(for_manual_resolve))
