# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import logging
from collections import Counter, defaultdict
from datetime import date, timedelta
from itertools import chain

from django.conf import settings
from django_docopt_command import DocOptCommand

from passport_grants_configurator.apps.core.exceptions import NotFoundError
from passport_grants_configurator.apps.core.models import *
from passport_grants_configurator.apps.core.network_apis import (
    NetworkResolver,
    expand_firewall_macro,
    get_conductor_group_hosts,
    get_hostname_responsible_people_mails,
)
from passport_grants_configurator.apps.core.utils import format_objects
from passport_grants_configurator.apps.core.mail import (
    notify,
    MAIL_TEMPLATES,
)

logger = logging.getLogger(__name__)


class Command(DocOptCommand):
    docs = '''usage: daily'''

    def handle_docopt(self, arguments):
        #  Временные гранты
        today = date.today()

        logger.debug('Delete expired ActiveAction & ActiveMacros')
        ActiveAction.objects.filter(expiration__lte=today).delete()
        ActiveMacros.objects.filter(expiration__lte=today).delete()

        # Получаем 2 даты истечения грантов о которых напишем в сегодняшних письмах
        first_notification_expiration_date = today + timedelta(days=settings.DAYS_BEFORE_FIRST_NOTIFICATION)
        second_notification_expiration_date = today + timedelta(days=settings.DAYS_BEFORE_SECOND_NOTIFICATION)

        logger.debug('Looking for nearest expirations')
        for expiration_date in (first_notification_expiration_date, second_notification_expiration_date):
            # Гранты и макросы, которые истекут в указанную дату и их потребителей
            consumer_object = chain(
                [(aa.consumer, aa.action) for aa in ActiveAction.objects.filter(expiration=expiration_date)],
                [(am.consumer, am.action) for am in ActiveMacros.objects.filter(expiration=expiration_date)],
            )

            consumer_objects = defaultdict(list)
            for consumer, object_ in consumer_object:
                consumer_objects[consumer].append(object_)

            logger.debug('Processing %d consumers with objects to expire %s', len(consumer_objects), expiration_date)
            for consumer, objects in consumer_objects.items():
                # Для потребителя с истекающими грантами находим его хосты и хосты в его фаервольных макросах
                networks = Network.objects.prefetch_related('activenetwork').filter(
                    activenetwork__consumer=consumer,
                    type__in=[Network.FIREWALL, Network.HOSTNAME, Network.CONDUCTOR]
                ).distinct()

                logger.debug('Getting hosts for consumer %s', consumer)
                hosts = set()
                for network in networks:
                    if network.type == Network.FIREWALL:
                        try:
                            expanded_macro = expand_firewall_macro(network)
                        except NotFoundError:
                            logger.warning('Macro %s not found while removing expired grants; ignoring' % network)
                            continue

                        for string in expanded_macro:
                            type_, _ = NetworkResolver.get_type_and_children(string)
                            if type_ == Network.HOSTNAME:
                                hosts.add(string)

                    elif network.type == Network.CONDUCTOR:
                        try:
                            conductor_hosts = get_conductor_group_hosts(network)
                        except NotFoundError:
                            logger.warning('Conductor Group %s not found while removing expired grants; ignoring' % network)
                            continue

                        hosts.update(conductor_hosts)

                    elif network.type == Network.HOSTNAME:
                        hosts.add(network.string)

                # Находим до 3х наиболее частых адресата для уникальнах хостов потребителя
                hosts = map(get_hostname_responsible_people_mails, hosts)
                mails = Counter(chain.from_iterable(hosts))
                mails = [mail for mail, count in mails.most_common(3)]

                logger.debug('Sending emails to %d recipients', len(mails))
                # Отправляем письма
                notify(
                    template=MAIL_TEMPLATES['expiration'],
                    recipient_list=mails,
                    context=dict(
                        consumer=consumer.name,
                        expiration=expiration_date.strftime(settings.PYTHON_DATE_FORMAT),
                        objects__genitive=format_objects(objects, case='genitive')
                    )
                )
