# -*- coding: utf-8 -*-
from __future__ import unicode_literals
import logging

from django_docopt_command import DocOptCommand

from passport_grants_configurator.apps.core.exceptions import (
    APIRequestFailedError,
    NetworkResolveError,
    NoDataError,
    SubprocessTimedOut,
)
from passport_grants_configurator.apps.core.models import (
    Network,
    PreviousResolve,
)
from passport_grants_configurator.apps.core.network_apis import (
    get_conductor_group_hosts,
    expand_firewall_macro,
    NetworkCacheManager,
)
from passport_grants_configurator.apps.core.network_search_tools import cache_all_active_networks

logger = logging.getLogger(__name__)


class Command(DocOptCommand):
    docs = '''usage: hourly'''

    @classmethod
    def rebuild_conductor_groups_cache(cls):
        """
        Прогреваем кэш состава всех известных кондукторных групп
        """
        logger.info('Rebuild cache for conductor groups')
        networks = Network.objects.conductor().exclude(activenetwork=None)\
            .values_list('string', flat=True)
        for network in networks:
            try:
                # Результат вызова кэшируется
                get_conductor_group_hosts(network, recache=True)
            except NetworkResolveError:
                logger.warning('Network resolve error for conductor group %s while rebuilding cache; ignoring', network)

    @classmethod
    def rebuild_firewall_macros_cache(cls):
        """
        Прогреваем кэш состава всех известных макросов
        Получам в кэше соответствие имя макроса - список объектов
        """
        logger.info('Rebuild cache for firewall macros')

        macros_list = Network.objects.macros().exclude(activenetwork=None)\
            .values_list('string', flat=True)

        # Результат каждого вызова кэшируется
        for macro in macros_list:
            try:
                expand_firewall_macro(macro, recache=True)
            except NetworkResolveError:
                logger.warning('Network resolve error for macro %s while rebuilding cache; ignoring', macro)

    def handle_docopt(self, arguments):
        self.rebuild_firewall_macros_cache()

        self.rebuild_conductor_groups_cache()

        logger.info('Force active networks cache rebuild')
        # Только для запуска из консоли:
        # тут все юникодное, но ошибка в недрах джанги будет приводиться к строке.
        # Более подробная инфа сохранится в логе default и network
        try:
            cache_all_active_networks(mild=False)
        except APIRequestFailedError as ex:
            logger.error('Request failed: %s', ex.message.encode('ascii', 'replace'))
            return
        except NoDataError:
            logger.error('No data for request')
            return

        # Этот список используется в suggest_consumer и match_networks
        logger.info('Rebuild cache for known active networks')
        active_network_values_list = NetworkCacheManager.calc_active_network_values()
        NetworkCacheManager.save_all_networks_values(active_network_values_list)

        # Подчищаем больше никем не используемые результаты предыдущего раскрытия сетей
        logger.info('Cleaning up PreviousResolve Table by usage')
        PreviousResolve.objects.filter(network__activenetwork=None).delete()

        logger.info('Task is done')
