# -*- coding: utf-8 -*-

import json
from optparse import make_option

from django.core.management.base import BaseCommand, CommandError
from passport_backend_core.services.services import services

from passport_grants_configurator.apps.core.models import Namespace, Grant, Action, Macros


class Command(BaseCommand):
    help = 'Imports json file with grants and macroses into grants DB'
    option_list = BaseCommand.option_list + (
        make_option('--delete',
                    action='store_true',
                    dest='delete',
                    default=False,
                    help='Delete all obsolete objects without prompting questions on each'),
        make_option('-f',
                    '--file',
                    dest='filename',
                    help='Json file to import',
                    metavar='FILE')
    )

    def handle(self, *args, **options):
        try:
            grants_string = open(options['filename']).read()
        except IOError:
            raise CommandError('Can\'t find or access json file')
        except (IndexError, TypeError):
            raise CommandError('Please provide json file location as a parameter')

        try:
            grants_json = json.loads(grants_string)
        except ValueError, e:
            raise CommandError('Json file parse error:\n' + e.message)

        try:
            NAMESPACE_NAME = grants_json['namespace']
            META_GRANTS = grants_json['grants']
            META_SUBSCRIPTIONS = grants_json['subscriptions']
            META_MACROSES = grants_json['macroses']
        except KeyError, e:
            raise CommandError('Can\'t find "%s" section in the json file ' % e.message)

        # Проверка, не присвоены ли макросам несуществующие гранты
        for macros_name, (description, meta_grants) in META_MACROSES.iteritems():
            meta_grants = meta_grants.copy()
            if NAMESPACE_NAME == 'passport':
                meta_subscriptions = meta_grants.pop('subscription', [])
            for grant_name, action_names in meta_grants.iteritems():
                if not grant_name in META_GRANTS:
                    raise CommandError(
                        'Can\'t find "%s" grant enlisted in macros "%s", in '
                        'grants section of the file' % (grant_name, macros_name)
                    )
                nonexistent_actions = set(action_names) - set(META_GRANTS[grant_name])
                if nonexistent_actions:
                    raise CommandError(
                        'Can\'t find "%s" action(s) of "%s" grant enlisted in '
                        'macros "%s", in grants section of the file'
                        % (grant_name, ', '.join(nonexistent_actions), macros_name)
                    )
            if NAMESPACE_NAME == 'passport':
                nonexistent_actions = (set(meta_subscriptions) -
                                       set(META_SUBSCRIPTIONS.keys() + META_GRANTS.get('subscription', {}).keys()))
                if nonexistent_actions:
                    raise CommandError(
                        'Can\'t find "%s" action(s) of "subscription" grant enlisted in'
                        ' macros "%s" in subscriptions section of the file'
                        % (', '.join(nonexistent_actions), macros_name)
                    )

        # При необходимости создаем неймспейс
        namespace, created = Namespace.objects.get_or_create(name=NAMESPACE_NAME)

        current_grants = set()  # Здесь и далее все сеты для проверки пропали ли гранты/действия/...
        current_actions = set()
        for grant_name, meta_actions in META_GRANTS.iteritems():
            # Создаем недостающие гранты
            grant, created = Grant.objects.get_or_create(name=grant_name, namespace=namespace)
            current_grants.add(grant)
            for action_name, description in meta_actions.iteritems():
                # Создаем недостающие действия к грантам
                action, created = Action.objects.get_or_create(name=action_name, grant=grant)
                current_actions.add(action)
                # Обновляем описания действий при необходимости
                if action.description != description:
                    action.description = description
                    action.save()

        if NAMESPACE_NAME == 'passport':
            # При необходимости создаем грант subscription
            current_subscription_actions = set()
            subscription, created = Grant.objects.get_or_create(name='subscription', namespace=namespace)
            # Создаем недостающие действия subscription для каждого сервиса
            for action_name, description in META_SUBSCRIPTIONS.iteritems():
                for service_name, service in services.items():
                    name = '%s.%s' % (service_name, action_name)
                    action, created = Action.objects.get_or_create(name=name, grant=subscription)
                    current_subscription_actions.add(action)
                    # Обновляем описания действий subscription при необходимости
                    if action.description != "%s (%s)" % (description, service_name):
                        action.description = "%s (%s)" % (description, service_name)
                        action.save()

        # Создаем недостающие макросы
        current_macros_actions = set()  # Сеты текущих привязанных действий и макросов
        current_macroses = set()        # для последующего удаления неактуальных
        for macros_name, (description, meta_grants) in META_MACROSES.iteritems():
            macros, created = Macros.objects.get_or_create(name=macros_name, namespace=namespace)
            current_macroses.add(macros)

            # Обновляем описания макросов при необходимости
            if macros.description != description:
                macros.description = description
                macros.save()

            # Находим гранты для привязки их действий к макросам
            grants = Grant.objects.filter(name__in=meta_grants, namespace=namespace)
            for grant in grants:
                # Находим нужные действия по гранту и названию действия
                actions = Action.objects.filter(name__in=meta_grants[grant.name], grant=grant)
                # Привязываем действия к макросам
                for action in actions:
                    macros.action.add(action)
                    current_macros_actions.add((macros, action))

        # Проверки на удаленные из json данные
        # Находим неактуальные действия для макросов и предлагаем удалить
        delete = ''
        all_macros_actions = set()
        for macros in Macros.objects.filter(namespace=namespace):
            for action in macros.action.all():
                all_macros_actions.add((macros, action))

        obsolete_macros_actions = all_macros_actions - current_macros_actions
        if obsolete_macros_actions:
            self.stdout.write('\nFound obsolete macros grants:\n')
            for m, a in obsolete_macros_actions:
                self.stdout.write('%s: [%s.%s]\n' % (m.name, a.grant.name, a.name))
            if not options['delete']:
                delete = raw_input('\nType "yes" to delete them or hit enter to skip: ')
            if delete.lower() == 'yes' or options['delete']:
                for m, a in obsolete_macros_actions:
                    m.action.remove(a)
                self.stdout.write('Obsolete macros actions have been deleted\n')

        # Находим неактуальные макросы и предлагаем удалить
        obsolete_macroses = set(Macros.objects.filter(namespace=namespace)) - current_macroses
        if obsolete_macroses:
            self.stdout.write('\nFound obsolete macroses:\n')
            for o in obsolete_macroses:
                self.stdout.write('%s: %s\n' % (o.name, o.description))
            if not options['delete']:
                delete = raw_input('\nType "yes" to delete them or hit enter to skip: ')
            if delete.lower() == 'yes' or options['delete']:
                for o in obsolete_macroses:
                    o.delete()
                self.stdout.write('Obsolete macroses have been deleted\n')

        # Находим неактуальные действия у подписки и предлагаем удалить
        if NAMESPACE_NAME == 'passport':
            all_subscription_actions = set(Action.objects.filter(grant__name='subscription',
                                                                 grant__namespace=namespace))
            undivided_actions = set(filter(lambda x: x.grant.name == 'subscription', current_actions))
            obsolete_subscription_actions = all_subscription_actions - current_subscription_actions - undivided_actions
            if obsolete_subscription_actions:
                self.stdout.write('\nFound obsolete subscription actions:\n')
                for o in obsolete_subscription_actions:
                    self.stdout.write('%s: %s\n' % (o.name, o.description))
                if not options['delete']:
                    delete = raw_input('\nType "yes" to delete them or hit enter to skip: ')
                if delete.lower() == 'yes' or options['delete']:
                    for o in obsolete_subscription_actions:
                        o.delete()
                    self.stdout.write('Obsolete subscription actions have been deleted\n')

            # Находим неактуальные действия и предлагаем удалить
            obsolete_actions = (set(Action.objects.filter(grant__namespace=namespace)) - current_actions -
                                current_subscription_actions - obsolete_subscription_actions)
        else:
            obsolete_actions = (set(Action.objects.filter(grant__namespace=namespace)) - current_actions)

        if obsolete_actions:
            self.stdout.write('\nFound obsolete actions:\n')
            for o in obsolete_actions:
                self.stdout.write('%s.%s: %s \n' % (o.grant.name, o.name, o.description))
            if not options['delete']:
                delete = raw_input('\nType "yes" to delete them or hit enter to skip: ')
            if delete.lower() == 'yes' or options['delete']:
                for o in obsolete_actions:
                    o.delete()
                self.stdout.write('Obsolete actions have been deleted\n')

        # Находим неактуальные гранты и предлагаем удалить
        if NAMESPACE_NAME == 'passport':
            obsolete_grants = (set(Grant.objects.filter(namespace=namespace)) - current_grants - {subscription})
        else:
            obsolete_grants = (set(Grant.objects.filter(namespace=namespace)) - current_grants)

        if obsolete_grants:
            self.stdout.write('\nFound obsolete grants:\n')
            for o in obsolete_grants:
                self.stdout.write('%s\n' % o.name)
            if not options['delete']:
                delete = raw_input('\nType "yes" to delete them or hit enter to skip: ')
            if delete.lower() == 'yes' or options['delete']:
                for o in obsolete_grants:
                    o.delete()
                self.stdout.write('Obsolete grants have been deleted\n')

        self.stdout.write('\nDownload finished.\n')
