# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import logging
import socket
from time import sleep

from django.conf import settings
from django.contrib.auth.models import User
from django_docopt_command import DocOptCommand

from passport_grants_configurator.apps.core.exceptions import (
    ExportError,
    NoDiffError,
)

from passport_grants_configurator.apps.core.export_utils import get_exporter
from passport_grants_configurator.apps.core.models import Timestamp
from passport_grants_configurator.apps.core.mail import (
    MAIL_TEMPLATES,
    notify,
)

logger = logging.getLogger(__name__)


class Command(DocOptCommand):
    """Запущенная в продовском окружении выгрузит новые гранты в центральный репозиторий"""
    docs = '''usage: noon'''

    def send_notifications(self, template, context):
        notify(
            template=template,
            recipient_list=settings.EMAIL_NOTIFICATION_TO,
            context=context,
        )

    def handle_docopt(self, arguments):
        logger.info('Starting daily export')
        this_stamp_name = socket.gethostname()
        # Это несовершенный подход к проблеме распределенных блокировок
        # Запишем заявку о себе в таблицу Timestamp
        Timestamp.objects.filter(name=this_stamp_name).delete()
        Timestamp.objects.create(name=this_stamp_name)
        # Ждем, если другие хосты запишут в БД свой Timestamp
        logger.debug('Waiting for peers one minute as %s', this_stamp_name)
        sleep(60)

        # Проверяем, кто последний - тот и выиграл
        if Timestamp.objects.latest('time').name != this_stamp_name:
            logger.debug('This instance have lost right to export')
            return

        logger.debug('This instance have won right to export')
        Timestamp.objects.all().delete()  # Очистим очередь Timestamp

        robot = User.objects.get(username=settings.ROBOT_NAME)

        for project, data in settings.GRANTS_PROJECTS.iteritems():
            logger.debug('Export for %s', project)
            exporter = get_exporter(
                project,
                user=robot,
            )
            try:
                exported = exporter.export()
                if exported['manual_resolve']:
                    self.send_notifications(
                        MAIL_TEMPLATES['manual_resolve'],
                        {
                            'project': project,
                            'manual_resolve': '\n'.join(exported['manual_resolve']),
                        },
                    )
                elif exported['diff']:
                    logger.info('Exported %s with not empty diff', project)
                    self.send_notifications(
                        MAIL_TEMPLATES['networks_changed'],
                        {
                            'project': project,
                            'diff': exported['diff'],
                            'warnings': '\n'.join(exported['warnings']),
                            'export_link': settings.EXPORT_GRANTS_URL,
                        },
                    )
            except NoDiffError:
                continue

            except ExportError as e:
                logger.error('Failed to export for %s: %s', project, e)
                self.send_notifications(
                    MAIL_TEMPLATES['export_failed'],
                    {
                        'project': project,
                        'errors': '\n'.join(e.message),
                    },
                )

        logger.info('All exports are done')
