# -*- coding: utf-8 -*-

from optparse import make_option

import netaddr
from django.core.management.base import BaseCommand, CommandError

from passport_backend_core.grants.apacheconfig import ApacheConfig
from passport_grants_configurator.apps.core.models import (
    Consumer,
    Network,
    Grant,
    Action,
    Namespace,
    ActiveAction,
    Macros,
    ActiveMacros,
    Environment,
    ActiveNetwork,
)


class Command(BaseCommand):
    help = 'Imports apache config file into grants DB'
    option_list = BaseCommand.option_list + (
        make_option('--add_dev_grants',
                    action='store_true',
                    dest='add_dev_grants',
                    default=False,
                    help=u"добавлять ли гранты потребителя для разработки 'dev'"),
        make_option('--add_kopalka_grants',
                    action='store_true',
                    dest='add_kopalka_grants',
                    default=False,
                    help=u"добавлять ли гранты потребителя для копалки 'kopalka'"),
        make_option('-f',
                    '--file',
                    dest='filename',
                    help='Конфиг апача с описанием присвоенных потребителям грантов',
                    metavar='FILE'),
        make_option('-n',
                    '--environment-name',
                    dest='environment_name',
                    help='Имя окружения, для которого следует импортировать гранты (intranet/localhost/stress)',
                    metavar='ENVIRONMENT_NAME'),
        make_option('-t',
                    '--environment-type',
                    dest='environment_type',
                    help='Тип окружения, для которого следует импортировать гранты (development/stress/testing/production)',
                    metavar='ENVIRONMENT_TYPE'),
    )

    def handle(self, *args, **options):
        namespace = Namespace.objects.get(name='passport')

        try:
            environment_name = options['environment_name']
            environment_type = options['environment_type']
        except AttributeError:
            raise CommandError('Пожалуйста, укажите окружение в параметрах (-n имя окружения -t тип окружения)')

        try:
            environment = Environment.objects.get(name=environment_name, type=environment_type)
        except Environment.DoesNotExist:
            raise CommandError('Пожалуйста, укажите существующее окружение')

        macros_names = Macros.objects.filter(namespace=namespace).values_list('name')
        grant_names = Grant.objects.filter(namespace=namespace).values_list('name')

        # инициализируем объект с будущими грантами
        grants = dict()

        for arg, consumer in (('add_dev_grants', 'dev'),
                              ('add_kopalka_grants', 'kopalka')):
            if options[arg]:
                grants[consumer] = {
                    'grants': {
                        'subscription': set(['*']) if consumer == 'dev' else {},
                        'password': set(['*']),
                        'person': set(['*']),
                        'account': set(['*']),
                        'karma': set(['*']),
                        'captcha': set(['*']),
                        'control_questions': set(['*']),
                        'country': set(['*']),
                        'gender': set(['*']),
                        'language': set(['*']),
                        'login': set(['*']),
                        'name': set(['*']),
                        'phone_number': set(['*']),
                        'retpath': set(['*']),
                        'session': set(['*']),
                        'oauth': set(['*']),
                        'statbox': set(['*']),
                        'track': set(['*']),
                        'ignore_stoplist': set(['*']),
                        'social_login': set(['*']),
                        'admchangereg': set(['*']),
                        'hint': set(['*']),
                    },
                    'networks': set(['127.0.0.1']) if consumer == 'dev' else set(['37.140.172.86'])
                }

        # получаем объект-представление конфигов в старом формате
        apache_config = ApacheConfig.parse_file(options['filename'])

        SUPPORT_OLD_GRANTS = [
            'admsubscribe',
            'admloginrule',
            'admblock',
            'admchangereg',
            'pyregistersimple',
            'pyphone',
            'admkarma',
            'admsocialreg',
            'admreg',
            'admsimplereg',
            'pyregisteralternative',
            'pyregisterrequireconfirmedphone',
            'pyregisterphonish',
            'pyphonecopy',
            'pyregisteruncompleted',
        ]

        SUPPORT_OLD_GRANTS_CREATE_TEST_YANDEX_LOGIN = ['internalNetwork', 'yandex-team']
        TEST_YANDEX_LOGIN_CONSUMER_PREFIX = 'create_test_yandex_login_'

        CONSUMER_BY_COMMENT = {
            'zeze': 'yasms', 'dw-spammers': 'narodadmin',
            '_kopalka_': 'kopalka', 'phone-passport-test': 'phone-passport-test',
            'spamstat-in': 'mail', 'shingler-reputation': 'shingler-reputation',
            'sologger.mail.yandex.net': 'sologger.mail.yandex.net',
        }

        def check_printable_ascii(string, additional_info=''):
            if not all(31 < ord(char) < 128 for char in string):
                raise ValueError('Non-printable or non-ASCII symbol in "%s"\n%s' % (string, additional_info))

        def check_network(network, name, values, additional=''):
            try:
                netaddr.IPNetwork(network)  # Проверка валидности IP или сети
            except netaddr.core.AddrFormatError, e:
                information = '\n%s config line: "%s, %s"' % (additional, name, ', '.join(values))
                raise ValueError(e.message + information)

        def subscription_grant(service, grant):
            return {service: grant}

        def mapping_old_to_new_grants(o, service):
            _mapping = {
                'admkarma': {'karma': set(['*'])},
                'admsubscribe': {'subscription': set(['create', 'delete', 'update'])},
                'admloginrule': {'subscription': set(['update'])} if service != 'passport' else {'password': set(['is_changing_required'])},
                'admblock': {'account': set(['is_enabled']), 'karma': set(['*']),
                             'subscription': set(['update'])},
                'admchangereg': {'admchangereg': set(['*']), 'person': set(['*'])},
                'pyregistersimple': {'account': set(['register_simple']),
                                     'track': set(['*']),
                                     'captcha': set(['*']),
                                     'retpath': set(['validate']),
                                     'statbox': set(['*']),
                                     'session': set(['check', 'create']),
                                     'control_questions': set(['*']),
                                     'language': set(['suggest']),
                                     'login': set(['suggest', 'validate']),
                                     'phone_number': set(['validate']),
                                     'password': set(['validate']),
                                     'country': set(['suggest']),
                                     'hint': set(['validate']),
                                     'gender': set(['suggest']),
                                     },
                'pyphone': {'phone_number': set(['update', 'delete'])},
                'admsocialreg': {'account': set(['register_social']),
                                 'track': set(['*']),
                                 'captcha': set(['*']),
                                 'social_login': set(['*']),
                                 },
                'admreg': {'admreg': set(['*'])},
                'admsimplereg': {'admsimplereg': set(['*'])},
                'pyregisteralternative': {'account': set(['register_alternative']),
                                          'track': set(['*']),
                                          'captcha': set(['*']),
                                          'retpath': set(['validate']),
                                          'statbox': set(['*']),
                                          'session': set(['check', 'create']),
                                          'control_questions': set(['*']),
                                          'language': set(['suggest']),
                                          'login': set(['suggest', 'validate']),
                                          'phone_number': set(['validate', 'confirm']),
                                          'password': set(['validate']),
                                          'country': set(['suggest']),
                                          'hint': set(['validate']),
                                          'gender': set(['suggest']),
                                          },
                'pyregisterrequireconfirmedphone': {'account': set(['register_require_confirmed_phone']),
                                                    'track': set(['*']),
                                                    'retpath': set(['validate']),
                                                    'statbox': set(['*']),
                                                    'session': set(['check', 'create']),
                                                    'language': set(['suggest']),
                                                    'login': set(['suggest', 'validate']),
                                                    'phone_number': set(['validate', 'confirm']),
                                                    'password': set(['validate']),
                                                    'country': set(['suggest']),
                                                    'gender': set(['suggest']),
                                                    },
                'pyregisterphonish': {'account': set(['register_phonish']),
                                      'track': set(['*']),
                                      'statbox': set(['*']),
                                      'session': set(['create']),
                                      'oauth': set(['token_create']),
                                      'phone_number': set(['validate', 'confirm']),
                                      },
                'pyphonecopy': {'phone_number': set(['copy'])},
                'pyregisteruncompleted': {
                    'account': set(['register_uncompleted', 'uncompleted_set_password']),
                    'track': set(['*']),
                    'statbox': set(['*']),
                    'login': set(['suggest', 'validate']),
                    'language': set(['suggest']),
                    'country': set(['suggest']),
                    'gender': set(['suggest']),
                    'captcha': set(['*']),
                    'phone_number': set(['validate', 'confirm']),
                    'password': set(['validate']),
                    'oauth': set(['token_create']),
                },
            }

            return _mapping.get(o)

        def mapping_old_grant_create_test_yandex_login_to_new(o):
            _mapping = {
                'internalNetwork': TEST_YANDEX_LOGIN_CONSUMER_PREFIX + 'yndx',
                'yandex-team': TEST_YANDEX_LOGIN_CONSUMER_PREFIX + 'yandex-team',
            }
            return _mapping.get(o)

        def consumer_for_admkarma(old, comment):
            for key, consumer in CONSUMER_BY_COMMENT.items():
                if key in comment:
                    return consumer
            else:
                raise RuntimeError('Unknown comment: %s for old grant %s' % (comment, old))

        def default_consumer_for_old_grant(old, comment):
            return '%s_%s' % (old, comment.split('.', 1)[0])

        def mapping_old_grant_with_comment_to_consumer(old, comment):
            _mapping = {
                'admkarma': consumer_for_admkarma,
                'admreg': default_consumer_for_old_grant,
                'admsimplereg': default_consumer_for_old_grant,
            }
            return _mapping.get(old)(old, comment)

        def add_grants(consumer, ip, old, service=None):
            # по умолчанию полагаем, что консумер совпадает с сервисом
            if service is None:
                service = consumer

            # маппинг старого гранта в новые
            # если грант сохранил свое имя, добавляем так
            try:
                if old == 'admchangereg':  # костыль, чтобы этот грант конвертнулся через mapping_old_to_new_grants
                    raise Grant.DoesNotExist
                if not old in grant_names:
                    raise Grant.DoesNotExist
                old_grants = {old: set(['*'])}
            except Grant.DoesNotExist:
                try:
                    # иначе ищем макрос с таким именем
                    if not old in macros_names:
                        raise Macros.DoesNotExist
                    meta_consumer = grants.setdefault(consumer, {'grants': {}, 'networks': set()})
                    meta_consumer['networks'] |= set([ip])
                    # добавляем этот макрос потребителю
                    meta_consumer.setdefault('macroses', []).append(old)
                    return grants
                except Macros.DoesNotExist:
                    # если и макроса нет, идем как обычно
                    old_grants = mapping_old_to_new_grants(old, service)

            # создаём консумера, если его нет
            if consumer not in grants:
                grants[consumer] = dict()

            # заполняем список ip, подсетей
            if 'networks' not in grants[consumer]:
                grants[consumer]['networks'] = set()

            grants[consumer]['networks'].add(ip)

            # выставляем собственно гранты
            if 'grants' not in grants[consumer]:
                grants[consumer]['grants'] = dict()

            for grant_name, grant_value in old_grants.items():
                if grant_name == 'subscription':
                    # обработка грантов для подписок
                    # 'subscription': {'service_name': ['grant1', ...]}
                    if grant_name not in grants[consumer]['grants']:
                        grants[consumer]['grants'].update(
                            {grant_name: subscription_grant(service, grant_value)}
                        )
                    else:
                        if service not in grants[consumer]['grants'][grant_name]:
                            grants[consumer]['grants'][grant_name].update(
                                subscription_grant(service, grant_value)
                            )
                        else:
                            grants[consumer]['grants'][grant_name][service].update(grant_value)
                else:
                    # остальные гранты вида 'grant': ['grant1', ...]
                    if grant_name not in grants[consumer]['grants']:
                        grants[consumer]['grants'].update({grant_name: grant_value})
                    else:
                        grants[consumer]['grants'][grant_name].update(grant_value)

            return grants

        def parse_old_grant_without_consumer(old_grant):
            # комментарии в секциях присутвуют к группе строке
            comments_with_ips = dict()
            for comment_line in old_grant:
                if comment_line.name.lower() == 'comment':
                    _comment = comment_line.values[0].lower()
                    comments_with_ips[_comment] = []
                    for ip_line in old_grant[old_grant.index(comment_line) + 1:]:
                        if ip_line.name.lower() != 'comment':
                            comments_with_ips[_comment].append(ip_line.values[0])
                        else:
                            break
            return comments_with_ips

        def get_kopalka_ips():
            # получаем список ip-ков для консумера kopalka в каждой секции AllowFrom
            # надо быть очень осторожными, при обновлении грантов
            kopalka_ips = set()
            for allow_from in apache_config.findall('AllowFrom'):
                # исключаем секцию yandex-team
                if allow_from.values[-1].lower() != 'yandex-team':
                    for line in allow_from.children:
                        # собираем ip-адреса только после комментария _KOPALKA_
                        if line.name.lower() == 'comment':
                            comment_name = line.values[0]
                            if comment_name.lower() == '_kopalka_':
                                for ip_line in allow_from.children[allow_from.children.index(line) + 1:]:
                                    if ip_line.name.lower() != 'comment':
                                        ip = ip_line.values[0]
                                        check_network(ip, ip_line.name, ip_line.values)
                                        kopalka_ips.add(ip)
                                    else:
                                        break
            return kopalka_ips

        KOPALKA_IPS = get_kopalka_ips()

        for allow_from in apache_config.findall('AllowFrom'):
            # получаем старый грант
            old = allow_from.values[-1]
            check_printable_ascii(old)

            # обрабатываем старые гранты, которые содержат консумеров
            if old in SUPPORT_OLD_GRANTS:
                for line in allow_from.children:
                    # не смотрим на комментарии к записям
                    if line.name.lower() != 'comment':
                        # ip для гранта
                        ip = line.values[0]
                        check_network(ip, line.name, line.values, 'Old grant: ' + old)

                        if len(line.values) > 1:
                            # Обрабатываем случаи, когда указан service
                            # (он же на данный момент consumer) в старом конфиге
                            services = line.values[1:]
                            for service in services:
                                check_printable_ascii(
                                    service,
                                    'Old grant name: %s, config line: "%s, %s"' % (old, line.name, ", ".join(line.values)))
                                if ip in KOPALKA_IPS:
                                    grants = add_grants('kopalka', ip, old, service)
                                else:
                                    grants = add_grants(service, ip, old)
                        else:
                            # обрабатываем гранты, которые не содержат консумеров
                            # каждый конкретный случай рассматривается отдельно
                            if old in ['admkarma', 'admreg', 'admsimplereg']:
                                comments_ips = parse_old_grant_without_consumer(allow_from.children)
                                for comment, ips in comments_ips.items():
                                    consumer = mapping_old_grant_with_comment_to_consumer(old, comment)
                                    for ip in ips:
                                        check_network(ip, comment, ips, 'Old grant: ' + old)
                                        grants = add_grants(consumer, ip, old)
                            elif old == 'admsocialreg':
                                service = 'social'
                                if ip in KOPALKA_IPS:
                                    grants = add_grants('kopalka', ip, old, service)
                                else:
                                    grants = add_grants(service, ip, old)

            # обрабатываем старые гранты, которые касаются регистрации
            # тестовых яндексовых логинов
            if old in SUPPORT_OLD_GRANTS_CREATE_TEST_YANDEX_LOGIN:
                for line in allow_from.children:
                    if line.name.lower() != 'comment':
                        # ip для гранта
                        ip = line.values[0]
                        check_network(ip, line.name, line.values, 'Old grant: ' + old)

                        # маппинг старого гранта в новые
                        consumer = mapping_old_grant_create_test_yandex_login_to_new(old)

                        # создаём консумера, если его нет
                        if consumer not in grants:
                            grants[consumer] = dict()

                        # заполняем список ip, подсетей
                        if 'networks' not in grants[consumer]:
                            grants[consumer]['networks'] = set()

                        grants[consumer]['networks'].add(ip)

                        # выставляем собственно гранты
                        if 'grants' not in grants[consumer]:
                            grants[consumer]['grants'] = {'create_test_yandex_login': set(['*'])}

        # постпроцессинг: если в грантах есть '*', то оставляем только её, set
        # приводим к list
        def splicing_grant(value):
            if '*' in value and len(value) > 1:
                return ['*']
            return list(value)

        for consumer in grants.keys():
            grants[consumer]['networks'] = list(grants[consumer]['networks'])

            for grant_name, grant_value in grants[consumer]['grants'].items():
                # если выполняется нижеописанное условие, то по значениям этого
                # словаря надо тоже итерировать.
                if grant_name == 'subscription' and isinstance(grant_value, dict):
                    for service_name, service_grant in grant_value.items():
                        grants[consumer]['grants'][grant_name][service_name] = splicing_grant(service_grant)
                else:
                    grants[consumer]['grants'][grant_name] = splicing_grant(grant_value)

        # вписываем дерево найденых грантов в БД грантушки
        for consumer_name, meta_consumer in grants.iteritems():
            consumer, _ = Consumer.objects.get_or_create(name=consumer_name, namespace=namespace)
            for grant_name, meta_grant in meta_consumer['grants'].iteritems():
                actions = []
                if grant_name != 'subscription':
                    if '*' in meta_grant:
                        actions = Action.objects.select_related().filter(grant__name=grant_name,
                                                                         grant__namespace=namespace)
                    else:
                        actions = Action.objects.select_related().filter(name__in=meta_grant,
                                                                         grant__name=grant_name,
                                                                         grant__namespace=namespace)
                else:
                    if '*' in meta_grant:
                        actions.extend(Action.objects.select_related().filter(grant__name='subscription',
                                                                              grant__namespace=namespace))
                    else:
                        for service_name, action_names in meta_grant.iteritems():
                            if '*' in action_names:
                                actions.extend(Action.objects.select_related().filter(grant__name='subscription',
                                                                                      grant__namespace=namespace,
                                                                                      name__startswith=service_name + '.'))
                            else:
                                actions.extend(Action.objects.select_related().filter(grant__name='subscription',
                                                                                      grant__namespace=namespace,
                                                                                      name__in=(service_name + '.' + an for an in action_names)))
                for action in actions:
                    ActiveAction.objects.get_or_create(consumer=consumer, action=action, environment=environment)

            for macros_name in meta_consumer.get('macroses', []):
                macros = Macros.objects.get(name=macros_name, namespace=namespace)
                ActiveMacros.objects.get_or_create(consumer=consumer, macros=macros, environment=environment)
            for network_string in meta_consumer.get('networks', []):
                network, _ = Network.objects.get_or_create(string=network_string)
                ActiveNetwork.objects.get_or_create(consumer=consumer, network=network, environment=environment)
