# -*- coding: utf-8 -*-
from datetime import datetime

from django.db import models
from django.conf import settings
from django.contrib.auth.models import User
from django.contrib.comments.signals import comment_was_posted
from django.contrib.sites.models import get_current_site
from django.core.mail import send_mail

MAX_LENGTH = 64  # Максимальная допустимая длина текстового поля на модели
MAX_DESCRIPTION_LENGTH = 256
MAX_HOSTNAME_LENGTH = 255


class EnvironmentManager(models.Manager):
    def get_by_natural_key(self, name, type):
        return self.get(name=name, type=type)


class Environment(models.Model):
    REVIEW_ISSUE = 'review_environment_issue'

    objects = EnvironmentManager()

    description = models.CharField(max_length=MAX_DESCRIPTION_LENGTH)
    name = models.CharField(max_length=MAX_LENGTH)
    type = models.CharField(max_length=MAX_LENGTH)
    priority = models.IntegerField()

    def natural_key(self):
        return self.name, self.type

    def __unicode__(self):
        return unicode(self.type) if self.name in ('localhost', 'stress') else u'%s %s' % (self.name, self.type)

    class Meta(object):
        permissions = (
            ('review_environment_issue', u'Рассмотреть заявку для окружения'),
        )
        ordering = ['priority']


class NamespaceManager(models.Manager):
    def get_by_natural_key(self, name):
        return self.get(name=name)


class Namespace(models.Model):
    REVIEW_ISSUE = 'review_namespace_issue'

    objects = NamespaceManager()

    name = models.CharField(max_length=MAX_LENGTH)
    environments = models.ManyToManyField(Environment, through='NamespaceEnvironments')

    @property
    def hidden(self):
        return self.name in settings.HIDDEN_NAMESPACES

    def natural_key(self):
        return self.name,

    def __unicode__(self):
        return unicode(self.name)

    class Meta(object):
        permissions = (
            ('review_namespace_issue', u'Рассмотреть заявку для проекта'),
        )
        ordering = ['name']


class NamespaceEnvironments(models.Model):
    """
    Если пользователь имеет права на управление грантами для указанного проекта
    в указанном окружении -- у него должны быть Permission'ы на этот экземпляр модели
    Это реализовано с помощью django_guardian
    """
    PERMISSION = 'core.namespace_environment'

    namespace = models.ForeignKey(Namespace)
    environment = models.ForeignKey(Environment)

    def __unicode__(self):
        return '%s - %s' % (self.namespace, self.environment)

    class Meta(object):
        db_table = 'core_namespace_environments'
        unique_together = ('namespace', 'environment')
        permissions = (
            ('namespace_environment', u'Права на окружения в проекте'),
        )


class NetworkManager(models.Manager):
    def get_by_natural_key(self, string, type):
        return self.get(string=string, type=type)

    def address(self):
        return self.get_query_set().filter(type__in=self.address)

    def host(self):
        return self.get_query_set().filter(type=Network.HOSTNAME)

    def conductor(self):
        return self.get_query_set().filter(type=Network.CONDUCTOR)

    def macros(self):
        return self.get_query_set().filter(type=Network.FIREWALL)


class Network(models.Model):
    """
    Константы Типа_Сети хранятся в классе модели, поскольку это в первую очередь данные
    Это одна буква, которая хранится в БД
    """

    IP = 'I'
    IPNETWORK = 'N'
    IP_TYPES = (IP, IPNETWORK)
    HOSTNAME = 'H'
    CONDUCTOR = 'C'
    FIREWALL = 'F'

    TYPES = (
        (IP, u'IP'),
        (HOSTNAME, u'Hostname'),
        (IPNETWORK, u'IP network'),
        (CONDUCTOR, u'Кондукторная группа'),
        (FIREWALL, u'Фаервольный макрос'),
    )

    objects = NetworkManager()

    string = models.CharField(max_length=250, verbose_name=u'Идентификатор сети')
    type = models.CharField(max_length=2, choices=TYPES, default='H', verbose_name=u'Тип сети')

    def natural_key(self):
        return self.string, self.type

    def __unicode__(self):
        return network_unicode(string=self.string, type_=self.type)

    class Meta(object):
        unique_together = ('string', 'type')


class ConsumerManager(models.Manager):
    def get_by_natural_key(self, namespace_name, name):
        return self.get(namespace__name=namespace_name, name=name)


class Consumer(models.Model):
    objects = ConsumerManager()

    name = models.CharField(max_length=MAX_LENGTH)
    namespace = models.ForeignKey(Namespace)
    description = models.CharField(max_length=MAX_DESCRIPTION_LENGTH)

    def natural_key(self):
        return self.namespace.name, self.name

    def __unicode__(self):
        return u'%s (%s)' % (self.name, self.namespace)

    class Meta(object):
        ordering = ['name']
        verbose_name = u'Потребитель'
        verbose_name_plural = u'Потребители'
        unique_together = ('name', 'namespace')


class ClientManager(models.Manager):
    def get_by_natural_key(self, client_id):
        return self.get(client_id=client_id)


class Client(models.Model):
    name = models.CharField(verbose_name=u'Имя клиента', max_length=MAX_LENGTH)
    client_id = models.BigIntegerField()
    namespace = models.ForeignKey(Namespace, default=12)    # Blackbox by client
    environment = models.ForeignKey(Environment)
    consumer = models.ForeignKey(Consumer, blank=True, null=True)

    def natural_key(self):
        return self.client_id

    def __unicode__(self):
        return u'{name} ({namespace})'.format(
            name=self.name or self.client_id,
            namespace=self.namespace,
        )

    class Meta(object):
        ordering = ['name']
        unique_together = ('client_id', 'namespace', 'environment')


class GrantManager(models.Manager):
    def get_by_natural_key(self, namespace_name, name):
        return self.get(namespace__name=namespace_name, name=name)


class Grant(models.Model):
    objects = GrantManager()

    name = models.CharField(max_length=MAX_LENGTH, verbose_name=u'Имя гранта')
    namespace = models.ForeignKey(Namespace, verbose_name=u'Проект')
    description = models.CharField(max_length=MAX_DESCRIPTION_LENGTH, blank=True, verbose_name=u'Описание гранта')

    def natural_key(self):
        return self.namespace.name, self.name

    def __unicode__(self):
        return u'%s (%s)' % (self.name, self.namespace)

    class Meta(object):
        unique_together = ('name', 'namespace')
        ordering = ['namespace', 'name']


class ActionManager(models.Manager):
    def get_by_natural_key(self, namespace_name, grant_name, name):
        return self.get(grant__namespace__name=namespace_name, grant__name=grant_name, name=name)


class Action(models.Model):
    objects = ActionManager()

    name = models.CharField(max_length=MAX_LENGTH, verbose_name=u'Операция гранта', default='*')
    description = models.CharField(max_length=MAX_DESCRIPTION_LENGTH, verbose_name=u'Описание')
    grant = models.ForeignKey(Grant, verbose_name=u'Имя гранта')
    dangerous = models.BooleanField(default=False, verbose_name=u'Опасный')

    def natural_key(self):
        return self.grant.namespace.name, self.grant.name, self.name

    def __unicode__(self):
        return u'%s.%s (%s)' % (self.grant.name, self.name, self.grant.namespace)

    grammar = {
        'singular': {
            'nominative': u'грант',
            'genitive': u'гранта',
        },
        'plural': {
            'nominative': u'гранты',
            'genitive': u'грантов',
        },
    }

    class Meta(object):
        unique_together = ('name', 'grant')
        ordering = ['grant', 'name']


class MacrosManager(models.Manager):
    def get_by_natural_key(self, namespace_name, name):
        return self.get(namespace__name=namespace_name, name=name)


class Macros(models.Model):
    objects = MacrosManager()

    name = models.CharField(max_length=MAX_LENGTH, verbose_name=u'Имя')
    description = models.CharField(max_length=MAX_DESCRIPTION_LENGTH, verbose_name=u'Описание', blank=True)
    namespace = models.ForeignKey(Namespace, verbose_name=u'Проект')
    action = models.ManyToManyField(Action, verbose_name=u'Грант')

    def natural_key(self):
        return self.namespace.name, self.name

    def __unicode__(self):
        return u'%s (%s)' % (self.name, self.namespace)

    grammar = {
        'singular': {
            'nominative': u'макрос',
            'genitive': u'макроса',
        },
        'plural': {
            'nominative': u'макросы',
            'genitive': u'макросов',
        },
    }

    class Meta(object):
        unique_together = ('name', 'namespace')
        ordering = ['namespace', 'name']


class ActiveMacros(models.Model):
    consumer = models.ForeignKey(Consumer)
    macros = models.ForeignKey(Macros)
    environment = models.ForeignKey(Environment)
    expiration = models.DateField(blank=True, null=True)

    class Meta(object):
        unique_together = ('consumer', 'macros', 'environment')


class ActiveAction(models.Model):
    consumer = models.ForeignKey(Consumer)
    action = models.ForeignKey(Action)
    environment = models.ForeignKey(Environment)
    expiration = models.DateField(blank=True, null=True)

    class Meta(object):
        unique_together = ('consumer', 'action', 'environment')


class ActiveNetwork(models.Model):
    consumer = models.ForeignKey(Consumer)
    network = models.ForeignKey(Network)
    environment = models.ForeignKey(Environment)

    class Meta(object):
        unique_together = ('consumer', 'network', 'environment')


class Email(models.Model):
    address = models.CharField(max_length=MAX_HOSTNAME_LENGTH, primary_key=True)


class Issue(models.Model):
    DRAFT = 'D'
    NEW = 'N'
    APPROVED = 'A'
    REJECTED = 'R'

    TO_CREATE = 'C'
    TO_CLONE = 'CC'
    TO_MODIFY = 'M'
    TO_DELETE = 'D'
    TO_SET_EXPIRATION = 'E'

    STATUSES = (
        (DRAFT, u'черновик'),
        (NEW, u'новая'),
        (APPROVED, u'одобрена'),
        (REJECTED, u'отклонена'),
    )
    TYPES = (
        (TO_CREATE, u'Создание потребителя'),
        (TO_CLONE, u'Клонирование потребителя'),
        (TO_MODIFY, u'Изменение потребителя'),
        (TO_DELETE, u'Удаление потребителя'),
        (TO_SET_EXPIRATION, u'Присвоение временных грантов'),
    )

    creator = models.ForeignKey(User, related_name='creator')
    creation_date = models.DateTimeField(auto_now_add=True)
    expiration = models.DateField(blank=True, null=True)

    approving_person = models.ForeignKey(User, related_name='approving_person', blank=True)
    approving_date = models.DateTimeField(auto_now=True)

    add_action = models.ManyToManyField(Action, related_name='issue_add', blank=True)
    del_action = models.ManyToManyField(Action, related_name='issue_del', blank=True)
    add_macros = models.ManyToManyField(Macros, related_name='issue_add', blank=True)
    del_macros = models.ManyToManyField(Macros, related_name='issue_del', blank=True)
    add_network = models.ManyToManyField(Network, through='AddNetworks', related_name='issue_add', blank=True)
    del_network = models.ManyToManyField(Network, through='DelNetworks', related_name='issue_del', blank=True)

    consumer = models.ForeignKey(Consumer, blank=True, null=True)
    consumer_name = models.CharField(max_length=MAX_LENGTH, blank=True)
    consumer_name_new = models.CharField(max_length=MAX_LENGTH, blank=True)
    consumer_description = models.CharField(max_length=MAX_DESCRIPTION_LENGTH, blank=True)

    clients = models.ManyToManyField(Client, blank=True, null=True)

    namespace = models.ForeignKey(Namespace)
    environments = models.ManyToManyField(Environment, related_name='environments')

    # TODO: editable=False
    status = models.CharField(max_length=2, choices=STATUSES, default=NEW)
    type = models.CharField(max_length=2, choices=TYPES, default=TO_MODIFY)

    emails = models.ManyToManyField(Email, blank=True, null=True)

    class Meta(object):
        ordering = ['-creation_date']

    def get_consumer_name(self):
        if self.type in (self.TO_CREATE, self.TO_CLONE):
            return self.consumer_name
        elif self.type in (self.TO_MODIFY, self.TO_SET_EXPIRATION) and self.id:
            return self.consumer.name
        return u''

    def get_consumer_description(self):
        if self.type in (self.TO_CREATE, self.TO_CLONE):
            return self.consumer_description
        elif self.type in (self.TO_MODIFY, self.TO_SET_EXPIRATION) and self.id:
            return self.consumer.description
        return u''

    def add_action_id(self):
        return self.add_action.values_list('id', flat=True)

    def del_action_id(self):
        return self.del_action.values_list('id', flat=True)

    def add_macros_id(self):
        return self.add_macros.values_list('id', flat=True)

    def del_macros_id(self):
        return self.del_macros.values_list('id', flat=True)

    def add_network_id(self):
        return self.add_network.values_list('id', flat=True)

    def del_network_id(self):
        return self.del_network.values_list('id', flat=True)

    def has_addition(self):
        return self.add_action.count() or self.add_macros.count() or self.add_network.count()

    def has_removal(self):
        return self.del_action.count() or self.del_macros.count() or self.del_network.count()

    def reject(self, user):
        self.status = self.REJECTED
        self.approving_date = datetime.now()
        self.approving_person = user
        self.save()

    def approve(self, user):
        self.status = self.APPROVED
        self.approving_date = datetime.now()
        self.approving_person = user
        self.save()


class IssueNetworks(models.Model):
    issue = models.ForeignKey(Issue)
    network = models.ForeignKey(Network)
    environment = models.ForeignKey(Environment)

    class Meta(object):
        abstract = True
        unique_together = ('issue', 'network', 'environment')
        ordering = ('issue', 'environment')


class DelNetworks(IssueNetworks):
    pass


class AddNetworks(IssueNetworks):
    pass


class PreviousResolve(models.Model):
    """
    Результат резолвинга макросов и кондукторных групп час назад
    Чистится командой hourly по наличию потребителей у сети
    """
    network = models.ForeignKey(Network, unique=True)
    children = models.TextField()


class Timestamp(models.Model):
    """
    Эта модель используется только для блокировки выполнения задачи noon на одном из хостов
    Задача выполняет симуляцию выгрузки грантов для всех проектов
    """
    time = models.DateTimeField(auto_now=True)
    name = models.CharField(max_length=MAX_HOSTNAME_LENGTH)


# TODO: Вынести и проверить
def __comment_notification(request, **kwargs):
    sender = request.user
    issue_id = request.POST['object_pk']
    issue = Issue.objects.get(pk=issue_id)

    if sender == issue.creator:
        if issue.status == issue.NEW:
            return
        else:
            to = [issue.approving_person.email]
    else:
        to = [issue.creator.email]

    subject = u'(Grant Configurator) %s прокомментировал(а) заявку №%s' % (
        sender.username,
        issue_id,
    )
    link = 'https://%s/grants/issue/%s/' % (get_current_site(request).domain, issue_id)
    message = u'%s оставил(а) комментарий к заявке №%s:\n%s\n%s' % (
        sender.username,
        issue_id,
        request.POST['comment'],
        link,
    )
    send_mail(
        subject,
        message,
        settings.EMAIL_NOTIFICATION_FROM,
        to,
        fail_silently=True,
    )

comment_was_posted.connect(__comment_notification)


# TODO: Вместо всего этого балагана можно пользовать pymorphy со своим словарем
NETWORK_TYPE_NAMES = {
    Network.IP: {
        'singular': {
            'nominative': u'IP адрес',
            'genitive': u'IP адреса',
            'accusative': u'IP адрес',
            'instrumental': u'IP адресом',
        },
    },
    Network.IPNETWORK: {
        'singular': {
            'nominative': u'сеть',
            'genitive': u'сети',
            'accusative': u'сеть',
            'instrumental': u'сетью',
        }
    },
    Network.HOSTNAME: {
        'singular': {
            'nominative': u'хост',
            'genitive': u'хоста',
            'accusative': u'хост',
            'instrumental': u'хостом',
        }
    },
    Network.CONDUCTOR: {
        'singular': {
            'nominative': u'кондукторная группа',
            'genitive': u'кондукторной группы',
            'accusative': u'кондукторную группу',
            'instrumental': u'кондукторной группой',
        }
    },
    Network.FIREWALL: {
        'singular': {
            'nominative': u'фаервольный макрос',
            'genitive': u'фаервольного макроса',
            'accusative': u'фаервольный макрос',
            'instrumental': u'фаервольным макросом',
        }
    }
}


def network_unicode(string, type_, number='singular', case='nominative'):
    return u'%s %s' % (NETWORK_TYPE_NAMES[type_][number][case], string)


__all__ = (
    'Environment',
    'Namespace',
    'NamespaceEnvironments',
    'Network',
    'Consumer',
    'Client',
    'Grant',
    'Action',
    'Macros',
    'ActiveNetwork',
    'ActiveAction',
    'ActiveMacros',
    'Issue',
    'IssueNetworks',
    'AddNetworks',
    'DelNetworks',
    'Timestamp',
)
