# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import json

from dns.exception import DNSException
from dns.resolver import Resolver

import ujson
import re
import string
import logging
import socket
import traceback

from collections import defaultdict
from operator import itemgetter

from netaddr import (
    IPNetwork,
    IPAddress,
    AddrFormatError,
)
from django.conf import settings

from passport_backend_core.builders.staff.staff import BaseStaffError

from .models import (
    Network,
    PreviousResolve,
)

from .caching import (
    cache_call,
    get_cached_network,
    cache_network,
    macro_key_getter,
    c_group_key_getter,
)
from .exceptions import (
    APIRequestFailedError,
    NotFoundError,
    NetworkResolveError,
    UnknownNetworkTypeError,
    NoDataError,
    HostResolvingFailed,
)
from .utils import (
    request_api,
    yesno_to_bool,
    is_decreased_by_trypo,
)
from passport_grants_configurator.apps.core.common import get_staff

logger = logging.getLogger(__name__)

YANDEX_ROOT_ZONES = ('ru', 'com', 'net', 'tr')
IP_ADDRESS_CHAR_SET = set(string.hexdigits) | set('.:')  # Цифры, буквы и разделители
HOSTNAME_CHAR_SET = set(string.ascii_letters) | set(string.digits) | set('._-')

REQUEST_FAILED_MESSAGE_TEMPLATE = u'Не удалось получить %s из %s: %s'
NO_DATA_MESSAGE_TEMPLATE = u'Пустой ответ для %s'

# Машины Грантушки за резолвом хостов настроены ходить в dns64-cache.yandex.ru,
# где ipv4-only хосты получают фейковые ipv6 адреса. В приложении они не нужны.
# Ходим в правильный DNS: ns-cache.yandex.ru
resolver = Resolver(configure=False)
resolver.nameservers = [settings.NS_CACHE_DNS_IP]


def _netaddr_validated(function):
    """Используется только для определения ip-адресов и ip-сетей"""
    def wrapper(*args, **kwargs):
        try:
            return function(*args, **kwargs)
        except (AddrFormatError, ValueError):
            text = args[0]
            logger.debug('IP address format error for %r', text)
            return
    return wrapper


def _pass_on_exceptions(*exceptions):
    """Используется только для определения типа сети"""

    def decorator(function):
        def wrapper(*args, **kwargs):
            try:
                return function(*args, **kwargs)
            except exceptions as ex:
                address = args[0]
                logger.debug('Exception %s with resolving %r in %s - skip it', ex, address, function.__name__)
        return wrapper

    return decorator


def _looks_like_macro(text):
    """Файрвольный макрос имеет следующий вид _MACRONAME_"""
    return len(text) > 2 and text[0] == text[-1] == '_'


def _looks_like_conductor_group(text):
    """Кондукторная группа имеет следующий вид: %passport-front-stable"""
    return text.startswith('%') and text.islower()


def looks_like_old_conductor_macro(text):
    return _looks_like_macro(text) and text.upper().startswith('_C_')


def _looks_like_ip_address(text):
    """
    Без регулярок проверим что строка похожа на ip-адрес v4 или v6
     * Должны быть разделительные символы,
     * Не должно быть букв кроме A-F
    """
    if not ('.' in text or ':' in text):
        return False

    return IP_ADDRESS_CHAR_SET.issuperset(text)


def _looks_like_ip_network(text):
    """IP-сеть всегда содержит символ слэш"""
    return '/' in text


def _looks_like_trypo_support_network(text):
    """IPv6-сеть вида 411a@2a02:6b8:c00::/40"""
    return bool(re.match(ur'[0-9a-f]{1,8}@[0-9a-f:/]+', text.lower()))


def _looks_like_hostname(text):
    """Имя узла всегда содержит точку и заканчивается на имя домена верхнего уровня"""
    if '.' not in text:
        return False
    if not HOSTNAME_CHAR_SET.issuperset(text):
        return False
    root = text.split('.')[-1]
    if root in YANDEX_ROOT_ZONES:
        return True

    return False


def looks_like_server_macro(text):
    """
    Макросы серверов, которые резолвятся в хосты и требуют
    проверки на права как кондукторные группы (по каждому хосту)
    """
    return text.lower().strip('_').endswith('srv')


def is_valid_address(network_string):
    """
    Можно предварительно проверить адреса-строки на ошибки
     * строки, которые не кодируются в idna(содержат две точки подряд)
    """
    try:
        network_string.encode('idna')
    except UnicodeError:
        # FIXME: Подменяем неправильный адрес на пустой, т.к. обработка исключений пока хромает
        return False

    return True


def getaddrinfo(address):
    """
    :param address: Имя узла
    :return: Список ip-адресов для узла
    :raise: DNSException
    """
    ipv4_data = resolver.query(address, 'A', raise_on_no_answer=False, tcp=True)
    ipv4_ips = [] if not ipv4_data.rrset else [ip.to_text() for ip in ipv4_data]
    ipv6_data = resolver.query(address, 'AAAA', raise_on_no_answer=False, tcp=True)
    ipv6_ips = [] if not ipv6_data.rrset else [ip.to_text() for ip in ipv6_data]
    return ipv6_ips + ipv4_ips


def getnameinfo(address):
    """
    :param address: Адрес узла
    :return: Имя узла
    """
    host, port = socket.getnameinfo((address, 80), 0)
    return host


def getipsfromaddr(address):
    """Если адрес валидный, попробуем разрешить его в ip-адреса, иначе вернем пустой список"""
    if is_valid_address(address):
        return getaddrinfo(address)
    else:
        return list()


class NetworkResolver(object):
    """
    Определяем тип сетевого объекта
    """
    @classmethod
    def get_children_getter(cls, network_type):
        return {
            Network.FIREWALL: expand_firewall_macro,
            Network.HOSTNAME: NetworkResolver.get_host_ips_with_retries,
            Network.CONDUCTOR: get_conductor_group_hosts,
        }.get(network_type)

    @classmethod
    def get_children_type(cls, network_type):
        # Знаем, что функция получения потомков по имени хоста возвращает ip-адреса,
        # А функция получения потомков по имени кондукторной группы возвращает хосты
        return {
            Network.HOSTNAME: Network.IP,
            Network.CONDUCTOR: Network.HOSTNAME
        }.get(network_type)

    @staticmethod
    @_netaddr_validated
    def is_ip(network_string):
        if _looks_like_ip_address(network_string):
            return bool(IPAddress(network_string))

    @staticmethod
    @_netaddr_validated
    def is_ipnetwork(network_string):
        if _looks_like_trypo_support_network(network_string):
            return bool(IPNetwork(network_string.split('@')[1]))
        if _looks_like_ip_network(network_string):
            return bool(IPNetwork(network_string))

    @classmethod
    def is_hostname(cls, network_string):
        return _looks_like_hostname(network_string)

    @staticmethod
    def is_macro(network_name):
        # Быстро выходим если передали что-то непохожее на макрос
        if not _looks_like_macro(network_name):
            return

        # Или попробуем получить состав этого макроса
        if expand_firewall_macro(network_name):
            return True

    @staticmethod
    def get_host_by_ip(ip_address):
        try:
            logger.debug('Getting host by ip %s', ip_address)
            return getnameinfo(ip_address)

        except socket.error:
            raise NetworkResolveError(u'Не удалось получить хост для ip %s' % ip_address)

    # TODO: Эту функцию нужно убрать и использовать везде get_type_and_children()
    @classmethod
    def get_type(cls, network_name):
        network_type, _ = cls.get_type_and_children(network_name)
        return network_type

    @classmethod
    def get_type_and_children(cls, network_name):
        """
        Определим тип объекта. Так как определение актуального макроса или кондукторной группы
        происходит через получение списка его потомков через внешнее АПИ, лучше сохранять результат вызова.
        В первую очередь проверяем макрос, тк макрос можно быстро отличить от адреса.

        Раньше определение типа сетевого объекта работало в следующим образом:
          - Это файрвольный макрос? (удается ли успешно раскрыть макрос в список имен?)
          - Это ip-сеть? (удается ли успешно создать объект netaddr.IPNetwork?)
          - Это ip-адрес? (удается ли успешно создать объект netaddr.IPAddress?)
          - Это имя хоста? (удается ли успешно получить ip-адреса хоста?)
          - Если ничего из вышеследующего -- ошибка
        Я добавил в каждую проверку быстрый тест -- содержит ли строка все необходимые символы?
        не содержит ли строка недопустимые символы? При этом не нужно получать ip-адрес узла для того,
        чтобы понять, что это имя узла.
        Отдельная задача - получение потомков текущего сетевого объекта. При это операции
        происходит запрос в dns

        :param network_name: Строка, представляющая сетевой объект
        :return: Кортеж из двух элементов
          - Тип сетевого объекта
          - Список имен (строк) объектов-потомков
        """
        if _looks_like_macro(network_name):
            children = expand_firewall_macro(network_name)
            if children:
                return Network.FIREWALL, children

        elif _looks_like_conductor_group(network_name):
            children = get_conductor_group_hosts(network_name)
            return Network.CONDUCTOR, children

        elif cls.is_ip(network_name):
            return Network.IP, None

        elif cls.is_ipnetwork(network_name):
            return Network.IPNETWORK, None

        elif cls.is_hostname(network_name):
            ip_list = cls.get_host_ips_with_retries(network_name)
            if ip_list:
                return Network.HOSTNAME, ip_list
            else:
                raise HostResolvingFailed(network_name)

        # Пришел какой-то мусор(кириллица, знаки припенания)
        message = 'Can\'t detect network object type: %s' % network_name
        logger.debug(message)
        raise UnknownNetworkTypeError('Can\'t detect network object type: %s' % network_name)

    @classmethod
    def split_ip_and_other(cls, names):
        """Отделим ip-адреса и ip-сети от всего остального"""
        ips, other = list(), list()

        for name in names:
            is_ip = cls.is_ip(name) or cls.is_ipnetwork(name)
            if is_ip:
                ips.append(name)
            else:
                other.append(name)

        return ips, other

    @classmethod
    def try_resolve(cls, names, method, error_class):
        resolved, unresolved = set(), set()

        for name in names:
            try:
                resolved.update(
                    method(name),
                )
            except error_class:
                unresolved.add(name)

        return list(resolved), list(unresolved)

    @classmethod
    def get_clean_ips_and_excluded_nets(cls, ips):
        """
        Удаляет IP-адреса и сети, в которые не разрешено
        резолвиться макросам и хостам
        :type ips: list
        :rtype: tuple
        """
        forbidden_addresses = {'::1', '0.0.0.0/0', '::/0'}
        for_exclusion = set()
        clean_ips = []
        for address in ips:
            if address in forbidden_addresses or address.startswith('127.'):
                for_exclusion.add(address)
                continue
            clean_ips.append(address)
        return clean_ips, list(for_exclusion)

    @classmethod
    def get_children_if_decreased(cls, network, current_children):
        """
        Выдаем предыдущий состав сети, если он уменьшился
        :type network: Network
        :type current_children: list
        :rtype: list
        """
        try:
            previous_resolved = PreviousResolve.objects.get(network=network)
        except PreviousResolve.DoesNotExist:
            previous_resolved = None
        if not previous_resolved or not previous_resolved.children:
            logger.error('No previous resolve data for %s', network.string)
            return []
        prev_children = ujson.loads(previous_resolved.children)
        if (
                len(prev_children) * settings.ANOREXIA_THRESHOLD > len(current_children) and
                not is_decreased_by_trypo(prev_children, current_children)
        ):
            return prev_children
        return []

    @classmethod
    def get_ips(cls, network):
        """
        Используется при экспорте грантов.
        Возвращает кортеж из двух объектов:
         - список адресов и сетей, которые удалось разрешить
         - словарь со списком имен, для которых не удалось определить ip-адреса,
         списком исключенных ip-адресов, словарем с данными сетей, размер которых
         сильно уменьшился после обновления
        """
        unresolved_data = {
            'manual_resolve': [],
            'auto_resolve': [],
            'excluded': [],
            'decreased': {},
        }
        if network.type in Network.IP_TYPES:
            return [network.string], unresolved_data

        elif network.type == Network.FIREWALL:
            try:
                expanded_macro = expand_firewall_macro(network.string)
            except NotFoundError:
                unresolved_data['manual_resolve'].append(network.string)
                return [], unresolved_data

            decreased_children = cls.get_children_if_decreased(network, expanded_macro)
            if decreased_children:
                unresolved_data['decreased'].update({
                    'name': network.string,
                    'old': decreased_children,
                    'new': expanded_macro,
                })

            ips, other = cls.split_ip_and_other(expanded_macro)
            resolved, unresolved = cls.try_resolve(
                other,
                cls.get_host_ips_with_retries,
                NetworkResolveError,
            )
            ips.extend(resolved)
            ips, excluded = cls.get_clean_ips_and_excluded_nets(ips)
            unresolved_data['auto_resolve'].extend(unresolved)
            unresolved_data['excluded'].extend(excluded)
            return ips, unresolved_data

        if network.type == Network.CONDUCTOR:
            try:
                hosts = get_conductor_group_hosts(network.string)
            except NotFoundError:
                unresolved_data['manual_resolve'].append(network.string)
                return [], unresolved_data

            decreased_children = cls.get_children_if_decreased(network, hosts)
            if decreased_children:
                unresolved_data['decreased'].update({
                    'name': network.string,
                    'old': decreased_children,
                    'new': hosts,
                })

            resolved, unresolved = cls.try_resolve(
                hosts,
                cls.get_host_ips_with_retries,
                NetworkResolveError,
            )
            resolved, excluded = cls.get_clean_ips_and_excluded_nets(resolved)
            unresolved_data['auto_resolve'].extend(unresolved)
            unresolved_data['excluded'].extend(excluded)
            return resolved, unresolved_data

        elif network.type == Network.HOSTNAME:
            hosts = [network.string]

        else:
            raise UnknownNetworkTypeError()

        resolved, unresolved = cls.try_resolve(
            hosts,
            cls.get_host_ips_with_retries,
            NetworkResolveError,
        )
        resolved, excluded = cls.get_clean_ips_and_excluded_nets(resolved)
        unresolved_data['auto_resolve'].extend(unresolved)
        unresolved_data['excluded'].extend(excluded)
        return resolved, unresolved_data

    @staticmethod
    def get_host_ips_with_retries(host):
        """
        Делаем запрос в dns и получаем адреса хоста
        Пробуем несколько раз. Если не получилось - бросим ошибку
        :param host: Имя узла
        :return: Список строк - ip-адресов
        """
        retry_left = attempts = settings.SOCKET_RETRY
        while retry_left:
            try:
                ips = getipsfromaddr(host)
                logger.debug('Host %r got ips: %s', host, ips)
                return ips
            except DNSException:
                retry_left -= 1
                logger.error(
                    'IP resolving failed for %s [%d/%d]; Traceback: %s',
                    host,
                    retry_left,
                    attempts,
                    traceback.format_exc(),
                )
            except Exception as ex:
                logger.exception('Can\'t resolve %r ips: %s', host, ex)
                raise

        raise HostResolvingFailed(u'Не удается получить ip для хоста %s' % host)

    @classmethod
    def check_permissions(cls, network_type, network_string, login):
        if network_type == Network.IP:
            return False, u'Не могу проверить права на ip-адрес %s' % network_string

        elif network_type == Network.IPNETWORK:
            # Запрашиваем права на эту сеть для этого пользователя
            permission_list = check_ipnetworks_permissions(login, network_string)
            _, is_allowed = permission_list[0]
            message = u'Нет прав на сеть %s' % network_string
            return is_allowed, message

        elif network_type == Network.HOSTNAME:
            try:
                responsible_people = get_hostname_responsible_people(network_string)
            except NoDataError:
                is_allowed = False
            else:
                is_allowed = (login in responsible_people)
            return is_allowed, u'Нет прав на хост %s' % network_string

        elif network_type == Network.FIREWALL and not looks_like_server_macro(network_string):
            admins_only, responsible = get_macro_responsible(network_string)
            is_allowed = (
                admins_only and login in settings.PASSPORT_TEAM_ADMIN or
                login in responsible.get('people', set())
             )
            return is_allowed, u'Нет прав на макрос %s' % network_string

        elif network_type in [Network.CONDUCTOR, Network.FIREWALL]:
            errors = []
            items = cls.get_children_getter(network_type)(network_string)
            network_permission_check_list = [
                cls.check_permissions(cls.get_type(network), network, login)
                for network in items
            ]
            is_allowed = all(
                map(
                    itemgetter(0),  # Возьмем результаты проверок
                    network_permission_check_list,
                ),
            )
            if not is_allowed:
                errors = map(
                    itemgetter(1),  # Выберем сообщения об ошибках
                    filter(
                        lambda item: not bool(item[0]),  # Выберем только ошибки
                        network_permission_check_list,
                    ),
                )
            return is_allowed, errors

        else:
            message = u'Для сети %s указан неизвестный тип %s' % (network_string, network_type)
            raise UnknownNetworkTypeError(message)


def parse_list_of_lines(response_raw, separator='\n'):
    """Разбивает текст ответа АПИ Кондуктора или Racktables в список имен узлов"""
    if not response_raw:
        return []
    return response_raw.strip().split(separator)


def parse_conductor_groups_json_to_list(response_json):
    response_data = ujson.loads(response_json)
    return ['%' + r['name'] for r in response_data]


def format_request_error(obj, target, message):
    return REQUEST_FAILED_MESSAGE_TEMPLATE % (
        obj,
        target,
        message,
    )


def format_nodata_error(obj):
    return NO_DATA_MESSAGE_TEMPLATE % (
        obj
    )


def get_native_conductor_group_name(name):
    if _looks_like_conductor_group(name):
        return name[1:]
    return name


@cache_call(key_getter=c_group_key_getter)
def get_conductor_group_hosts(group):
    logger.debug('Request Conductor-API for group %s', group)
    try:
        hosts_raw = request_api(
            settings.CONDUCTOR_API,
            'groups2hosts/%s' % get_native_conductor_group_name(group),
        )
    except APIRequestFailedError as ex:
        raise APIRequestFailedError(format_request_error(group, 'Кондуктора', ex.message))

    return parse_list_of_lines(hosts_raw)


@cache_call(key_getter=macro_key_getter)
def expand_firewall_macro(macro_name):
    """
    Получает список сетей, включая trypo, по имени макроса
    :param macro_name: Имя макроса для раскрытия
    :return: Список имен сетевых объектов
    :raises: (NotFoundError, APIRequestFailedError)
    """
    logger.debug('Request HBF for macro %s', macro_name)
    try:
        objects_json = request_api(
            settings.HBF_API,
            'macros/%s' % macro_name.upper(),
        )
    except APIRequestFailedError as ex:
        raise APIRequestFailedError(format_request_error(macro_name, 'HBF', ex.message))

    return ujson.loads(objects_json)


def request_firewall_macro_list():
    """Запрашивает актуальный список известных сетей из hbf"""
    try:
        objects_json = request_api(
            settings.HBF_API,
            'enum-macros',
        )
    except APIRequestFailedError as ex:
        raise APIRequestFailedError(format_request_error(u'список макросов', 'HBF', ex.message))

    if not objects_json:
        raise NoDataError(format_nodata_error(u'списка известных макросов'))

    return ujson.loads(objects_json)


def request_conductor_groups_list():
    """Запрашивает актуальный список кондукторных групп из http://c.yandex-team.ru/api/"""
    try:
        conductor_groups_json = request_api(
            settings.CONDUCTOR_API,
            'groups',
            params={'format': 'json'},
        )
    except APIRequestFailedError as ex:
        raise APIRequestFailedError(format_request_error(u'список групп', 'conductor', ex.message))

    if not conductor_groups_json:
        # Кондукторные группы пропали из Кондуктора
        raise NoDataError(format_nodata_error(u'списка известных кондукторных групп'))

    try:
        conductor_groups_list = parse_conductor_groups_json_to_list(conductor_groups_json)
    except (ValueError, KeyError) as ex:
        logger.error('Error while parsing conductor groups %s', ex)
        raise NoDataError(format_nodata_error(u'списка известных кондукторных групп'))

    return conductor_groups_list


def parse_ipnetwork_permission(line):
    """
    Разбивает ответ о правах пользователя
    >>> '127.0.0.1\tyes'
    <<< ('127.0.0.1', True)
    """
    name, granted = line.split('\t', 1)
    return name, yesno_to_bool(granted)


def parse_ipnetworks_permissions(response_raw):
    return map(
        parse_ipnetwork_permission,
        parse_list_of_lines(response_raw),
    )


def parse_macro_responsible(response_raw):
    """
    :type response_raw: str
    :rtype: dict
    :raises: APIRequestFailedError
    """
    try:
        rv = json.loads(response_raw)
    except (ValueError, TypeError) as e:
        raise APIRequestFailedError(u'Malformed JSON: %s' % e)

    if rv['status'] != 'success' or 'responsibles' not in rv:
        raise APIRequestFailedError('Bad response: %s' % rv)

    responsible = {
        'people': set([
            item.strip('%')
            for item in rv['responsibles']
        ])
    }
    admins_only = False

    return admins_only, responsible


def get_macro_responsible(macro):
    """
    Документация: https://st.yandex-team.ru/PUNCHER-736#5c9cda154eedbc001f05674b
    :rtype: tuple
    """
    try:
        response_raw = request_api(
            settings.PUNCHER_API,
            'dynfw/responsibles',
            params={'q': macro},
            headers={'Authorization': 'OAuth %s' % settings.PUNCHER_OAUTH_TOKEN},
        )
    except APIRequestFailedError as ex:
        raise APIRequestFailedError(format_request_error(macro, u'puncher', ex.message))

    response_raw = response_raw.strip()

    return parse_macro_responsible(response_raw)


def parse_user_staff_groups(user_groups):
    """
    :type user_groups: list
    :rtype: list
    """
    flatten_groups = set()
    for user_group in user_groups:
        user_group = user_group.get('group', {})
        flatten_groups.add(user_group.get('url'))
        ancestors = {ancestor.get('url') for ancestor in user_group.get('ancestors', [])}
        flatten_groups.update(ancestors)
    return [group for group in flatten_groups if group]


def check_staff_groups(staff_groups, login):
    """
    :type staff_groups: set
    :param login: str
    :rtype: bool
    """
    staff_api = get_staff(settings.STAFF_API_URL)

    try:
        user_groups = staff_api.get_user_groups(
            oauth_token=settings.STAFF_OAUTH_TOKEN,
            login=login,
        )
    except BaseStaffError as e:
        logger.error('Request to Staff failed: {}'.format(e.message))
        raise APIRequestFailedError(e.message)

    parsed_user_groups = parse_user_staff_groups(user_groups)
    return bool(staff_groups & set(parsed_user_groups))


def check_ipnetworks_permissions(login, ip_networks):
    """
    Ходит в racktables
    Возвращает список кортежей с разрешениями на сети для указанного пользователя, например,
    [
        ('37.140.181.0/28', True),
        ('87.250.232.64/27', False)
    ]
    Здесь пользователь имеет права только на сеть '37.140.181.0/28'
    """
    if isinstance(ip_networks, list):
        ip_networks = ','.join(ip_networks)

    try:
        response_raw = request_api(
            settings.RACKTABLES_API,
            'golem-net-resp-check.php',
            params={'login': login, 'nets': ip_networks},
        )
    except APIRequestFailedError as ex:
        raise APIRequestFailedError(format_request_error(ip_networks, 'golem-racktables', ex))

    if not response_raw:
        raise NoDataError(format_nodata_error(ip_networks))

    return parse_ipnetworks_permissions(response_raw)


def process_golem_response(response_raw):
    return parse_list_of_lines(response_raw, ',')


def get_hostname_responsible_people(hostname):
    try:
        response_raw = request_api(
            settings.GOLEM_API,
            'get_host_resp.sbml',
            params={'hostname': hostname},
        )
    except APIRequestFailedError as ex:
        raise APIRequestFailedError(format_request_error(hostname, u'Голема', ex.message))

    if not response_raw.strip():
        raise NoDataError(u'Не найден список ответственных для хоста %s' % hostname)

    return process_golem_response(response_raw)


def get_hostname_responsible_people_mails(hostname):
    try:
        response_raw = request_api(
            settings.GOLEM_API,
            'get_host_resp.sbml',
            params={'output': 'email-short', 'hostname': hostname},
        )
    except APIRequestFailedError as ex:
        raise APIRequestFailedError(format_request_error(hostname, u'Голема', ex.message))

    if not response_raw.strip():
        raise NoDataError(format_nodata_error(hostname))

    return process_golem_response(response_raw)


class NetworkCacheManager(object):
    """
    Управляет кэшем сетевых объектов
    Позволяет получить и сохранить в кэше
      * Все известные сетевые объекты
      * Все известные ip-сети и адреса
      * Все известные сетевые макросы
      * Все известные кондукторные группы
    """

    @classmethod
    def all_networks(cls):
        return get_cached_network('known_network_list')

    @classmethod
    def save_all_networks(cls, value):
        return cache_network('known_network_list', value)

    @classmethod
    def active_network_values(cls):
        """
        Получить из кеша список JSON-like сетей, привязанных к какому-нибудь потребителю
        Если в кэше пусто, вычислить и закэшировать
        """
        cached = get_cached_network('known_network_values')

        if cached is None:
            cached = cls.calc_active_network_values()
            cls.save_all_networks_values(cached)

        return cached

    @classmethod
    def calc_active_network_values(cls):
        """Выбираем всю таблицу сетевых объектов"""
        return list(Network.objects.exclude(activenetwork=None).values())

    @classmethod
    def save_all_networks_values(cls, value):
        """Кэшируем переданное значение в redis"""
        return cache_network('known_network_values', value)

    @classmethod
    def ip_networks(cls):
        return get_cached_network('known_ip_network_list')

    @classmethod
    def save_ip_networks(cls, value):
        return cache_network('known_ip_network_list', value)

    @classmethod
    def all_macros_list(cls):
        """Возьмем из кэша или из racktables и положим в кэш"""
        cached = get_cached_network('all_macro_list')

        if cached is None:
            cached = cls.calc_all_macros_list()
            cls.save_all_macros_list(cached)

        return cached

    @classmethod
    def calc_all_macros_list(cls):
        """Выберем все известные макросы из racktables"""
        return request_firewall_macro_list()

    @classmethod
    def save_all_macros_list(cls, value):
        return cache_network('all_macro_list', value)

    @classmethod
    def calc_all_conductor_groups_list(cls):
        """Выберем все известные макросы из racktables"""
        return request_conductor_groups_list()

    @classmethod
    def save_all_conductor_groups_list(cls, value):
        return cache_network('all_conductor_groups_list', value)

    @classmethod
    def all_conductor_groups_list(cls):
        """Возьмем из кэша или из Кондуктора и положим в кэш"""
        cached = get_cached_network('all_conductor_groups_list')

        if cached is None:
            cached = cls.calc_all_conductor_groups_list()
            cls.save_all_conductor_groups_list(cached)

        return cached
