# -*- coding: utf-8 -*-

"""
Сети в конфигураторе ищутся следующим образом:
  * Сначала ищется включение ключевого слова в название какой-либо "присвоенной"(активной)
    сети из закешированного списка названий сетей, присвоенных потребителям.
    При неточном соответствии родственные сетевые объекты не "высматриваются".
  * Далее находятся присвоенные сети по точному соответствию ключа названию сети без учета регистра.
    Возвращаются как найденные сети так "присвоенные" родственные сетевые объекты.
    Например, искали макрос, а оказалось, что и он и его хосты присвоены потребителям.
    Возвращаем и его и его хосты, которые присвоены потребителям.
  * Если ключевое слово - ip-адрес или сеть, ищем его пересечение со всеми ip-адресами
    или сетями в кеше. Возвращаем все найденные присвоенные потребителям ip-адреса
    и/или сети а также их присвоенных родителей. Например, искали сеть, возвращаем
    присвоенный макрос, которому принажлежит ip-адрес из этой сети.
  * Если ключевое слово - сетевой объект с дочерними сетями и ip-адресами - ищем их
    пересечение с присвоенными сетями или ip-адресами и возвращаем, пересекающиеся
    сети и ip-адреса. Например искали макрос - нашли присвоенные ip-адреса,
    входящие в его подсеть.

В результате поиска отдаются id инстансов модели Network.

В кеше отдельно хранятся JSON модели для каждого присвоенного сетевого
объекта и его потомков со списком id присвоенных родительских и дочерних
объектов. Ключом для поиска модели является строка имени сетевого объекта
в нижнем регистре.
Также в кеш сохраняется полный список ключей (id) всех сохраненных моделей.
Кэш обновляется раз в час по management-задаче. Поиск потребителей использует этот кэш.

Под сетевым объектом подразумевается IP, IP network, hostname, Фаервольный макрос.
Резолвинг идет от макроса до айпи/подсетей, соответственно родительским
объектом к искомому считается тот объект, который резолвится до или
влючает в себя искомый. Дочерим объектом/потомком к искомому считается
объект, который включен в или резолвится до искомого.
Объединение множеств родительских и дочерних объектов от искомого - родственные объекты.
"""

import logging
from operator import itemgetter
from itertools import chain, product

from django.conf import settings

from .exceptions import (
    NetworkResolveError,
    NotFoundError,
    HostResolvingFailed,
    UnknownNetworkTypeError,
)
from .models import Network
from .caching import (
    get_caching_wrapper,
    network_key_getter,
)
from .network_apis import (
    NetworkResolver,
    NetworkCacheManager,
    request_conductor_groups_list,
    request_firewall_macro_list,
)
from .utils import (
    normalize_network_name,
    normalize_ip_name,
    get_project_id_and_network,
    net_overlap_with_trypo,
)

ALL_NETS = {'0.0.0.0/0', '::/0'}

logger = logging.getLogger(__name__)


def from_descendants_items(nodes, key):
    """Вытащить айтемы по ключу из всех переданных JSON-моделей и из всех их потомков"""
    for node in nodes:
        children = node.get('children')
        if children:
            for item in from_descendants_items(nodes=children, key=key):
                yield item
        yield node[key]


def default_network_id_getter(network_name):
    """
    Простейшая функция получения id сети
    Всегда ходит в БД (+1 sql-запрос)
    Возвращает None если ничего не найдено
    """
    try:
        return Network.objects.get(string__iexact=network_name).id
    except Network.DoesNotExist:
        logger.debug('Network %r not found in DB', network_name)


def build_network_id_getter_from_queryset(queryset):
    """
    Получаем на вход queryset сетевых объектов и строим функцию, которая быстро возвращает id
    сетевого объекта по его строковому имени
    """
    name_to_id_map = {
        normalize_network_name(name): id
        for id, name in queryset.values_list('id', 'string')
    }

    def wrap(name):
        return name_to_id_map.get(name)

    return wrap


def get_network_children_and_type(network_node):
    """
    Возвращает список имен дочерних сетевых объектов
    :param network_node: Словарь с информацией о сети
    :return: Список имен сетей-потомков
    """
    # Потомков нет - возвращаем пустой список
    children_getter = NetworkResolver.get_children_getter(network_node['type'])
    if not children_getter:
        return list(), None

    try:
        return children_getter(network_node['string']), NetworkResolver.get_children_type(network_node['type'])
    except NetworkResolveError:
        logger.debug('Failed to get children for %s', network_node['string'])
        return list(), None


def cached_lookup_factory(caching_decorator, network_id_getter=None):
    """
    Тут все не просто так!
    Поиск дочерних сетевых объектов должен использовать кэш. Поскольку поиск по всем потомкам
    происходит рекурсивно, нужно каким-то образом использовать уже полученные данные.
    Здесь одна функция умеет искать сетевые объекты рекурсивно, другая функция умеет кэшировать
    вызовы декорируемой функции.
    Применяем кэширующий декоратор сразу после объявления функции поиска. Таким образом, при
    рекурсивном вызове в функции поиска, будет происходить поиск в кэше.

    :param caching_decorator: Функция-декоратор, сохраняющая в кэш результаты вызовов, или
    берущая из кэша результат аналогичного вызова. Например, может быть передана функция
    принудительного обновления кэша.
    :param network_id_getter: функция для получения Network.id по строке
    :return: Собранная функция поиска сетевых объектов и потомков.
    """
    network_id_getter = network_id_getter or default_network_id_getter

    # Изменяем фукнцию поиска чтобы она использовала тот метод кэширования, что указан
    # Т.о. даже при рекурсивном вызове будет использоваться кэш в redis
    @caching_decorator
    def get_network_with_children(network_name, network_type=None, parents=None):
        """
        Получить по строке json-friendly словарь с данными о сети
        Рекурсивно найти всех потомков и поместить их в словарь

        :param network_name: Имя сетевого объекта
        :param network_type: Тип сети
        :param parents: Список идентификаторов родительских сетей
        :return: Словарь, содержащий информацию о сети, родителях(если были указаны) и потомках
        """
        parents = parents or set()

        try:
            network_type = network_type or NetworkResolver.get_type(network_name)
        except NotFoundError:
            logger.warning('Not found network %r', network_name)
            return
        except UnknownNetworkTypeError:
            logger.warning('Unknown network type of %r', network_name)
            return
        except HostResolvingFailed:
            logger.warning('Host %r is not resolved', network_name)
            return

        node_id = network_id_getter(network_name)

        node = dict(
            id=node_id,
            string=normalize_network_name(network_name),
            type=network_type,
            parent_ids=list(parents - {None}),
            children=list(),
            descendant_ids=list(),
        )

        # Идентификатор сети может быть None, это значит, что такой сети еще нет в БД
        if node_id is None:
            logger.warning('Unknown network object! Empty node_id for %r', network_name)
            return node

        parents.add(node_id)
        children_names, child_type = get_network_children_and_type(node)
        for child_name in children_names:
            # Рекурсия будет использовать кэш вызовов
            child = get_network_with_children(
                child_name,
                network_type=child_type,
                parents=set(parents),
            )
            if child is not None:
                node['children'].append(child)

        # Сохраним список id всех потомков
        node['descendant_ids'] = list(set(from_descendants_items([node], 'id')) - {None})
        return node

    return get_network_with_children


def _lookup_collision_resolver(old_node, new_node):
    """Объединяет родителей у модели, если в сессии кеширования у одной сети есть много родительских линий"""
    node = new_node.copy()
    node['parent_ids'] = list(set(new_node['parent_ids']) | set(old_node['parent_ids']))
    return node


# Строим функцию для получения сетевого объекта и потомков с использованием кэша redis
cached_network_lookup = cached_lookup_factory(
    get_caching_wrapper(
        prefix=settings.REDIS_NETWORK_PREFIX,
        key_getter=network_key_getter,
    )
)


def cache_all_active_networks(mild=False):
    """
    Производит сессию кеширования всех известных присвоенных сетей
      * Список всех существующих в rackables имен сетевых макросов
      * Список всех существующих имен кондукторных групп в Кондукторе
      * Строит дерево всех присвоенных сетевых объектов и их потомков
      * Список всех известных Грантушке ip-адресов и ip-сетей
      * Список всех известных Грантушке имен сетевых объектов (включая ip-адреса и ip-сети)
    """
    if mild:
        lookup = cached_network_lookup

    else:
        # Готовится обход всех сетевых объектов и поиск их потомков
        # Предварительно выберем все известные на данный момент сетевые объекты
        # При построении дерева сетей и потомков не будем ходить в БД -- будем использовать то,
        # что уже выбрали
        id_getter = build_network_id_getter_from_queryset(Network.objects.all())

        # Заново строим функцию поиска с принудительным обновлением кэша
        # Новая функция для каждой сессии кэширования. Каждый раз новое внутреннее состояние
        # Используется только периодической management-задачей
        lookup = cached_lookup_factory(
            get_caching_wrapper(
                prefix=settings.REDIS_NETWORK_PREFIX,
                key_getter=network_key_getter,
                recache=True,
                collision_resolver=_lookup_collision_resolver,
            ),
            network_id_getter=id_getter,
        )

    # FIXME: Вынести в hourly
    logger.debug('Rebuild cache for list of all existing firewall macros')
    macro_list = request_firewall_macro_list()
    NetworkCacheManager.save_all_macros_list(macro_list)

    logger.debug('Rebuild cache for list of all existing conductor groups')
    conductor_groups_list = request_conductor_groups_list()
    NetworkCacheManager.save_all_conductor_groups_list(conductor_groups_list)

    meta_networks = []

    # FIXME: Все активные сети в случайном порядке - зачем так?
    queryset = Network.objects.exclude(activenetwork=None).values_list('string', 'type').order_by('?')

    logger.debug('Rebuild cache for %d active networks and their children', queryset.count())
    for string, network_type in queryset:
        # Получим (и закэшируем) информацию о сети и всех ее потомках
        node = lookup(
            normalize_network_name(string),  # В БД имя сети может храниться в любом виде - нормализуем
            network_type=network_type,
        )

        # Словарь node содержит вложенные словари с информацией обо ВСЕХ своих потомках
        meta_networks.append(node)
        meta_networks.extend(
            chain.from_iterable(
                from_descendants_items([node], 'children')
            )
        )
    logger.debug('Rebuild cache is done for %d meta-networks', len(meta_networks))

    # Сохраним в кэш уникальные нормализованые имена всех известных сетевых объектов
    network_list = list(set(map(itemgetter('string'), meta_networks)))
    logger.debug('Caching %d unique network names', len(network_list))
    NetworkCacheManager.save_all_networks(network_list)

    # Сохраним отдельно список всех известных нормализованых ip-адресов и ip-сетей
    ip_network_list = list(set(map(
        itemgetter('string'),
        filter(
            lambda node: node['type'] in Network.IP_TYPES,
            meta_networks,
        ),
    )))
    logger.debug('Caching %d unique ip networks and ip addresses', len(ip_network_list))
    # Единственное место, где пишем в этот элемент кэша
    NetworkCacheManager.save_ip_networks(ip_network_list)


def match_networks(keyword, deep=False, network_list=None):
    """Ищет сетевые объекты по ключевому слову"""
    # Всюду в кэше хранится нормализованное имя сети поэтому нормализуем и поисковый запрос
    keyword = normalize_network_name(keyword)

    id_set = set()
    if len(keyword) < 2:
        return id_set

    known_network_list = network_list or NetworkCacheManager.all_networks()
    if not known_network_list:
        logger.warning('Known network list is empty. Doing recache')
        cache_all_active_networks(mild=True)
        known_network_list = NetworkCacheManager.all_networks()

    # Ищем во всех известных сетях - простое вхождение подстроки в имя сетевого объекта
    # У найденных объектов не ищем родственников
    matched_networks = []
    for to_match in known_network_list:
        if keyword in to_match:
            net = cached_network_lookup(to_match)
            if net:
                matched_networks.append(net)

    id_set.update(network['id'] for network in matched_networks)

    network = cached_network_lookup(keyword)
    if not network:
        return id_set - {None}  # Выходим, если ничего не нашли

    # Искомая строка это ip-адрес или подсеть. Интересуют все родители и потомки.
    id_set.update(network['descendant_ids'])
    id_set.update(network['parent_ids'])

    # Находим пересечение с известными сетями и возвращаем эти пересечения и их родителей
    if network['type'] in Network.IP_TYPES:
        # Сюда могут придти и адреса и сети
        # 192.168.1.10 -> 192.168.1.10/32
        # 192.168.1.1/24 -> 192.168.1.1/24
        # 611@2a02:6b8:c00::/40 -> 611@2a02:6b8:c00::/40
        project_id, ip_network = get_project_id_and_network(keyword)

        if ip_network.prefixlen < 2:
            return id_set - {None}

        # Проходит ВСЕ ip-сети и ip-адреса из кэша в поисках пересечений
        known_ip_network_list = set(NetworkCacheManager.ip_networks()) - ALL_NETS
        matched_ip_networks = []
        for ipn in known_ip_network_list:
            ipn_project_id, ipn_net = get_project_id_and_network(ipn)
            if net_overlap_with_trypo(project_id, ip_network, ipn_project_id, ipn_net):
                matched_ip_networks.append(ipn)

        matched_networks = []
        for ip_net in matched_ip_networks:
            lookup = cached_network_lookup(ip_net)
            if lookup is not None:
                matched_networks.append(lookup)

        id_set.update(network['id'] for network in matched_networks if network)
        # Здесь должны быть найдены все файрвольные макросы, или сети, содержащие искомый ip-адрес/сеть
        id_set.update(
            chain.from_iterable(
                network['parent_ids']
                for network in matched_networks
                if network
            ),
        )

    if not deep:
        return id_set - {None}

    # Когда потомки неизвестного тут объекта - подсети или ip-адреса,
    # ищем их пересечение с подсетями и ip-адресами потребителей
    flat_descendants = list(chain.from_iterable(from_descendants_items([network], 'children')))
    ip_networks = [
        get_project_id_and_network(n['string'])
        for n in flat_descendants
        if n['type'] in Network.IP_TYPES
    ]
    if ip_networks:
        networks = NetworkCacheManager.active_network_values()
        active_ip_networks = [
            get_project_id_and_network(n['string'])
            for n in networks
            if n['type'] in Network.IP_TYPES and n['id'] and n['string'] not in ALL_NETS
        ]

        # Найдем пересечения с известными сетями
        matched_ipn = []
        for aipn, ipn in product(active_ip_networks, ip_networks):
            if net_overlap_with_trypo(*(aipn + ipn)):
                matched_ipn.append(cached_network_lookup(normalize_ip_name(aipn)))

        # Выберем всех родителей, какие могут быть потомки у сетевых объектов ¯\_(ツ)_/¯
        id_set.update(
            chain.from_iterable(
                ip_network['parent_ids']
                for ip_network in matched_ipn
                if ip_network
            )
        )

    return id_set - {None}
