# -*- coding: utf-8 -*-

from django.db.models import Q
from django.utils.functional import cached_property
from guardian.shortcuts import get_objects_for_user

from .models import (
    NamespaceEnvironments,
    Namespace,
)
from .utils import grouped


class UserPermissions(object):

    def __init__(self, user):
        self.user = user

    @property
    def queryset(self):
        return get_objects_for_user(self.user, NamespaceEnvironments.PERMISSION).select_related()

    @cached_property
    def pair_set(self):
        return set(self.queryset.values_list('namespace__id', 'environment__id'))

    def have_all(self, namespace, environments):
        to_check = set((namespace.id, environment.id) for environment in environments)
        return to_check <= self.pair_set

    def do_not_have(self, namespace, environments):
        to_check = set((namespace.id, environment.id) for environment in environments)
        return [NamespaceEnvironments.objects.get(namespace=n, environment=e) for n, e in to_check - self.pair_set]

    def intersect_grouped(self, namespaces):
        for namespace in namespaces:
            intersected = [e for e in namespace.environments.all() if (namespace.id, e.id) in self.pair_set]
            if intersected:
                yield namespace, intersected

    def Q(self):
        q = Q()
        for namespace_id, namespace_environment in grouped(self.queryset, key=lambda ne: ne.namespace_id):
            q |= Q(
                namespace=namespace_id,
                environments__in=[ne.environment_id for ne in namespace_environment],
            )
        return q


# TODO: Это должно быть middleware
def namespace_selection(request):
    namespaces = Namespace.objects.all()
    namespace_id = request.GET.get('namespace_id') or request.COOKIES.get('namespace')

    try:
        namespace = Namespace.objects.get(id=namespace_id)
    except Namespace.DoesNotExist:
        if namespaces:
            namespace = namespaces[0]
        else:
            namespace = None

    return {
        'namespaces': namespaces,
        'namespace': namespace,
    }


def get_permissions(func):
    def wrapper(self, **kwargs):
        self.user_namespace_environments = UserPermissions(self.request.user).queryset
        context = func(self, **kwargs)
        context['user_namespace_environments'] = self.user_namespace_environments
        return context
    return wrapper
