# -*- coding: utf-8 -*-
import logging
import os
import select
import subprocess

from django.conf import settings

from ..exceptions import SubprocessTimedOut
from ..utils import (
    get_unixtime,
    process_handling,
)

logger = logging.getLogger(__name__)

CHUNK_SIZE = 2 ** 16


def build_pipe_result(buffer_list):
    if len(buffer_list) == 0:
        return

    return ''.join(buffer_list)


def process_with_timeout(process, timeout):
    """
    Реализация функции subprocess.Popen.communicate() с таймаутом (через select.select)
    Позволяет ждать событий чтения из потоков STDOUT & STDERR переданного подпроцесса с помощью select.select()

    :param process: Экземпляр subprocess.Popen
    :param timeout: float время в секундах на ожидание завершения подпроцесса
    :return: аналогично subprocess.Popen.communicate()
    """

    deadline = get_unixtime() + timeout
    stdout_buffer, stderr_buffer = list(), list()

    # Считаю, что все, что надо было передать на вход процесса уже было передано
    if process.stdin:
        process.stdin.flush()
        process.stdin.close()

    # Будем читать из stdout & stderr подпроцесса
    read_pipe_list = list()
    if process.stdout:
        read_pipe_list.append(process.stdout)
    if process.stderr:
        read_pipe_list.append(process.stderr)

    try:
        while read_pipe_list:
            now = get_unixtime()
            if now >= deadline:
                raise SubprocessTimedOut()

            time_remain = deadline - now
            rlist, wlist, xlist = select.select(read_pipe_list, [], [], time_remain)

            if process.stdout in rlist:
                buffer = os.read(process.stdout.fileno(), CHUNK_SIZE)
                if buffer == '':
                    process.stdout.close()
                    read_pipe_list.remove(process.stdout)
                stdout_buffer.append(buffer)

            if process.stderr in rlist:
                buffer = os.read(process.stderr.fileno(), CHUNK_SIZE)
                if buffer == '':
                    process.stderr.close()
                    read_pipe_list.remove(process.stderr)
                stderr_buffer.append(buffer)

    finally:
        # Если процесс не завершен, пошлем ему KILL-сигнал
        return_code = process.poll()
        if return_code is None:
            process.kill()
            # И подождем завершения процесса
            process.wait()

    return build_pipe_result(stdout_buffer), build_pipe_result(stderr_buffer)


def cvs_checkout(repository_uri, repository_dir, path='/', output_dir='.', timeout=None):
    """Выкачивает указанный cvs-репозиторий"""

    cvs_co = subprocess.Popen(
        ['cvs', '-q', '-d', repository_uri, 'co', '-d', output_dir, path],
        stdout=subprocess.PIPE,
        stderr=subprocess.PIPE,
        cwd=repository_dir,
    )

    if timeout:
        return process_with_timeout(cvs_co, timeout)

    else:
        return process_handling('cvs co', cvs_co)
