# -*- coding: utf-8 -*-
import ipaddress
import json
import logging
import os
import re
import sys
import time
from urlparse import urljoin
from collections import namedtuple, OrderedDict
from datetime import datetime
from functools import (
    partial,
    wraps,
)
from itertools import izip_longest, groupby
from netaddr import IPNetwork
from subprocess import Popen, PIPE

import ujson
import pytz
import requests
import requests.exceptions

from django.conf import settings
from django.core.exceptions import ValidationError
from django.http import HttpResponse

from passport_backend_core.grants.trypo_compatible_radix import (
    BITS_AFTER_PROJECT_ID,
    PROJECT_ID_MASK,
)

from . import zookeeper
from .exceptions import (
    NotFoundError,
    ProcessError,
    APIRequestFailedError,
)

logger = logging.getLogger(__name__)

DEFAULT_M4_MACRO_SEPARATOR = ' or '
DEFAULT_TIMEZONE = pytz.timezone('Europe/Moscow')

GITHUB_UTC_DATETIME_FORMAT = '%Y-%m-%dT%H:%M:%SZ'

# Отображаем русские символы в английские - интересуют только буквы
ENG_LAYOUT = u'qwertyuiop[]asdfghjkl;\'zxcvbnm,.QWERTYUIOP{}ASDFGHJKL:"ZXCVBNM<>'
RUS_LAYOUT = u'йцукенгшщзхъфывапролджэячсмитьбюЙЦУКЕНГШЩЗХЪФЫВАПРОЛДЖЭЯЧСМИТЬБЮ'
RUS_TO_ENG_MAP = dict(zip(RUS_LAYOUT, ENG_LAYOUT))

RUS_RE = re.compile(ur'[%s]' % RUS_LAYOUT, re.UNICODE)

YANDEXNETS = {'_YANDEXNETS_'}
LOCALHOST = {'127.0.0.1', '::1'}

CommitInfo = namedtuple('CommitInfo', 'date message')


def grouped(iterable, key):
    """
    Это обертка над itertools.groupby -- предварительно сортируем переданный итератор по тому же ключу
    Превый элемент кортежа - уникальное значение ключевой функции
    Второй элемент кортежа содержит итератор на группу оригинальных элементов,
    Если вложенный итератор не сохранить при проходе внешнего итератора, данные будут потеряны
    см документацию на itertools.groupby()
    :param iterable: Список или генератор объектов для группировки
    :param key: Функция получения ключевого значения для группировки
    :return: Функция возвращает итератор, из кортежей
    """
    return groupby(sorted(iterable, key=key), key=key)


def format_objects(object_list, case='nominative'):
    form = 'singular' if len(object_list) == 1 else 'plural'

    grouped_objects = grouped(object_list, key=lambda o: o.__class__)
    group_texts = []
    for model, objects in grouped_objects:
        group_texts.append(
            u'%s %s' % (
                model.grammar[form][case],
                u', '.join(unicode(o) for o in objects),
            )
        )

    return u' и '.join(group_texts)


def json_response(func):
    @wraps(func)
    def wrapper(request, *args, **kwargs):
        result = func(request, *args, **kwargs)
        response = HttpResponse(mimetype='application/json')
        if result:
            response.status_code = result.get('status', 200)
            # TODO: Проверить, насколько важно отправлять в фронт отсортированный по ключам json
            response.write(json.dumps(result, sort_keys=True, ensure_ascii=False))
        else:
            response.write('{}')
        return response
    return wrapper


def get_subprocess_pipe(args, **kwargs):
    """Возвращает дочерний процесс с доступом к stdin, stdout,  stderr"""
    return Popen(args, stdin=PIPE, stdout=PIPE, stderr=PIPE, **kwargs)


def get_client_ip(request):
    x_real_ip = request.META.get('HTTP_X_REAL_IP')
    if x_real_ip:
        return x_real_ip.strip()

    x_forwarded_for = request.META.get('HTTP_X_FORWARDED_FOR')
    if x_forwarded_for:
        return x_forwarded_for.split(',')[-1].strip()

    return request.META.get('REMOTE_ADDR')


def process_handling(name, process, error=ProcessError, handle_err=True, ignore_returncode=False):
    out, err = process.communicate()
    if handle_err and err and (process.returncode > 0 or ignore_returncode):
        raise error(u'команда %s вернула ошибку "%s" и код ошибки %i: ' % (
            name,
            err,
            process.returncode,
        ))

    return out, err


def git_pull(repository, repository_dir, branch='master'):
    logger.debug('Pulling repository from %s', repository)
    clone = get_subprocess_pipe(['git', 'clone', repository, repository_dir], cwd=repository_dir)
    process_handling('git clone', clone)

    logger.debug('Check out to %s branch', branch)
    checkout = get_subprocess_pipe(['git', 'checkout', branch], cwd=repository_dir)
    out, err = process_handling('git checkout', checkout, handle_err=False)
    if not (err.startswith('Switched to a new branch') or err.startswith('Already on')):
        raise ProcessError(err)

    git_branch = get_subprocess_pipe(['git', 'branch'], cwd=repository_dir)
    out, _ = process_handling('git branch', git_branch)
    if '* %s' % branch not in out:
        raise ProcessError(u'Не удалось сменить ветвь репозитория на %s' % branch)


def git_push(repository, repository_dir, committer, message, branch='master'):
    logger.debug('Adding changes to stage')
    add = get_subprocess_pipe(['git', 'add', '-u'], cwd=repository_dir)
    process_handling('git add', add)

    logger.debug('Committing changes with message %r', message)
    author = '--author="%(DEBFULLNAME)s <%(DEBEMAIL)s>"' % committer
    commit = get_subprocess_pipe(['git', 'commit', '-m', message, author], cwd=repository_dir)
    process_handling('git commit', commit)

    logger.debug('Pushing repository to origin %s', repository)
    push = get_subprocess_pipe(['git', 'push', 'origin', branch], cwd=repository_dir)
    out, err = process_handling('git push', push, handle_err=False)
    if not err.startswith('To ' + repository):
        raise ProcessError(err)


def git_add(repository_dir):
    add = get_subprocess_pipe(['git', 'add', '-N', '.'], cwd=repository_dir)
    add_, err = process_handling('git add -N .', add)
    if err:
        raise ProcessError(err)


def git_status(repository_dir):
    status = get_subprocess_pipe(['git', 'status'], cwd=repository_dir)
    status_, err = process_handling('git status', status)
    if err:
        raise ProcessError(err)
    return status_


def git_diff(repository_dir):
    diff = get_subprocess_pipe(['git', 'diff'], cwd=repository_dir)
    diff_out, _ = process_handling('git diff', diff)
    return diff_out


def check_grants(project_name, repository_dir, checker_script_name):
    """
    Скрипт дополнительной валидации грантов. Живет в проекте с грантами.
    :rtype: dict
    :return {environment: {consumer: message}}
    """
    full_name = os.path.join(repository_dir, checker_script_name)

    if not os.path.exists(full_name):
        logger.info('No file %s', full_name)
        return {}

    logger.debug('Executing grants checker script for %s', project_name)
    script_runner = get_subprocess_pipe(['python', full_name], cwd=repository_dir)
    out, err = process_handling(['python', full_name], script_runner, ignore_returncode=True)
    if err:
        logger.error('Error while checking grants: %s', err)
        raise ProcessError(err)

    errors = json.loads(out)
    return errors


def deb_changelog(repository_dir, committer, message):
    dch = get_subprocess_pipe(
        ['dch', '-i', message, '--distributor=debian', '--no-auto-nmu'],
        cwd=repository_dir, env=committer,
    )
    process_handling('dch', dch)


def request_git_last_commits_info_raw(api, n=1):
    # https://bb.yandex-team.ru/plugins/servlet/restbrowser#/resource/api-1-0-projects-projectkey-repos-repositoryslug-commits
    # (дефолтный лимит 25)
    if n > 25:
        raise ValueError('N is bigger than max page length (25)')
    try:
        logger.debug('Fetching last %s commits info from %s', n, api['commits'])
        commit_response = requests.get(
            api['commits'],
            headers={'Authorization': 'Bearer %s' % api['api_token']},
            timeout=api['api_timeout'],
        )
        commits = ujson.loads(commit_response.text)['values']
        if not isinstance(commits, list) or len(commits) < n:
            logger.warning('Commit info is not full for project {}; Response: {}'.format(
                api['project'],
                commit_response,
            ))
            return []
        if n == 1:
            return commits[0]
        return commits[:n]
    except (requests.ConnectionError, requests.Timeout, ValueError, KeyError) as ex:
        logger.error('Error during fetching last %s commits info: %s', n, ex)
        return []


def parse_last_commit_info(commit):
    last_commit_unixtime = commit['committerTimestamp'] // 1000
    last_commit_message = commit['message']
    return CommitInfo(
        datetime.fromtimestamp(last_commit_unixtime),
        last_commit_message,
    )


def request_git_last_commit_info(api):
    last_commit = request_git_last_commits_info_raw(api)
    if not last_commit:
        return CommitInfo(None, None)
    return parse_last_commit_info(last_commit)


def request_git_last_commits_info(api, n=1):
    last_commits = request_git_last_commits_info_raw(api, n=n)
    result = []
    if not last_commits:
        return result

    for commit in last_commits:
        result.append(parse_last_commit_info(commit))

    return result


class Config(OrderedDict):
    """
    Парсит необходимое для конфигуратора подмножество конфигов апача,
    метод _parse базируется на использованном парсере в Паспорте
    """
    re_comment = re.compile(r'^#.*$')
    re_start = re.compile(r'^<(?P<name>[^/\s>]+)\s*(?P<values>[^>]+)?>$')
    re_end = re.compile(r'^</(?P<name>[^\s>]+)\s*>$')

    def __init__(self, _config=None, *args, **kwargs):
        OrderedDict.__init__(self, *args, **kwargs)
        if _config:
            self._parse(clip=iter(_config))

    def _parse(self, clip, name=None):
        for line in clip:
            line = line.strip()
            if not line or self.re_comment.match(line):
                continue

            match = self.re_start.match(line)
            if match:
                name = match.group('name')
                values = match.group('values').split()
                parsed = Config()._parse(clip, name)
                self.setdefault(name, Config()).update({v: parsed for v in values})
                continue

            match = self.re_end.match(line)
            if match:
                if name != match.group('name'):
                    raise Exception('Section mismatch: "{}" should be "{}"'.format(match.group("name"), name))
                return self

            values = line.split()
            self[values[0]] = values[1:]


def modify_dict(dictionary, add=None, remove=None):
    """Возвращает копию словаря с добавленными и удаленными элементами"""
    dictionary = dictionary.copy()
    dictionary.update(**(add or {}))
    map(dictionary.__delitem__, filter(dictionary.__contains__, remove or []))
    return dictionary


def deep_merge(graph1, graph2):
    """Принимает два словаря и возвращает копию первого, обновленную значениями из второго"""
    if graph1 is None:
        return graph2
    elif graph2 is None:
        return graph1

    elif isinstance(graph2, dict):
        return {key: deep_merge(graph1.get(key), graph2.get(key)) for key in set(graph1) | set(graph2)}
    elif isinstance(graph2, list):
        return [deep_merge(item1, item2) for item1, item2 in izip_longest(graph1, graph2)]

    else:
        return graph2


def yesno_to_bool(yesno_string):
    yesno_string = unicode(yesno_string).lower()
    if yesno_string == 'yes':
        return True
    elif yesno_string == 'no':
        return False


def get_unixtime():
    return time.time()


def get_dict_by_id(queryset):
    return {item.id: item for item in queryset}


def switch_keyboard_layout_to_eng(text):
    """Заменим все русские буквы на английские - полагаем что пользователь ошибся раскладкой"""
    return ''.join(map(
        lambda char: RUS_TO_ENG_MAP.get(char, char),
        text,
    ))


def normalize_network_name(network_name):
    """Приводим имя сетевого объекта к нижнему регистру - для хранения в кэше"""
    return unicode(network_name).lower()


def not_empty_string(value):
    """Валидатор для строк - запрещает пустой ввод - пробелы, табуляции"""
    stripped_value = unicode(value).strip()
    if stripped_value == '':
        raise ValidationError(u'Введите хотя бы один непустой символ')


def request_api(api, path, params=None, headers=None, **kwargs):
    retry_left = api['retry']
    url = urljoin(api['address'], path)
    logger.debug('requesting url %s', url)
    while retry_left:
        try:
            request = requests.get(
                url,
                params=params,
                headers=headers,
                timeout=api['timeout'],
                **kwargs
            )
            if request.status_code == 404:
                raise NotFoundError('Failed as 404 on requesting %s' % url)
            request.raise_for_status()
            logger.debug('Got appropriate response for %s', url)
            return request.text

        except requests.exceptions.RequestException as ex:
            logger.debug('Failed as %s on requesting %s', ex, url)
            retry_left -= 1

        except Exception as ex:
            logger.exception('Unexpected exception: %s', ex)
            raise

    message = 'No retries left for %r' % api['address']
    logger.error(message)
    raise APIRequestFailedError(message)


def ascii_string(value):
    """Валидатор для строк - запрещает ввод не ascii-символов"""
    if RUS_RE.search(value):
        raise ValidationError(u'Русские символы недопустимы')


def is_keyword_forbidden(word):
    """
    Не разрешаем искать некоторые случаи
      - _YANDEXNETS_ - слишком обширная сеть, сгенерированный sql-запрос получится слишком длинным и БД его отвергнет
      - '' - пустая строка значит нечего искать
    """
    if word in YANDEXNETS:
        return True

    elif word == '':
        return True

    return False


def format_form_errors(form_errors):
    return [
        u'%s: %s' % (field_name, ', '.join(errors))
        for field_name, errors in form_errors.iteritems()
    ]


def retries(count, func, retry_on=Exception):
    """
    Декоратор для повторных вызовов фукнции заданное число раз в случае возникновения ошибки
    :param count: Сколько раз пытаться выполнить функцию
    :param func: Декорируемая функция
    :param retry_on: Исключение, которое отлавливать и пробовать еще раз
    :return: Декорированная функция
    """

    def wrap(*args, **kwargs):
        retries_left = count
        while retries_left > 0:
            try:
                return func(*args, **kwargs)
            except retry_on as ex:
                retries_left -= 1
                logger.debug('[%d/%d] Retry on error %s', count - retries_left, count, ex)

    return wrap


def run_exclusively(lock_name, log):
    def decorator(func):
        @wraps(func)
        def wrapper(*args, **kwargs):
            try:
                zookeeper.run_exclusively(
                    func=partial(func, *args, **kwargs),
                    lock_name=lock_name,
                    zookeeper_hosts=settings.ZOOKEEPER_HOSTS,
                )
            except zookeeper.TimeoutError:
                log.exception('zookeeper unavailable')
                sys.exit(1)
        return wrapper
    return decorator


def looks_like_trypo_support_network(text):
    """IPv6-сеть вида 411a@2a02:6b8:c00::/40"""
    return '@' in text


def net_overlap(network1, network2):
    """Проверяет вхождение сетей или ip-адресов"""
    return (network1 in network2) or (network2 in network1)


def get_trypo_network_and_mask(project_id, network):
    # Подставим project_id в сеть с 64 по 95 биты
    project_id_int = int(project_id, 16) << BITS_AFTER_PROJECT_ID | network.value
    network_with_project_id = IPNetwork(str(ipaddress.ip_address(project_id_int)))
    # Сохраним префикс оригинальной сети
    network_with_project_id.prefixlen = network.prefixlen
    # Делаем дырявую маску на основе оригинальной сети
    trypo_mask = network.netmask.value | PROJECT_ID_MASK
    return network_with_project_id, trypo_mask


def net_overlap_with_trypo(p_id1, net1, p_id2, net2):
    if all([p_id1, p_id2]) or not any([p_id1, p_id2]):
        return p_id1 == p_id2 and net_overlap(net1, net2)
    if net1.version == 6 and net2.version == 6:
        if p_id1 is not None:
            new_net1, trypo_mask = get_trypo_network_and_mask(p_id1, net1)
            masked_net2 = IPNetwork(str(ipaddress.IPv6Address(trypo_mask & net2.value)))
            masked_net2.prefixlen = net2.prefixlen
            return net_overlap(masked_net2, new_net1)
        if p_id2 is not None:
            new_net2, trypo_mask = get_trypo_network_and_mask(p_id2, net2)
            masked_net1 = IPNetwork(str(ipaddress.IPv6Address(trypo_mask & net1.value)))
            masked_net1.prefixlen = net1.prefixlen
            return net_overlap(masked_net1, new_net2)
    return False


def get_project_id_and_network(network_string):
    if looks_like_trypo_support_network(network_string):
        project_id, cut_network = network_string.split('@')
        return project_id, IPNetwork(cut_network)
    return None, IPNetwork(network_string)


def is_decreased_by_trypo(previous, current):
    trypo_in_previous = any([looks_like_trypo_support_network(net) for net in previous])
    trypo_in_current = any([looks_like_trypo_support_network(net) for net in current])
    return not trypo_in_previous and trypo_in_current


def normalize_ip_name(project_network):
    project_id, network = project_network
    net_part = normalize_network_name(
        str(network.ip if network.size == 1 else network),
    )
    if not project_id:
        return net_part
    return u'@'.join([normalize_network_name(project_id), net_part])
