# -*- coding: utf-8 -*-
import json
import logging

from collections import defaultdict
from copy import deepcopy

from django.conf import settings
from django.forms import model_to_dict
from django.shortcuts import render

from ..exceptions import (
    NetworkResolveError,
    NoDataError,
)
from ..common_deserializers import (
    serialize_grants,
    deserialize_macroses,
)
from ..forms import (
    CreateGrantForm,
    ValidateNetworkForm,
    GetGrantsListForm,
    NetworkSuggestForm,
    NotifyCreatorsForm,
)
from ..network_apis import (
    NetworkResolver,
    NetworkCacheManager,
    looks_like_old_conductor_macro,
)
from ..mail import notify_package_in_stable
from ..models import (
    Client,
    Namespace,
    Grant,
    Network,
    Action,
    Macros,
    ActiveNetwork,
    ActiveAction,
    ActiveMacros,
    Issue,
)
from ..permissions import (
    UserPermissions,
    namespace_selection,
)
from ..utils import (
    format_form_errors,
    json_response,
    modify_dict,
    request_git_last_commit_info,
    request_git_last_commits_info,
)

logger = logging.getLogger(__name__)

MAX_NETWORK_SUGGEST_STARTS_WITH = 30
MAX_NETWORK_SUGGEST_CONTAINS = 10


# TODO: Покрыть тестами
def operations(request):
    grants_projects = deepcopy(settings.GRANTS_PROJECTS)

    for project in grants_projects.values():
        project['last_commit_info'] = request_git_last_commit_info(api=project['git_api'])

    user_permissions = UserPermissions(request.user)
    _namespaces = set(ne.namespace.name for ne in user_permissions.queryset)
    project_permissions = set(name for name, conf in settings.PROJECTS.iteritems() if set(conf['namespaces']) & _namespaces)

    return render(request, 'operations.html', {
        'form': CreateGrantForm(),
        'grant_names': Grant.objects.values_list('name', flat=True),
        'project_permissions': project_permissions,
        'projects': grants_projects,
        'simulation_mode': settings.SIMULATE_EXPORT_GRANTS,
    })


@json_response
def network_validation(request):
    """
    AJAX запрос валидации маски сети
    Используется только на странице создания заявки
    """
    if not request.user:
        return {
            'success': False,
            'errors': [u'Не удалось определить Ваш логин'],
            'network_valid': False,
        }

    form = ValidateNetworkForm(request.GET)
    if not form.is_valid():
        return {
            'success': False,
            'errors': format_form_errors(form.errors),
            'network_valid': False,
        }

    network = form.cleaned_data['network']
    objects = form.cleaned_data['objects']

    environment = form.cleaned_data['environment']
    namespace = form.cleaned_data['namespace']

    network, _ = Network.objects.get_or_create(string=network.string, type=network.type)
    errors = []
    response = {
        'network_valid': True,
        'string': network.string,
        'type': network.type,
        'id': network.id,
        'success': False,
        'errors': errors,
    }

    user_permissions = UserPermissions(request.user)
    if not user_permissions.have_all(namespace=namespace, environments=[environment]):
        # TODO: Вынести в функцию
        try:
            is_allowed, message = NetworkResolver.check_permissions(
                network.type,
                network.string,
                request.user.username,
            )
        except (NetworkResolveError, NoDataError) as e:
            errors.append(e.message)
            return response

        # TODO: Может ли быть так, что нет прав пользователя в Грантушке, но есть права в Големе?
        # нужен ли этот if?
        if not is_allowed:
            if isinstance(message, list):
                errors.extend(message)
            else:
                errors.append(message)
            return response

    response['success'] = True

    if network.type in Network.IP_TYPES:
        response['network'] = network.string

    else:
        if network.type == Network.HOSTNAME:
            objects, _ = NetworkResolver.try_resolve(
                [network.string],
                NetworkResolver.get_host_ips_with_retries,
                NetworkResolveError,
            )
        response['network'] = {network.string: objects}

    return response


@json_response
def network_suggest(request):
    form = NetworkSuggestForm(request.GET)
    if not form.is_valid():
        return {'success': False, 'errors': format_form_errors(form.errors)}

    keyword = form.cleaned_data.get('keyword')

    macros_list = []
    c_groups = []

    if not keyword.startswith('%'):
        # Нам нужны только макросы
        try:
            macros_list = NetworkCacheManager.all_macros_list()
            macros_list = [m for m in macros_list if not looks_like_old_conductor_macro(m)]
        except NetworkResolveError:
            pass

    if not keyword.startswith('_'):
        # Нам нужны только кондукторные группы
        try:
            c_groups = NetworkCacheManager.all_conductor_groups_list()
        except NetworkResolveError:
            pass

    stripped_kwd = keyword.strip('_%')

    macro_contains = [macro for macro in macros_list if stripped_kwd in macro.strip('_').lower()]
    macro_starts_with = [macro for macro in macro_contains if macro.lower().strip('_').startswith(stripped_kwd)]
    macro_contains = list(set(macro_contains) - set(macro_starts_with))

    c_group_contains = [group for group in c_groups if stripped_kwd in group.lower().strip('%')]
    c_group_starts_with = [group for group in c_group_contains if group.lower().strip('%').startswith(stripped_kwd)]
    c_group_contains = list(set(c_group_contains) - set(c_group_starts_with))

    matched = (
        macro_starts_with[:MAX_NETWORK_SUGGEST_STARTS_WITH] +
        macro_contains[:MAX_NETWORK_SUGGEST_CONTAINS] +
        c_group_starts_with[:MAX_NETWORK_SUGGEST_STARTS_WITH] +
        c_group_contains[:MAX_NETWORK_SUGGEST_CONTAINS]
    )

    return {
        'success': True,
        'suggestions': matched,
    }


@json_response
def get_grants_list(request):
    """Отдает полный список грантов для окружения. Ставит active: true, у грантов, если у запрошенного потребитея есть
    такой грант. Ставит, active_new в соответствии с изменениями в заявке (если передан потребитель или заявка)"""
    form = GetGrantsListForm(request.GET)
    if not form.is_valid():
        return {'success': False, 'errors': format_form_errors(form.errors)}

    issue = form.cleaned_data.get('issue')
    consumer = form.cleaned_data.get('consumer')
    namespace = form.cleaned_data.get('namespace')
    environments = namespace.environments.all()

    if issue and issue.creator != request.user:
        return {'success': False, 'errors': [u'Вы не являетесь создателем этой заявки']}

    actions = Action.objects.select_related('grant').filter(grant__namespace=namespace)
    grants = serialize_grants(actions)

    macroses = []
    for macros in Macros.objects.prefetch_related('action__grant').filter(namespace=namespace):
        macroses.append(model_to_dict(macros, ['id', 'name', 'description']))
        macroses[-1].update(grants=serialize_grants(macros.action.all(), items=('name',)))

    an_queryset = ActiveNetwork.objects.select_related('network', 'environment').filter(consumer=consumer)
    aa_queryset = ActiveAction.objects.select_related('action', 'environment').filter(consumer=consumer)
    am_queryset = ActiveMacros.objects.select_related('macros', 'environment').filter(consumer=consumer)

    networks = defaultdict(list)
    clients = {}
    namespace_name = None
    if issue:
        namespace_name = issue.namespace.name
    elif consumer:
        namespace_name = consumer.namespace.name

    for environment in environments:
        id_ = environment.id
        active_actions = active_actions_new = \
            {aa.action.id: aa.expiration and aa.expiration.strftime("%d.%m.%Y") for aa in aa_queryset if aa.environment == environment}
        active_macroses = active_macroses_new = \
            {am.macros.id: am.expiration and am.expiration.strftime("%d.%m.%Y") for am in am_queryset if am.environment == environment}
        active_networks = active_networks_new = set(an.network for an in an_queryset if an.environment == environment)
        all_networks = active_networks

        if issue and issue.namespace == namespace:
            till = issue.expiration and issue.expiration.strftime("%d.%m.%Y")
            active_actions_new = \
                modify_dict(active_actions, add=dict.fromkeys(issue.add_action_id(), till), remove=issue.del_action_id())
            active_macroses_new = \
                modify_dict(active_macroses, add=dict.fromkeys(issue.add_macros_id(), till), remove=issue.del_macros_id())

            add_networks_set = set(issue.add_network.filter(addnetworks__environment=environment))
            del_networks_set = set(issue.del_network.filter(delnetworks__environment=environment))
            active_networks_new = (active_networks_new | add_networks_set) - del_networks_set
            all_networks = active_networks | add_networks_set | del_networks_set

        for grant in grants:
            for action in grant['actions']:
                a_id = action['id']
                action.setdefault('active_by', {})[id_] = a_id in active_actions
                action.setdefault('active_new_by', {})[id_] = a_id in active_actions_new
                action.setdefault('expiration_by', {})[id_] = active_actions.get(a_id)
                action.setdefault('expiration_new_by', {})[id_] = active_actions_new.get(a_id)

        for macros in macroses:
            m_id = macros['id']
            macros.setdefault('active_by', {})[id_] = m_id in active_macroses
            macros.setdefault('active_new_by', {})[id_] = m_id in active_macroses_new
            macros.setdefault('expiration_by', {})[id_] = active_macroses.get(m_id)
            macros.setdefault('expiration_new_by', {})[id_] = active_macroses_new.get(m_id)

        networks_ = networks[id_]
        for network in sorted(all_networks, key=lambda n: n.string):
            networks_.append(model_to_dict(network, 'id string type'))
            networks_[-1].update(active=network in active_networks, active_new=network in active_networks_new)

        if namespace_name in settings.NAMESPACES_BY_CLIENT:
            env_client = None
            if consumer:
                try:
                    env_client = Client.objects.get(consumer=consumer, environment=environment)
                except Client.DoesNotExist:
                    pass
            if issue:
                env_clients = issue.clients.filter(environment=environment)
                if env_clients:
                    env_client = env_clients[0]
            if env_client:
                clients[id_] = dict(
                    id=env_client.id,
                    client_id=env_client.client_id,
                    name=env_client.name,
                )
    return {
        'success': True,
        'errors': [],
        'consumer_grants': dict(
            grants=grants,
            macroses=macroses,
            networks=networks,
            clients=clients,
        ),
    }


def reference(request):
    """Страница грантов и их групп"""
    context = namespace_selection(request)
    context['namespaces'] = [n for n in context['namespaces'] if not n.hidden]

    environment_macroses = Macros.objects.filter(namespace=context['namespace'])
    context.update({
        'grants': Grant.objects.filter(namespace=context['namespace']),
        'meta_macroses': deserialize_macroses(environment_macroses),
    })

    if getattr(context['namespace'], 'name', None) == 'passport':
        context['subscription_grants'] = context['grants'].get(name='subscription')
        context['grants'] = context['grants'].exclude(name='subscription')

    rendered = render(request, 'reference.html', context)
    rendered.set_cookie('namespace', context['namespace'].id)
    return rendered


@json_response
def grant_create(request):
    """Здесь namespace берем из поля формы `grant_namespace`"""

    form = CreateGrantForm(request.POST)
    if not form.is_valid():
        return {'success': False, 'errors': format_form_errors(form.errors)}

    namespace = Namespace.objects.get(name=form.cleaned_data['grant_namespace'])
    permissions = UserPermissions(request.user).queryset.filter(namespace=namespace)
    if not permissions.count():
        message = u'У Вас нет прав для добавления гранта в проект %s' % namespace
        return {'success': False, 'errors': [message]}

    form.save()
    return {'success': True}


@json_response
def notify_creators(request):
    form = NotifyCreatorsForm(json.loads(request.body))

    if not form.is_valid():
        logging.error('Error form data from Conductor WebHook: %s', format_form_errors(form.errors))
        return {'success': False, 'errors': format_form_errors(form.errors)}

    if form.cleaned_data['ticket_tasks_total'] != form.cleaned_data['ticket_tasks_finished']:
        # Accepted, not finished
        return {'status': 202, 'success': True}

    project = form.cleaned_data['project']

    commit_info = request_git_last_commits_info(settings.PROJECTS[project]['git_api'], n=2)

    if len(commit_info) != 2:
        # Хотим, чтобы Кондуктор попробовал еще раз
        return {'status': 400, 'success': False}

    last_commit_date = commit_info[0].date.strftime(settings.DB_DATE_FORMAT)
    pre_last_date = commit_info[1].date.strftime(settings.DB_DATE_FORMAT)

    issues = Issue.objects.select_related('creator', 'environments').filter(
        status=Issue.APPROVED,
        namespace__name=project,
        approving_date__range=[pre_last_date, last_commit_date],
    )

    # Админам отсылаем письмо, только если их
    # вручную добавили на странице создания заявки
    # Группируем письма по получателям
    recipients_and_issues = defaultdict(set)
    for issue in issues:
        if issue.creator.username not in settings.PASSPORT_TEAM_ADMIN:
            recipients_and_issues[issue.creator.email].add(issue)
        for email in issue.emails.all():
            recipients_and_issues[email.address].add(issue)

    notify_package_in_stable(request, recipients_and_issues, project)

    return {'success': True}
