# -*- coding: utf-8 -*-
from __future__ import division, unicode_literals
import json
import logging
from itertools import chain
from collections import defaultdict
from operator import itemgetter

from django.conf import settings
from django.core.paginator import (
    EmptyPage,
    Paginator,
)
from django.db.models import Q
from django.forms import model_to_dict
from django.shortcuts import render

from passport_grants_configurator.apps.core.exceptions import BaseError

from ..forms import (
    SearchConsumersForm,
    ConsumerSuggestForm,
)
from ..network_apis import NetworkCacheManager
from ..network_search_tools import match_networks
from ..models import (
    Action,
    ActiveAction,
    ActiveMacros,
    ActiveNetwork,
    Client,
    Consumer,
    Grant,
    Macros,
    Network,
    NETWORK_TYPE_NAMES,
)
from ..permissions import namespace_selection
from ..utils import (
    json_response,
    grouped,
    get_dict_by_id,
    switch_keyboard_layout_to_eng,
    is_keyword_forbidden,
    format_form_errors,
)

SUGGEST_CONSUMER_MAX_RESULTS = 12

logger = logging.getLogger(__name__)


def get_actions_by_macros(namespace):
    """
    Один запрос в БД выбирает все id макросов (групп грантов) для указанного проекта
    Другой запрос выбирает значения из промежуточной  many-to-many таблицы -> получаем список кортежей вида

    [ (macros_id1, action_id1), (macros_id1, action_id2), (macros_id2, action_id3) ]

    Из него строим словарь, где ключи - уникальные идентификаторы macros, а значения - список идентификаторов action

    { macros_id1: [action_id1, action_id2], macros_id2: [action_id3] }

    :param namespace: Модель проекта, например, blackbox или Паспорт
    :return: Словарь где по id макроса можно найти список id его action
    """
    macros_ids = Macros.objects.filter(namespace=namespace).values_list('id', flat=True)
    _qs = Macros.action.through.objects.filter(macros_id__in=macros_ids)\
        .order_by('macros__id', 'action__name').values_list('macros__id', 'action__id')

    actions_by_macros = grouped(_qs, key=itemgetter(0))  # Выбираем первый элемент - macros_id
    return dict(
        (macros_id, map(itemgetter(1), values_list))  # Выбираем вторые элементы из кортежей
        for macros_id, values_list in actions_by_macros
    )


def get_filter_query(environments, grants=None, macroses=None,
                     actions=None, networks=None):
    """Построим sql-выражение для выборки"""
    query = Q()

    if grants:
        query |= Q(
            activeaction__action__grant__in=grants,
            activeaction__environment__in=environments,
        )
    if actions:
        query |= Q(
            activeaction__action__in=actions,
            activeaction__environment__in=environments,
        )
    if macroses:
        query |= Q(
            activemacros__macros__in=macroses,
            activemacros__environment__in=environments,
        )
    if networks:
        query |= Q(
            activenetwork__network__in=networks,
            activenetwork__environment__in=environments,
        )

    return query


@json_response
def consumer_search(request):
    form = SearchConsumersForm(request.GET)
    if not form.is_valid():
        logger.debug('Form is invalid')
        return {'success': False, 'errors': format_form_errors(form.errors)}

    namespace = form.cleaned_data['namespace']
    environments = set(form.cleaned_data['environments'])

    actions_by_id = get_dict_by_id(
        Action.objects.select_related('grant').filter(grant__namespace=namespace)
    )

    macroses = get_dict_by_id(Macros.objects.filter(namespace=namespace))
    actions_by_macros = get_actions_by_macros(namespace)

    consumers_qs = Consumer.objects.filter(namespace=namespace)

    # Пустое множество по умолчанию, если что-то найдется, будет создано новое множество
    matched_grants = matched_actions = matched_macroses = matched_networks = set()

    # original_keyword может содержать русские буквы, keyword - нет
    original_keyword = form.cleaned_data['keyword']
    # если есть ключ для поиска сужаем потребителей со всего списка до списка сматченных по их грантам и сетям
    # TODO: Вынести в функцию обе ветки if
    if not is_keyword_forbidden(original_keyword):
        keyword = switch_keyboard_layout_to_eng(original_keyword)
        keyword = keyword.strip('_%')
        logger.debug('Searching for %r', keyword)

        # Ищем простые совпадения по имени/описанию потребителя
        consumers_by_keyword = consumers_qs.filter(
            Q(name__icontains=keyword) |
            Q(description__icontains=original_keyword)
        )
        logger.debug('Found %d consumers by keyword', consumers_by_keyword.count())

        # Ищем Гранты и Действия по имени
        matched_grants = set(
            Grant.objects.filter(namespace=namespace, name__icontains=keyword)
                         .values_list('id', flat=True)
        )
        matched_actions = set(
            Action.objects.filter(grant__namespace=namespace, name__icontains=keyword)
                          .values_list('id', flat=True)
        )

        # Ищем группы грантов
        matched_macroses = set()
        _matched_grants_actions_id = set(
            id
            for id, a in actions_by_id.iteritems()
            if a.grant_id in matched_grants
        )
        for mid, macros in macroses.iteritems():
            if keyword.lower() in macros.name.lower():
                matched_macroses.add(mid)

            # TODO: Группа грантов будет "подсвечена" в то время как подсветить надо Грант или Действие в составе Группы
            elif set(actions_by_macros[mid]) & (matched_actions | _matched_grants_actions_id):
                matched_macroses.add(mid)

        # Ищем Сети
        try:
            matched_networks = match_networks(keyword=keyword, deep=True)
        except BaseError as e:
            logger.debug('Networks not matched: {}'.format(e))
            matched_networks = []

        logger.debug(
            'Found %d grants, %d macroses, %d actions and %d networks',
            len(matched_grants),
            len(matched_macroses),
            len(matched_actions),
            len(matched_networks),
        )

        filter_query = get_filter_query(
            environments,
            grants=matched_grants,
            macroses=matched_macroses,
            actions=matched_actions,
            networks=matched_networks,
        )
        if filter_query:
            # Ищем потребителей по найденным Грантам, Действиям, Группам грантов и Сетям
            consumers_qs = consumers_qs.select_related(
                'activeaction_set',
                'activeaction_set__action',
                'activeaction_set__action__grant',
                'activemacros_set',
                'activenetwork_set',
            ).filter(filter_query).distinct()
            logger.debug('Found %d matched consumers', consumers_qs.count())

            # Потребители, найденные по имени или описанию идут вверху поисковой выдачи
            if consumers_qs.exists():
                consumers_qs = consumers_qs.exclude(id__in=consumers_by_keyword.values_list('id', flat=True))
                # Пагинатор умеет работать с queryset, но не умееет работать с итераторами
                # поэтому оборачиваю в list() и вычисляю ВЕСЬ queryset
                consumers_qs = list(chain(consumers_by_keyword, consumers_qs))

        else:
            consumers_qs = consumers_by_keyword

    # если мы запрашиваем не все окружения проекта, удалим из потребителей, у которых нет грантов в этих окружениях
    elif set(namespace.environments.all()) - set(environments):
        logger.debug('Getting consumers for environments %s', environments)
        consumers_qs = consumers_qs.select_related(
            'activeaction',
            'activemacros',
            'activenetwork',
        ).filter(
            Q(activeaction__environment__in=environments) |
            Q(activemacros__environment__in=environments) |
            Q(activenetwork__environment__in=environments)
        ).distinct()

        logger.debug('Found %d matched consumers', consumers_qs.count())

    paginator = Paginator(
        consumers_qs,
        settings.CONSUMER_PAGE_SIZE,
        allow_empty_first_page=True,
    )
    try:
        page = form.cleaned_data['page'] or 1
        page_consumers = paginator.page(page)
    except EmptyPage:
        page_consumers = paginator.page(paginator.page_range[-1])

    page_consumers_ids = list(page_consumers.object_list)

    # сериализуем список действий для каждого окружения каждого потребителя
    grants_by_id = get_dict_by_id(Grant.objects.filter(namespace=namespace))
    grants_by = defaultdict(dict)
    active_actions = ActiveAction.objects.filter(consumer__in=page_consumers_ids)\
        .select_related('action').order_by('action__name')
    for aa in active_actions:
        action = actions_by_id[aa.action_id]
        grant = grants_by_id[action.grant_id]
        action_dict = {
            'name': action.name,
            'description': action.description,
            'dangerous': action.dangerous,
            'expiration': aa.expiration and aa.expiration.strftime("%d.%m.%Y"),
            'highlighted': action.id in matched_actions
        }
        grant_dict = grants_by[aa.consumer_id, aa.environment_id].setdefault(grant.id, {
            'name': grant.name,
            'highlighted': grant.id in matched_grants,
            'actions': []
        })
        grant_dict['actions'].append(action_dict)

    networks_queryset = Network.objects.select_related('activenetwork__consumer')\
        .filter(activenetwork__consumer__namespace=namespace).distinct()
    networks = get_dict_by_id(networks_queryset)
    # сериализуем список сетей для каждого окружения каждого потребителя
    networks_by = defaultdict(list)
    for an in ActiveNetwork.objects.filter(consumer__in=page_consumers_ids):
        network = networks[an.network_id]
        networks_by[an.consumer_id, an.environment_id].append({
            'string': network.string,
            'highlighted': network.id in matched_networks,
        })

    # сериализуем словарь списков макросов для каждого окружения каждого потребителя
    macroses_by = defaultdict(list)
    for am in ActiveMacros.objects.filter(consumer__in=page_consumers_ids):
        macros = macroses[am.macros_id]
        # Ожидаем, что идентификаторы Действий в списке отсортированы по имени Действия
        macros_actions = [actions_by_id[k] for k in actions_by_macros[am.macros_id]]

        macros_grants = dict()
        for action in macros_actions:
            grant = grants_by_id[action.grant_id]
            grant_dict = macros_grants.setdefault(grant.id, {
                'name': grant.name,
                'actions': []
            })
            grant_dict['actions'].append({
                'name': action.name,
                'description': action.description
            })

        macros_dict = {
            'name': macros.name,
            'description': macros.description,
            'highlighted': macros.id in matched_macroses,
            'expiration': am.expiration and am.expiration.strftime("%d.%m.%Y"),
            'grants': sorted(macros_grants.values(), key=itemgetter('name'))
        }
        macroses_by[am.consumer_id, am.environment_id].append(macros_dict)

    # FIXME: Разобрать эту логику
    # сериализуем гранты и сети в окружения_потребителей в потребителей
    meta_consumers = []
    for consumer in page_consumers.object_list:
        meta_environments = []

        for env in environments:
            key = (consumer.id, env.id)
            if grants_by[key] or networks_by[key] or macroses_by[key]:
                data = dict(
                    id=env.id,
                    string=unicode(env),
                    networks=networks_by[key],
                    grants=sorted(grants_by[key].values(), key=itemgetter('name')),
                    macroses=sorted(macroses_by[key], key=itemgetter('name')),
                )
                if namespace.name in settings.NAMESPACES_BY_CLIENT:
                    try:
                        client = Client.objects.get(consumer=consumer, environment=env)
                        client = dict(
                            client_id=client.client_id,
                            name=client.name,
                        )
                    except Client.DoesNotExist:
                        client = None
                    data.update(client=client)
                meta_environments.append(data)

        # не показываем пустые окружения
        if meta_environments:
            meta_consumers += [model_to_dict(consumer, fields=('id', 'name', 'description'))]
            meta_consumers[-1]['environments'] = meta_environments

    logger.debug(u'Done %d meta-consumers for %r', len(meta_consumers), original_keyword)
    return {
        'success': True,
        'consumer_grants': meta_consumers,
        'pages': paginator.num_pages,
    }


def consumer_list(request):
    """Страница со списком потребителей"""
    context = namespace_selection(request)
    namespace = context['namespace']
    namespaces = context['namespaces']
    environments = namespace.environments.all()
    environment_names = sorted(settings.ENVIRONMENT_HUMAN_NAMES[namespace.name].items())
    context.update({
        'namespaces': json.dumps([{'id': n.id, 'name': n.name} for n in namespaces if not n.hidden]),
        'environments': json.dumps(list(environments.values('id', 'name', 'type'))),
        'env_groups': json.dumps(environment_names),
    })

    rendered = render(request, 'consumer_list.html', context)
    rendered.set_cookie('namespace', context['namespace'].id)
    return rendered


@json_response
def consumer_suggest(request):
    form = ConsumerSuggestForm(request.GET)
    if not form.is_valid():
        return {'success': False, 'errors': format_form_errors(form.errors)}

    keyword = form.cleaned_data['keyword']
    keyword_lowercase = keyword.lower()
    namespace_id = form.data['namespace']

    errors = []
    response = {'success': False, 'errors': errors}
    max_result = SUGGEST_CONSUMER_MAX_RESULTS

    already_in_suggestions = set()
    suggestions = []

    matched_by_consumer = Consumer.objects.filter(
        Q(name__icontains=keyword_lowercase) | Q(description__icontains=keyword_lowercase),
        namespace=namespace_id,
    )
    for consumer in matched_by_consumer[:max_result]:
        suggestions.append(dict(group=u'Потребители:', description=consumer.description, name=consumer.name))
        already_in_suggestions.add(consumer.id)

    networks_in_cache = NetworkCacheManager.all_networks()

    if len(suggestions) < max_result and networks_in_cache:
        try:
            matched_network_ids = match_networks(keyword=keyword_lowercase, network_list=networks_in_cache)
        except BaseError as e:
            logger.debug('Networks not matched: {}'.format(e))
            matched_network_ids = []

        networks = {n['id']: n for n in NetworkCacheManager.active_network_values()}
        queryset = ActiveNetwork.objects.select_related('consumer')\
            .filter(network__in=matched_network_ids, consumer__namespace=namespace_id)
        for active_network in queryset:
            if active_network.consumer_id not in already_in_suggestions:
                network = networks[active_network.network_id]
                name, description = active_network.consumer.name, active_network.consumer.description
                type_localized = NETWORK_TYPE_NAMES[network['type']]['singular']
                if keyword_lowercase in network['string'].lower():
                    group = u'Потребители с {} "{}":'.format(type_localized['instrumental'], network['string'])
                else:
                    _template = u'Потребители имеющие {} "{}" из "{}":'
                    group = _template.format(type_localized['accusative'], network['string'], keyword)
                suggestions.append(dict(group=group, description=description, name=name, network_type=network['type']))
                already_in_suggestions.add(active_network.consumer.id)

    type_order = (None, Network.FIREWALL, Network.CONDUCTOR, Network.HOSTNAME, Network.IPNETWORK, Network.IP)
    ordering = lambda s: (type_order.index(s.get('network_type')), s['name'])

    response.update(
        success=True,
        suggestions=sorted(suggestions, key=ordering)[:max_result],
        caching_networks=len(suggestions) < max_result and not networks_in_cache,
    )

    return response
