# -*- coding: utf-8 -*-
import json
from itertools import chain

from django.conf import settings
from django.contrib.comments import Comment
from django.contrib.contenttypes.models import ContentType
from django.core.exceptions import ValidationError
from django.forms import model_to_dict
from django.http import Http404
from django.db.models import Q
from django.views.generic import (
    ListView,
    DetailView,
    TemplateView,
)

from ..forms import (
    IssueListForm,
    CreateIssueForm,
    ReviewIssueForm,
    SubmitIssueForm,
    AddNetworksFormSet,
    DelNetworksFormSet,
)
from ..network_apis import NetworkResolver
from ..mail import (
    notify_issue_reviewed,
    notify_new_issue_created,
)
from ..models import (
    Issue,
    Namespace,
    Client,
    Consumer,
    ActiveAction,
    ActiveMacros,
    AddNetworks,
    ActiveNetwork,
    DelNetworks,
)
from ..permissions import (
    get_permissions,
    UserPermissions,
)
from ..utils import (
    json_response,
    get_client_ip,
    format_form_errors,
)


# TODO: Это можно сделать отдельной ручкой, а не рендерить в теле шаблона как сейчас
def all_consumers_by_namespaces():
    namespaces = Namespace.objects.order_by('name')
    namespace_consumers = []
    for n in namespaces:
        if n.hidden:
            continue
        namespace_consumers.append(
            dict(
                id=n.id,
                name=n.name,
                consumers={
                    c.name: dict(id=c.id, description=c.description)
                    for c in Consumer.objects.filter(namespace=n)
                },
                by_client=n.name in settings.NAMESPACES_BY_CLIENT,
                client_required=n.name in settings.NAMESPACES_WITH_REQUIRED_CLIENT,
                environments=[
                    dict(id=e.id, description=unicode(e))
                    for e in n.environments.all()
                ],
            )
        )
    return json.dumps(namespace_consumers)


def check_can_edit_namespace(namespace_name, username):
    return (
        namespace_name not in settings.NAMESPACES_FOR_ADMINS or
        username in settings.PASSPORT_TEAM_ADMIN
    )


def get_editable_namespaces(username):
    if username in settings.PASSPORT_TEAM_ADMIN:
        return [n.id for n in Namespace.objects.all()]
    return [n.id for n in Namespace.objects.exclude(name__in=settings.NAMESPACES_FOR_ADMINS)]


class IssueListView(ListView):
    """
    Страница списка заявок с возможностями
      - фильтровать заявки по проекту (blackbox, passport, etc)
      - фильтровать заявки по их статусу (обработана или нет)
      - искать заявки по имени или описанию потребителя
      - искать заявки по имени автора или рецензента
      - искать заявки по тексту комментариев к заявке
    """

    context_object_name = 'issues'
    model = Issue
    paginate_by = 50
    form_class = IssueListForm

    def process_form(self):
        self.form = self.form_class(self.request.GET)
        if not self.form.is_valid():
            raise Http404()

    def _filter_by_status(self, queryset, status):
        user = self.request.user
        if status == IssueListForm.PENDING:
            user_permissions = UserPermissions(user)
            return queryset.filter(user_permissions.Q(), status=Issue.NEW)\
                .exclude(status=Issue.DRAFT).distinct()
        elif status == IssueListForm.MY:
            return queryset.filter(creator=user).exclude(status=Issue.DRAFT)
        elif status == IssueListForm.DRAFTS:
            return queryset.filter(status=Issue.DRAFT, creator=user)
        else:
            return queryset.exclude(status=Issue.DRAFT)

    def _filter_by_keyword(self, queryset, kw):
        # Ищем по вхождению строки в имя или описание потребителя
        kw_query = Q(
            Q(consumer__name__icontains=kw) |
            Q(consumer_name__icontains=kw) |
            Q(consumer_name_new__icontains=kw) |
            Q(consumer_description__icontains=kw)
        )
        # Ищем по точному соответствию username автора заявки или принявшего ее
        kw_query |= Q(
            Q(creator__username__iexact=kw) |
            Q(approving_person__username__iexact=kw)
        )

        # Ищем по тексту комментариев к заявкам
        comment_qs = Comment.objects.for_model(Issue).filter(
            is_removed=False,
            is_public=True,
            comment__icontains=kw,
        )
        if comment_qs.count():
            issue_ids = comment_qs.values_list('object_pk', flat=True)
            kw_query |= Q(id__in=issue_ids)

        return queryset.filter(kw_query)

    def filter(self, queryset):
        """Покажем пользователю только подходящие заявки"""
        # Сначала отфильтруем по проекту
        if self.form.cleaned_data['namespace']:
            queryset = queryset.filter(namespace=self.form.cleaned_data['namespace'])

        # Отфильтруем по статусу заявки
        if self.form.cleaned_data['display']:
            queryset = self._filter_by_status(queryset, self.form.cleaned_data['display'])

        # Поиск по ключевому слову
        if self.form.cleaned_data['keyword']:
            queryset = self._filter_by_keyword(queryset, self.form.cleaned_data['keyword'])

        return queryset

    def get_queryset(self):
        queryset = super(IssueListView, self).get_queryset()
        # Предварительно выберем все связанные модели
        queryset = queryset.select_related(
            'approving_person',
            'creator',
            'consumer',
            'namespace',
        ).prefetch_related(
            'add_action',
            'del_action',
            'add_macros',
            'del_macros',
            'add_network',
            'del_network',
            'environments',
        )
        return self.filter(queryset)

    def get_context_data(self, **kwargs):
        context = super(IssueListView, self).get_context_data(**kwargs)
        context.update(
            display=self.form.cleaned_data.get('display'),
            # В контексте уже есть имя namespace
            filter_namespace=self.form.cleaned_data.get('namespace'),
            keyword=self.form.cleaned_data.get('keyword'),
        )
        return context

    def get(self, request, *args, **kwargs):
        self.process_form()  # Наполняем self.form и валидируем параметры
        return super(IssueListView, self).get(request, *args, **kwargs)


class IssueDetailView(DetailView):
    context_object_name = 'issue'
    model = Issue

    def get_context_data(self, **kwargs):
        context = super(IssueDetailView, self).get_context_data()

        is_new_consumer = (self.object.type == Issue.TO_CREATE)
        user_permissions = UserPermissions(self.request.user)
        has_permission = user_permissions.have_all(
            namespace=self.object.namespace,
            environments=self.object.environments.all(),
        )
        can_edit = check_can_edit_namespace(self.object.namespace.name, self.request.user.username)
        context.update(
            has_permission=has_permission and can_edit,
            new_consumer=is_new_consumer,
            with_clients=self.object.namespace.name in settings.NAMESPACES_BY_CLIENT,
        )

        return context


class EditIssueView(TemplateView):
    template_name = 'issue_new.html'
    form_class = CreateIssueForm

    def process_form(self):
        self.form = self.form_class(self.request.GET)
        if not self.form.is_valid():
            raise Http404(self.form.errors)

    def get(self, request, *args, **kwargs):
        self.process_form()
        self.cookie_namespace = request.COOKIES.get('namespace')
        return super(EditIssueView, self).get(request, *args, **kwargs)

    def get_context_by_consumer(self, consumer, clone=False):
        environments_id = [self.form.cleaned_data['environment'].id]

        issue_dict = dict(
            id=None,
            type=Issue.TO_CLONE if clone else Issue.TO_MODIFY,
            namespace=consumer.namespace.id,
            environments_id=environments_id,
            consumer_name=consumer.name,
            consumer_name_new='',
            consumer_description=consumer.description,
            consumer_description_new=consumer.description,
        )
        if consumer.namespace.name in settings.NAMESPACES_BY_CLIENT:
            clients = Client.objects.filter(consumer=consumer)
            issue_dict.update(
                clients={
                    c.environment.id: {
                        'id': c.id,
                        'client_id': c.client_id,
                        'name': c.name,
                    } for c in clients
                },
            )
        return issue_dict

    @get_permissions
    def get_context_data(self, clone=False, **kwargs):
        context = super(EditIssueView, self).get_context_data(**kwargs)

        issue = self.form.cleaned_data.get('issue')
        consumer = self.form.cleaned_data.get('consumer')

        if issue:
            if check_can_edit_namespace(issue.namespace.name, self.request.user.username):
                issue.status = Issue.DRAFT
                issue.save()

            issue_dict = model_to_dict(issue, ['id', 'type', 'namespace', 'consumer_name_new'])
            issue_dict.update(
                consumer_name=issue.get_consumer_name(),
                consumer_description=issue.get_consumer_description(),
                consumer_description_new=issue.consumer_description,
                environments_id=[e.id for e in issue.environments.all()],
                expiration_date=issue.expiration and issue.expiration.strftime("%d.%m.%Y"),
                emails=[{'address': e.address} for e in issue.emails.all()],
            )
            if issue.namespace.name in settings.NAMESPACES_BY_CLIENT:
                issue_dict.update(
                    clients={
                        c.environment.id: {
                            'id': c.id,
                            'client_id': c.client_id,
                            'name': c.name,
                        } for c in issue.clients.all()
                    },
                )

        elif consumer:
            issue_dict = self.get_context_by_consumer(consumer, clone=clone)

        else:
            issue_dict = dict.fromkeys([
                'consumer_description',
                'consumer_description_new',
                'consumer_name',
                'consumer_name_new',
            ], '')
            namespaces = Namespace.objects.all()
            namespace = [n for n in namespaces if not n.hidden][0]
            issue_dict.update(
                id=None,
                type=Issue.TO_CREATE,
                namespace=self.cookie_namespace or namespace.id,
                environments_id=[namespace.environments.all()[0].id],
            )

        context.update(
            namespace_consumers=all_consumers_by_namespaces(),
            issue=json.dumps(issue_dict),
            editable_namespaces=json.dumps(get_editable_namespaces(self.request.user.username)),
        )
        return context


class CloneIssueView(EditIssueView):

    @get_permissions
    def get_context_data(self, **kwargs):
        return super(CloneIssueView, self).get_context_data(clone=True, **kwargs)


@json_response
def issue_review(request):
    """AJAX запрос на согласование или отклонение заявки"""
    response = {'success': False, 'errors': []}
    form = ReviewIssueForm(request.POST)
    if not form.is_valid():
        response['errors'] = format_form_errors(form.errors)
        return response

    issue = form.cleaned_data['issue']

    user_permissions = UserPermissions(request.user)
    if not request.user:
        response['errors'] = [u'Пользователь не найден']
        return response

    namespace = issue.namespace

    if not check_can_edit_namespace(namespace.name, request.user.username):
        response['errors'] = [u'У Вас нет прав для действий с заявками в этом проекте']
        return response

    environments = issue.environments.all()
    expiration = issue.expiration
    permissions_needed = user_permissions.do_not_have(namespace=namespace, environments=environments)
    if permissions_needed:
        response['errors'] = [
            u'У Вас нет прав для действий с заявкой в окружении(ях): %s' %
            ', '.join(map(str, permissions_needed))
        ]
        return response

    if request.POST.get('result') == 'rejected':
        issue.reject(request.user)
        notify_issue_reviewed(request, issue, is_rejected=True)
        response.update({'success': True, 'result': 'rejected'})
        return response

    if issue.type in (Issue.TO_MODIFY, Issue.TO_SET_EXPIRATION):
        consumer = issue.consumer
        if issue.consumer_name_new:
            consumer.name = issue.consumer_name_new
        if issue.consumer_description:
            consumer.description = issue.consumer_description
        consumer.save()

    elif issue.type in (Issue.TO_CREATE, Issue.TO_CLONE):
        consumer = Consumer.objects.create(
            name=issue.consumer_name,
            namespace=namespace,
            description=issue.consumer_description,
        )

    for environment in environments:
        for action in issue.add_action.all():
            aa, _ = ActiveAction.objects.get_or_create(
                consumer=consumer,
                action=action,
                environment=environment,
            )
            aa.expiration = expiration
            aa.save()
        for macros in issue.add_macros.all():
            am, _ = ActiveMacros.objects.get_or_create(
                consumer=consumer,
                macros=macros,
                environment=environment,
            )
            am.expiration = expiration
            am.save()

    ActiveAction.objects.filter(
        consumer=consumer,
        action__in=issue.del_action.all(),
        environment__in=environments,
    ).delete()
    ActiveMacros.objects.filter(
        consumer=consumer,
        macros__in=issue.del_macros.all(),
        environment__in=environments,
    ).delete()

    for ian in AddNetworks.objects.filter(issue=issue):
        ActiveNetwork.objects.get_or_create(consumer=consumer, network=ian.network, environment=ian.environment)
    for idn in DelNetworks.objects.filter(issue=issue):
        ActiveNetwork.objects.filter(consumer=consumer, network=idn.network, environment=idn.environment).delete()

    # Отвязываем предыдущих clients от этого потребителя
    if issue.type in (Issue.TO_MODIFY, Issue.TO_SET_EXPIRATION):
        old_clients = Client.objects.filter(consumer=consumer, environment__in=environments)
        old_clients.update(consumer=None)

    # Привязываем новых clients к потребителю
    issue.clients.all().update(consumer=consumer)

    issue.approve(request.user)
    notify_issue_reviewed(request, issue, is_approved=True)
    response.update({'success': True, 'result': 'confirmed'})
    return response


@json_response
def issue_submit(request):
    """AJAX запрос отправки заявки"""
    post = request.POST
    id = post.get('id') or 0
    try:
        instance = Issue.objects.get(id=id)
    except Issue.DoesNotExist:
        instance = None

    emails = post.getlist('emails', [])
    if emails:
        mutable = post._mutable
        post._mutable = True
        post['emails'] = list(emails)
        post._mutable = mutable

    form = SubmitIssueForm(post, instance=instance)
    try:
        del_formset = DelNetworksFormSet(post, prefix='del_networks')
        add_formset = AddNetworksFormSet(post, prefix='add_networks')
    except ValidationError as e:
        return {'success': False, 'errors': e.messages}

    for f in chain([form], del_formset, add_formset):
        if not f.is_valid():
            return {'success': False, 'errors': format_form_errors(f.errors)}

    del_clients = False
    old_clients = Client.objects.filter(
        consumer=form.cleaned_data.get('consumer'),
        environment__in=form.cleaned_data.get('environments'),
        namespace=form.cleaned_data.get('namespace'),
    )
    if old_clients and not form.cleaned_data.get('clients'):
        del_clients = True
    if instance and instance.clients and not form.cleaned_data.get('clients'):
        del_clients = True

    is_subforms_empty = not (any(del_formset) or any(add_formset))
    if form.is_empty() and is_subforms_empty and not del_clients:
        return {'success': False, 'errors': [u'В форме заявки не отражено ни одно изменение данных']}

    if instance and instance.creator != request.user and not request.user.is_superuser:
        return {'success': False, 'errors': [u'У Вас нет необходимых прав на изменение этой заявки']}

    namespace = form.cleaned_data['namespace']
    environments = form.cleaned_data['environments']
    if not UserPermissions(request.user).have_all(namespace=namespace, environments=environments):
        errors = []
        for add_network_form in add_formset:
            network = add_network_form.cleaned_data['network']
            is_allowed, message = NetworkResolver.check_permissions(
                network.type,
                network.string,
                request.user.username,
            )
            if not is_allowed:
                if isinstance(message, list):
                    errors.extend(message)
                else:
                    errors.append(message)
        if errors:
            return {'success': False, 'errors': errors}

    if not check_can_edit_namespace(namespace.name, request.user.username):
        return {'success': False, 'errors': [u'У Вас нет необходимых прав на изменение этой заявки']}

    is_draft = form.cleaned_data.get('draft')

    form.instance.status = Issue.DRAFT if is_draft else Issue.NEW
    form.instance.creator = request.user
    form.instance.approving_person = request.user

    issue = form.save()
    issue.emails = form.cleaned_data['emails']
    issue.save()

    for form_ in list(del_formset) + list(add_formset):
        form_.instance.issue = issue
    issue.add_network.clear()
    issue.del_network.clear()
    del_formset.save()
    add_formset.save()

    content_type_id = ContentType.objects.get(model='issue', app_label='core').id
    comment = Comment.objects.create(
        content_type_id=content_type_id,
        object_pk=issue.pk,
        user_id=request.user.id,
        site_id=settings.SITE_ID,
        user_name=request.user.username,
        user_email=request.user.email,
        comment=form.cleaned_data['comment'],
        ip_address=get_client_ip(request),
    )

    if not is_draft:
        notify_new_issue_created(request, issue, comment)

    return {'success': True, 'issue_id': issue.id}
