# -*- coding: utf-8 -*-

from __future__ import (
    absolute_import,
    unicode_literals,
)

import logging
import os
import signal
import sys

from kazoo.client import (
    KazooClient,
    KazooState,
)
from kazoo.handlers.threading import KazooTimeoutError
from kazoo.recipe.lock import Lock

log = logging.getLogger('passport_db_scripts.zookeeper')


def run_exclusively(func, lock_name, zookeeper_hosts):
    signal.signal(signal.SIGUSR1, _handle_signal)

    zk = KazooClient(hosts=','.join(zookeeper_hosts))
    zk.add_listener(_listen_state)

    zk.start()
    try:
        lock = Lock(zk, lock_name)
        is_acquired = lock.acquire(blocking=False)
        if is_acquired:
            try:
                return func()
            finally:
                if zk.connected:
                    lock.release()
    finally:
        # Т.к. zk.stop приводит к переходу CONNECTED -> LOST, а обработчик
        # состояний сразу же выходит, что может замаскировать исключение.
        zk.remove_listener(_listen_state)
        zk.stop()


def _handle_signal(signum, frame):
    log.debug('SIGUSR1 caught')
    sys.exit(1)


def _listen_state(state):
    if state == KazooState.SUSPENDED:
        log.error('Zookeeper connection is suspended')
        os.kill(os.getpid(), signal.SIGUSR1)
    elif state == KazooState.LOST:
        # Как показала практика, переход в такое состояние возможен, когда
        # 1) Выбрасывается необработанное исключение в сопроцессе Kazoo.
        # 2) Вызывается zk.stop.
        log.error('Zookeeper connection is lost')
        os.kill(os.getpid(), signal.SIGUSR1)


# Чтобы клиенты не привязывались к Kazoo
TimeoutError = KazooTimeoutError
